package com.elitescloud.cloudt.system.controller.common.system;

import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.constant.BusinessObjectConstant;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonAppInstanceRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.ServiceInstancesVo;
import com.elitescloud.cloudt.system.service.PlatformQueryService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.discovery.DiscoveryClient;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.http.*;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

/**
 * 平台通用接口.
 *
 * @author Kaiser（wang shao）
 * @date 2023/11/23
 */
@Api(tags = "平台通用接口")
@RestController
@RequestMapping(value = "/common/platform", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@Slf4j
@BusinessObject(businessType = BusinessObjectConstant.SYS_COMMON)
public class PlatformCommonController {

    private PlatformQueryService platformQueryService;
    private final DiscoveryClient discoveryClient;

    private final RestTemplate restTemplate;

    public PlatformCommonController(DiscoveryClient discoveryClient) {
        this.discoveryClient = discoveryClient;
        this.restTemplate = RestTemplateFactory.instance();
    }

    @ApiOperation(value = "版本列表")
    @ApiOperationSupport(order = 1)
    @GetMapping(value = "/versions")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "版本列表")
    public ApiResult<List<CodeNameParam>> getVersionList() {
        return platformQueryService.listProjectVersion();
    }

    @Autowired
    public void setPlatformQueryService(PlatformQueryService platformQueryService) {
        this.platformQueryService = platformQueryService;
    }

    @ApiOperation(value = "注册服务清单", hidden = true)
    @ApiOperationSupport(order = 1)
    @GetMapping(value = "/serviceInstances")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "注册服务清单")
    public ApiResult<List<ServiceInstancesVo>> printServiceInstances() {
        List<String> services = discoveryClient.getServices();
        List<ServiceInstancesVo> servicesInstances = new ArrayList<>();
        for (String service : services) {
            discoveryClient.getInstances(service).forEach(instance -> {
                ServiceInstancesVo serviceInstancesVo = new ServiceInstancesVo();
                serviceInstancesVo.setServices(service);
                serviceInstancesVo.setServiceId(instance.getServiceId());
                serviceInstancesVo.setHost(instance.getHost());
                serviceInstancesVo.setPort(instance.getPort());
                serviceInstancesVo.setSecure(instance.isSecure());
                serviceInstancesVo.setUri(instance.getUri());
                servicesInstances.add(serviceInstancesVo);
            });
        }
        return ApiResult.ok(servicesInstances);
    }

    @ApiOperation(value = "服务实例列表")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "serviceId", value = "服务ID"),
            @ApiImplicitParam(name = "appOnly", value = "是否仅返回应用列表中的", defaultValue = "false"),
    })
    @GetMapping(value = "/serviceInstance/list")
    public ApiResult<List<CommonAppInstanceRespVO>> serviceInstanceList(@RequestParam(name = "serviceId", required = false) String serviceId,
                                                                        @RequestParam(name = "appOnly", required = false, defaultValue = "false") Boolean appOnly) {
        return platformQueryService.serviceInstanceList(serviceId, appOnly);
    }


    private static String OPENAPI_GET_JSON_URL = "/sys/openapi/get/json";
    private static String OPENAPI_DOWNLOAD_JSON_URL = "/sys/openapi/download/json";

    private static String UDC_LOCAL_CACHE_URL = "/cloudt/core/udc/clear/localCache";

    @ApiOperation(value = "本地调用服务的openApi获取json接口")
    @ApiOperationSupport(order = 1)
    @GetMapping(value = "/openapiGetJson")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "openApi查询")
    public ApiResult<Object> callOpenApiJsonDServiceInstances(String serviceName) {
        List<ServiceInstance> instances = discoveryClient.getInstances(serviceName);
        if (instances.isEmpty()) {
            return ApiResult.fail("实例注冊信息不存在，请检查环境：" + serviceName);
        }
        StringBuffer errMessage = new StringBuffer();
        // 从当前请求中复制headers
        HttpHeaders headers = createHeadersFromCurrentRequest();
        // 创建一个新的HttpEntity，包含headers
        HttpEntity<String> entity = new HttpEntity<>("body", headers);
        // 选择一个实例（这里简单地选择第一个）
        for (ServiceInstance serviceInstance : instances) {
            String url = serviceInstance.getUri().toString() + OPENAPI_GET_JSON_URL;

            try {
                log.info("内部远程调用：url={} serviceInstance={}", url, serviceInstance);
                // 使用RestTemplate调用服务
                ResponseEntity<String> response = restTemplate.exchange(url, HttpMethod.GET, entity, String.class);
                log.info("内部远程调用：url {}  返回结果：{}", url, response);
                return ApiResult.ok(response.getBody());
            } catch (Exception e) {
                // 异常处理
                log.error("内部远程调用异常: " + e.getMessage()); // 记录异常，尝试下一个实例
                errMessage.append("内部远程调用异常: " + url + e.getMessage());
            }

        }
        return ApiResult.fail("远程调用实例全部失败:" + errMessage);
    }


    @ApiOperation(value = "本地调用服务的openApi下载json接口")
    @ApiOperationSupport(order = 2)
    @GetMapping(value = "/openapiDownJson")
    @BusinessObjectOperation(operationType = OperationTypeEnum.DOWNLOAD, operationDescription = "openApi下载")
    public ResponseEntity<?> callOpenApiDownJsonDServiceInstances(String serviceName) {
        List<ServiceInstance> instances = discoveryClient.getInstances(serviceName);
        if (instances.isEmpty()) {
            // 包装错误信息在 ResponseEntity 中
            return ResponseEntity
                    .status(HttpStatus.NOT_FOUND)
                    .body("实例注册信息不存在，请检查环境：" + serviceName);
        }
        StringBuffer errMessage = new StringBuffer();
        // 从当前请求中复制headers
        HttpHeaders headers = createHeadersFromCurrentRequest();


        // 创建一个新的HttpEntity，包含headers
        HttpEntity<String> entity = new HttpEntity<>("body", headers);
        for (ServiceInstance serviceInstance : instances) {
            String url = serviceInstance.getUri().toString() + OPENAPI_DOWNLOAD_JSON_URL;
            try {
                log.info("内部远程调用：url={} serviceInstance={}", url, serviceInstance);

                ResponseEntity<byte[]> response = restTemplate.exchange(url, HttpMethod.GET, entity, byte[].class);
                ByteArrayResource resource = new ByteArrayResource(response.getBody());
                // 使用 LocalDateTime 替代 LocalDate

                log.info("下载文件成功：{}", url);
                return ResponseEntity.ok()
                        .headers(response.getHeaders())
                        .contentLength(response.getBody().length)
                        .body(resource);
            } catch (Exception e) {
                log.error("内部远程调用异常: " + e.getMessage());
                errMessage.append("内部远程调用异常: " + url + e.getMessage());
            }
        }
        return ResponseEntity
                .status(HttpStatus.NOT_FOUND)
                .body("远程调用实例全部失败:" + errMessage);
    }

    @ApiOperation(value = "远程服务调用清理远端服务UDC二级缓存")
    @ApiOperationSupport(order = 2)
    @GetMapping(value = "/udcLocalCache")
    @BusinessObjectOperation(operationType = OperationTypeEnum.DOWNLOAD, operationDescription = "远程服务调用清理远端服务UDC二级缓存")
    public ApiResult udcLocalCacheServiceInstances(String serviceName) {
        List<ServiceInstance> instances = discoveryClient.getInstances(serviceName);
        if (instances.isEmpty()) {
            return ApiResult.fail("实例注册信息不存在，请检查环境：" + serviceName);
        }
        StringBuffer errMessage = new StringBuffer();
        // 从当前请求中复制headers
        HttpHeaders headers = createHeadersFromCurrentRequest();
        // 创建一个新的HttpEntity，包含headers
        HttpEntity<String> entity = new HttpEntity<>("body", headers);
        for (ServiceInstance serviceInstance : instances) {
            String url = serviceInstance.getUri().toString() + UDC_LOCAL_CACHE_URL;

            try {
                log.info("内部远程调用：url={} serviceInstance={}", url, serviceInstance);
                ResponseEntity<ApiResult<Boolean>> response = restTemplate.exchange(
                        url,
                        HttpMethod.DELETE,
                        entity,
                        new ParameterizedTypeReference<ApiResult<Boolean>>() {
                        } // 使用ParameterizedTypeReference来指定具体的泛型类型
                );
                // 检查响应状态和数据
                if (response.getStatusCode().is2xxSuccessful()) {
                    log.info("内部远程调用成功：url={} serviceInstance={}", url, serviceInstance);
                } else {
                    errMessage.append("内部远程调用状态异常：" + url + " 异常码：" + response.getStatusCode() + ";");
                    log.error("Request failed with status: " + response.getStatusCode());
                }
            } catch (Exception e) {
                log.error("内部远程调用异常: " + e.getMessage());
                errMessage.append("内部远程调用异常: ").append(url).append(e.getMessage()).append(";");
            }
        }
        if (errMessage.length() > 0) {
            return ApiResult.fail(errMessage.toString());
        } else {
            return ApiResult.ok();
        }
    }

    private HttpHeaders createHeadersFromCurrentRequest() {
        HttpHeaders headers = new HttpHeaders();
        HttpServletRequest request = getCurrentHttpRequest();
        Enumeration<String> headerNames = request.getHeaderNames();
        while (headerNames.hasMoreElements()) {
            String headerName = headerNames.nextElement();
            headers.set(headerName, request.getHeader(headerName));
        }
        return headers;
    }

    private HttpServletRequest getCurrentHttpRequest() {
        ServletRequestAttributes attributes =
                (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        return attributes.getRequest();
    }
}
