package com.elitescloud.cloudt.system.controller.mng.menu;

import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.model.vo.resp.menu.*;
import com.elitescloud.cloudt.system.model.vo.save.menu.CustomMenuBoundSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.menu.CustomMenuGroupSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.menu.MenuTreeSaveVO;
import com.elitescloud.cloudt.system.service.MenuMngService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.HttpEntity;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;
import springfox.documentation.annotations.ApiIgnore;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * 菜单管理.
 *
 * @author Kaiser（wang shao）
 * 2022/9/19
 */
@Api(tags = "自定义菜单管理")
@RestController
@RequestMapping(value = "/mng/menu", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class CustomMenuMngController {

    private final MenuMngService menuMngService;

    public CustomMenuMngController(MenuMngService menuMngService) {
        this.menuMngService = menuMngService;
    }

    @ApiOperation(value = "获取是否启用自定义菜单树")
    @ApiOperationSupport(order = 1)
    @GetMapping(value = "/tree/custom/enabled")
    public ApiResult<Boolean> getEnabledCustom() {
        return menuMngService.getCustomEnabled();
    }

    @ApiOperation(value = "切换是否启用自定义菜单树")
    @ApiOperationSupport(order = 2)
    @PatchMapping(value = "/customEnabled")
    public ApiResult<Boolean> switchEnabled() {
        return menuMngService.updateEnabledCustom();
    }

    @ApiOperation(value = "清空自定义菜单")
    @ApiOperationSupport(order = 3)
    @DeleteMapping(value = "/custom/tree")
    public ApiResult<Boolean> deleteAll() {
        return menuMngService.removeAllCustom();
    }

    @ApiOperation(value = "获取自定义菜单树")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParam(name = "tree", value = "是否返回树形结构数据", dataType = "boolean", defaultValue = "true")
    @GetMapping(value = "/custom/tree")
    public ApiResult<List<CustomMenuTreeRespVO>> getCustomMenuTree(@RequestParam(name = "tree", defaultValue = "true") Boolean tree) {
        return menuMngService.getCustomMenuTree(tree);
    }

    @ApiOperation(value = "获取自定义树的节点信息")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "菜单ID", dataType = "long", required = true)
    @GetMapping(value = "/custom/tree/node")
    public ApiResult<CustomMenuNodeDetailRespVO> getNodeDetail(Long id) {
        return menuMngService.getCustomNodeDetail(id);
    }

    @ApiOperation(value = "保存自定义菜单分组")
    @ApiOperationSupport(order = 6)
    @PostMapping(value = "/custom/group")
    public ApiResult<Long> saveCustomMenuTree(@Valid @RequestBody CustomMenuGroupSaveVO saveVO) {
        return menuMngService.saveCustomMenuGroup(saveVO);
    }

    @ApiOperation(value = "获取分组列表")
    @ApiOperationSupport(order = 6)
    @GetMapping(value = "/custom/groupList")
    public ApiResult<List<IdCodeNameParam>> listGroup() {
        return menuMngService.listGroup();
    }

    @ApiOperation(value = "获取自定义菜单分组的编辑信息")
    @ApiOperationSupport(order = 7)
    @GetMapping(value = "/custom/group")
    public ApiResult<CustomMenuEditRespVO> getEdit(Long id) {
        return menuMngService.getCustomEdit(id);
    }

    @ApiOperation(value = "修改菜单的启用状态")
    @ApiOperationSupport(order = 8)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "菜单ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "enabled", value = "是否启用", dataType = "boolean", required = true),
    })
    @PatchMapping(value = "/custom/enabled")
    public ApiResult<Long> updateEnabled(@RequestParam(name = "id") @NotNull(message = "菜单ID为空") Long id,
                                         @RequestParam(name = "enabled") @NotNull(message = "启用状态为空") Boolean enabled) {
        return menuMngService.updateCustomEnabled(id, enabled);
    }

    @ApiOperation(value = "修改菜单的自定义名称")
    @ApiOperationSupport(order = 8)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "菜单ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "customName", value = "自定义名称"),
    })
    @PatchMapping(value = "/custom/name")
    public ApiResult<Long> updateName(@RequestParam(name = "id") @NotNull(message = "菜单ID为空") Long id,
                                      @RequestParam(name = "customName", required = false) String customName) {
        return menuMngService.updateCustomName(id, customName);
    }

    @ApiOperation(value = "删除自定义菜单")
    @ApiOperationSupport(order = 9)
    @ApiImplicitParam(name = "cascade", value = "是否级联删除", defaultValue = "true", dataType = "boolean")
    @DeleteMapping(value = "/custom")
    public ApiResult<List<Long>> remove(@RequestBody @NotEmpty(message = "ID为空") List<Long> ids,
                                        @RequestParam(name = "cascade", required = false, defaultValue = "true") Boolean cascade) {
        return menuMngService.removeCustom(ids, cascade);
    }

    @ApiOperation(value = "删除自定义菜单的下级菜单")
    @ApiOperationSupport(order = 9)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "菜单ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "withSelf", value = "是否删除自己", defaultValue = "false", dataType = "boolean"),
    })
    @DeleteMapping(value = "/custom/children")
    public ApiResult<Long> remove(@RequestParam(name = "id") @NotNull(message = "菜单ID为空") Long id,
                                  @RequestParam(name = "withSelf", required = false, defaultValue = "false") Boolean withSelf) {
        return menuMngService.removeCustomChildren(id, withSelf);
    }

    @ApiOperation(value = "移动菜单")
    @ApiOperationSupport(order = 10)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "菜单ID", dataType = "long", required = true),
            @ApiImplicitParam(name = "newParentId", value = "新的上级菜单ID", dataType = "long"),
            @ApiImplicitParam(name = "sortNo", value = "排序号", dataType = "int"),
    })
    @DeleteMapping(value = "/custom/move")
    public ApiResult<Long> moveCustom(@RequestParam(name = "id") @NotNull(message = "菜单ID为空") Long id,
                                      @RequestParam(name = "newParentId", required = false) Long newParentId,
                                      @RequestParam(name = "sortNo", required = false) Integer sortNo) {
        return menuMngService.moveCustom(id, newParentId, sortNo);
    }

    @ApiOperation(value = "获取分组下绑定的菜单")
    @ApiOperationSupport(order = 11)
    @ApiImplicitParam(name = "code", value = "分组编码", required = true)
    @GetMapping(value = "/custom/menu/bound")
    public ApiResult<List<CustomMenuBoundRespVO>> getBoundMenus(@RequestParam(name = "code") @NotBlank(message = "分组编码为空") String code) {
        return menuMngService.getBoundMenus(code);
    }

    @ApiOperation(value = "保存分组下绑定的菜单")
    @ApiOperationSupport(order = 12)
    @PostMapping(value = "/custom/menu/bound")
    public ApiResult<String> saveBoundMenus(@RequestBody @Valid CustomMenuBoundSaveVO saveVO) {
        return menuMngService.saveBoundMenus(saveVO);
    }

    @GetMapping(value = "/custom/export")
    @ApiOperation(value = "导出数据", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    @ApiOperationSupport(order = 13)
    public HttpEntity<StreamingResponseBody> exportData() {
        return menuMngService.export();
    }

    @PostMapping("/custom/import")
    @ApiOperation(value = "导入数据")
    @ApiOperationSupport(order = 14)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "file", value = "数据文件", required = true),
            @ApiImplicitParam(name = "increment", value = "是否是增量", dataType = "boolean")
    })
    public ApiResult<Boolean> importData(@NotNull(message = "数据文件为空") @RequestParam(name = "file") MultipartFile file,
                                         @RequestParam(name = "increment", required = false, defaultValue = "false") Boolean increment) {
        return menuMngService.importByFile(file, increment);
    }

    @PostMapping("/custom/importByMenu")
    @ApiOperation(value = "根据菜单导入数据")
    @ApiOperationSupport(order = 15)
    public ApiResult<Boolean> importData() {
        return menuMngService.importByPlatformMenu();
    }

    /**
     * 获取默认菜单树
     *
     * @return 菜单树
     */
    @Deprecated(since = "3.3.0")
    @ApiIgnore
    @ApiOperation(value = "获取默认菜单树")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParam(name = "tree", value = "是否返回树形结构数据", dataType = "boolean", defaultValue = "true")
    @GetMapping(value = "/tree/default")
    public ApiResult<List<MenuTreeRespVO>> getTreeDefault(@RequestParam(name = "tree", defaultValue = "true") Boolean tree) {
        return menuMngService.getTreeDefault(tree);
    }

    /**
     * 获取自定义菜单树
     *
     * @return 菜单树
     * @deprecated 已弃用，改用 {@link #getCustomMenuTree(Boolean)}
     */
    @Deprecated(since = "3.3.0")
    @ApiIgnore
    @ApiOperation(value = "获取自定义菜单树")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "tree", value = "是否返回树形结构数据", dataType = "boolean", defaultValue = "true")
    @GetMapping(value = "/tree/custom")
    public ApiResult<List<MenuTreeCustomRespVO>> getTreeCustom(@RequestParam(name = "tree", defaultValue = "true") Boolean tree) {
        return menuMngService.getTreeCustom(tree);
    }

    /**
     * 保存菜单树
     *
     * @param saveVOList 菜单树
     * @return 保存结果
     * @deprecated 已弃用，改用 {@link #saveCustomMenuTree(CustomMenuGroupSaveVO)}
     */
    @Deprecated(since = "3.3.0")
    @ApiIgnore
    @ApiOperation(value = "保存菜单树")
    @ApiOperationSupport(order = 4)
    @PostMapping(value = "/tree/save")
    public ApiResult<Boolean> saveTree(@RequestBody @Valid List<MenuTreeSaveVO> saveVOList) {
        return menuMngService.saveTree(saveVOList);
    }
}
