package com.elitescloud.cloudt.system.modules.wecom.util;

import cn.hutool.core.lang.Assert;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.boot.util.RestTemplateHelper;
import com.elitescloud.cloudt.system.modules.wecom.model.AccessToken;
import com.elitescloud.cloudt.system.modules.wecom.model.CommonResult;
import com.elitescloud.cloudt.system.modules.wecom.model.department.DepartmentDeleteResult;
import com.elitescloud.cloudt.system.modules.wecom.model.department.DepartmentIdListResult;
import com.elitescloud.cloudt.system.modules.wecom.model.department.DepartmentSaveParam;
import com.elitescloud.cloudt.system.modules.wecom.model.department.DepartmentSaveResult;
import com.elitescloud.cloudt.system.modules.wecom.model.login.Code2UserInfoResult;
import com.elitescloud.cloudt.system.modules.wecom.model.login.Ticket2UserDetailResult;
import com.elitescloud.cloudt.system.modules.wecom.model.msg.BaseMsg;
import com.elitescloud.cloudt.system.modules.wecom.model.msg.MsgResult;
import com.elitescloud.cloudt.system.modules.wecom.model.user.*;
import com.fasterxml.jackson.core.type.TypeReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.Map;

/**
 * 企业微信工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/16
 */
public class WeComTool {
    private static final Logger logger = LoggerFactory.getLogger(WeComTool.class);
    private static final RestTemplateHelper restTemplateHelper = RestTemplateHelper.instance(RestTemplateFactory.instance());

    /**
     * 获取Token
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/91039'>API文档</a>
     *
     * @param corpid     企业ID
     * @param corpsecret 应用的凭证秘钥
     * @return token信息
     */
    public static AccessToken getToken(@NotBlank String corpid, @NotBlank String corpsecret) {
        Assert.notBlank(corpid, "企业ID为空");
        Assert.notBlank(corpsecret, "应用的凭证秘钥为空");

        if (logger.isDebugEnabled()) {
            logger.debug("创建token：{}, {}", corpid, corpsecret);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/gettoken?corpid=" + corpid + "&corpsecret=" + corpsecret;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.GET, null, new TypeReference<>() {
        });
    }

    /**
     * 创建部门
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90205'>API文档</a>
     *
     * @param accessToken token
     * @param saveParam   部门参数
     * @return 部门ID
     */
    public static DepartmentSaveResult departmentCreate(@NotBlank String accessToken, @NotNull DepartmentSaveParam saveParam) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notNull(saveParam, "部门参数为空");
        Assert.notBlank(saveParam.getName(), "部门名称为空");
        if (saveParam.getParentid() == null) {
            saveParam.setParentid(1L);
        }

        if (logger.isDebugEnabled()) {
            logger.debug("创建部门：{}", JSONUtil.toJsonString(saveParam));
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/department/create?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(saveParam), new TypeReference<>() {
        });
    }

    /**
     * 更新部门
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90206'>API文档</a>
     *
     * @param accessToken token
     * @param saveParam   部门参数
     * @return 部门ID
     */
    public static DepartmentSaveResult departmentUpdate(@NotBlank String accessToken, @NotNull DepartmentSaveParam saveParam) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notNull(saveParam, "部门参数为空");
        Assert.notBlank(saveParam.getName(), "部门名称为空");
        if (saveParam.getParentid() == null) {
            saveParam.setParentid(1L);
        }

        if (logger.isDebugEnabled()) {
            logger.debug("更新部门：{}", JSONUtil.toJsonString(saveParam));
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/department/update?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(saveParam), new TypeReference<>() {
        });
    }

    /**
     * 删除部门
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90207'>API文档</a>
     *
     * @param accessToken token
     * @param id          部门ID
     * @return token信息
     */
    public static DepartmentDeleteResult departmentDelete(@NotBlank String accessToken, @NotNull Long id) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notNull(id, "部门ID为空");

        if (logger.isDebugEnabled()) {
            logger.debug("删除部门：{}", id);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/department/delete?access_token=" + accessToken + "&id=" + id;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.GET, null, new TypeReference<>() {
        });
    }

    /**
     * 获取子部门ID列表
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/95350'>API文档</a>
     *
     * @param accessToken token
     * @param id          部门ID，为空则获取所有的
     * @return token信息
     */
    public static DepartmentIdListResult departmentIdList(@NotBlank String accessToken, Long id) {
        Assert.notBlank(accessToken, "AccessToken为空");

        if (logger.isDebugEnabled()) {
            logger.debug("获取子部门ID列表：{}", id);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/department/simplelist?access_token=" + accessToken;
        if (id != null) {
            url += "&id=" + id;
        }
        return restTemplateHelper.exchangeSafely(url, HttpMethod.GET, null, new TypeReference<>() {
        });
    }

    /**
     * 创建成员
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90195'>API文档</a>
     *
     * @param accessToken token
     * @param saveParam   成员参数
     * @return 创建结果
     */
    public static UserCreateResultParam userCreate(@NotBlank String accessToken, @NotNull UserSaveParam saveParam) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notNull(saveParam, "成员参数为空");
        Assert.notBlank(saveParam.getUserid(), "成员ID为空");
        Assert.notBlank(saveParam.getName(), "成员名称为空");

        if (logger.isDebugEnabled()) {
            logger.debug("创建成员：{}", JSONUtil.toJsonString(saveParam));
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/create?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(saveParam), new TypeReference<>() {
        });
    }

    /**
     * 更新成员
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90197'>API文档</a>
     *
     * @param accessToken token
     * @param saveParam   成员参数
     * @return 更新结果
     */
    public static CommonResult userUpdate(@NotBlank String accessToken, @NotNull UserSaveParam saveParam) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notNull(saveParam, "成员参数为空");
        Assert.notBlank(saveParam.getUserid(), "成员ID为空");
        Assert.notBlank(saveParam.getName(), "成员名称为空");

        if (logger.isDebugEnabled()) {
            logger.debug("更新成员：{}", JSONUtil.toJsonString(saveParam));
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/update?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(saveParam), new TypeReference<>() {
        });
    }

    /**
     * 删除成员
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90198'>API文档</a>
     *
     * @param accessToken token
     * @param userId      成员ID
     * @return 删除结果
     */
    public static CommonResult userDelete(@NotBlank String accessToken, @NotBlank String userId) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notBlank(userId, "成员ID为空");

        if (logger.isDebugEnabled()) {
            logger.debug("删除成员：{}", userId);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/delete?access_token=" + accessToken + "&userid=" + userId;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.GET, null, new TypeReference<>() {
        });
    }

    /**
     * 批量删除成员
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90199'>API文档</a>
     *
     * @param accessToken token
     * @param param       参数
     * @return 删除结果
     */
    public static CommonResult userDeleteBatch(@NotBlank String accessToken, @NotNull UserDeleteBatchParam param) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notEmpty(param == null ? null : param.getUseridlist(), "成员ID为空");

        if (logger.isDebugEnabled()) {
            logger.debug("删除成员：{}", JSONUtil.toJsonString(param));
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/batchdelete?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(param), new TypeReference<>() {
        });
    }

    /**
     * userid转openid
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90202'>API文档</a>
     *
     * @param accessToken token
     * @param userId      成员ID
     * @return 删除结果
     */
    public static UserId2OpenIdResult userId2OpenId(@NotBlank String accessToken, @NotBlank String userId) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notBlank(userId, "成员ID为空");

        if (logger.isDebugEnabled()) {
            logger.debug("userid转openid：{}", userId);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/convert_to_openid?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(Map.of("userid", userId)), new TypeReference<>() {
        });
    }

    /**
     * openid转userid
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90202'>API文档</a>
     *
     * @param accessToken token
     * @param openId      成员的OpenId
     * @return 删除结果
     */
    public static OpenId2UserIdResult openId2UserId(@NotBlank String accessToken, @NotBlank String openId) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notBlank(openId, "OpenId为空");

        if (logger.isDebugEnabled()) {
            logger.debug("openid转userid：{}", openId);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/convert_to_userid?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(Map.of("openid", openId)), new TypeReference<>() {
        });
    }

    /**
     * 获取用户登录身份
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/98176'>API文档</a>
     *
     * @param accessToken token
     * @param code      授权码
     * @return 身份信息结果
     */
    public static Code2UserInfoResult code2UserInfo(@NotBlank String accessToken, @NotBlank String code) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notBlank(code, "授权码为空");

        if (logger.isDebugEnabled()) {
            logger.debug("code转用户身份：{}", code);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/auth/getuserinfo?access_token=" + accessToken + "&code=" + code;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.GET, null, new TypeReference<>() {
        });
    }

    /**
     * 获取用户登录身份
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/98176'>API文档</a>
     *
     * @param accessToken token
     * @param ticket      票据
     * @return 身份信息结果
     */
    public static Ticket2UserDetailResult ticket2UserDetail(@NotBlank String accessToken, @NotBlank String ticket) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notBlank(ticket, "票据为空");

        if (logger.isDebugEnabled()) {
            logger.debug("ticket转用户身份：{}", ticket);
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/auth/getuserdetail?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(Map.of("user_ticket", ticket)), new TypeReference<>() {
        });
    }

    /**
     * 发送应用消息
     * <p>
     * <a href='https://developer.work.weixin.qq.com/document/path/90236'>API文档</a>
     *
     * @param accessToken token
     * @param msg         消息体
     * @return 发送结果
     */
    public static MsgResult sendMessage(@NotBlank String accessToken, @NotNull Object msg) {
        Assert.notBlank(accessToken, "AccessToken为空");
        Assert.notNull(msg, "消息体为空");

        if (logger.isDebugEnabled()) {
            logger.debug("发送应用消息：{}", JSONUtil.toJsonString(msg));
        }

        String url = "https://qyapi.weixin.qq.com/cgi-bin/message/send?access_token=" + accessToken;
        return restTemplateHelper.exchangeSafely(url, HttpMethod.POST, new HttpEntity<>(msg), new TypeReference<>() {
        });
    }
}
