package com.elitescloud.cloudt.system.provider.orgsync;

import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.constant.SysThirdPartyAccountBusinessType;
import com.elitescloud.cloudt.system.modules.wecom.util.WeComTool;
import com.elitescloud.cloudt.system.service.ThirdPartAccountQueryService;
import com.elitescloud.cloudt.system.service.common.constant.ThirdPartAccountType;
import com.elitescloud.cloudt.system.service.common.constant.ThirdPartyConstant;
import org.springframework.util.StringUtils;

import java.util.concurrent.TimeUnit;

/**
 * 企微同步.
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/26
 */
abstract class BaseWecomSync {

    /**
     * 获取token
     *
     * @param cfg
     * @return
     */
    static String getAccessTokenOfWecom(WecomConfig cfg) {
        var redisUtils = SpringContextHolder.getBean(RedisUtils.class);
        String tokenKey = "wecom:accessToken:" + cfg.corpid + ":" + cfg.corpsecret;

        // 先从缓存获取
        String token = (String) redisUtils.get(tokenKey);
        if (StringUtils.hasText(token)) {
            return token;
        }

        // 生成token
        var accessToken = WeComTool.getToken(cfg.corpid, cfg.corpsecret);
        if (accessToken.isSuccess()) {
            token = accessToken.getAccess_token();
            redisUtils.set(tokenKey, token, accessToken.getExpires_in() - 60, TimeUnit.SECONDS);
            return token;
        }
        throw new BusinessException("企业微信授权失败," + ObjUtil.defaultIfNull(accessToken.getErrcode(), -1) +
                ObjUtil.defaultIfNull(accessToken.getErrmsg(), ""));
    }

    /**
     * 获取配置信息
     *
     * @return
     */
    static WecomConfig getConfig() {
        var tenant = SpringContextHolder.getBean(TenantClientProvider.class).getSessionTenant();
        var accountCfg = SpringContextHolder.getBean(ThirdPartAccountQueryService.class).get(tenant == null ? TenantConstant.DEFAULT_TENANT_ID : tenant.getId(),
                ThirdPartAccountType.WECOM, SysThirdPartyAccountBusinessType.DEFAULT).computeData();
        if (accountCfg == null) {
            return null;
        }
        WecomConfig cfg = new WecomConfig((String) accountCfg.getConfigValue(ThirdPartyConstant.CFG_WECOM_CORPID), (String) accountCfg.getConfigValue(ThirdPartyConstant.CFG_WECOM_CORPSECRET));
        cfg.setToken((String) accountCfg.getConfigValue(ThirdPartyConstant.CFG_WECOM_TOKEN));
        cfg.setEncodingAesKey((String) accountCfg.getConfigValue(ThirdPartyConstant.CFG_WECOM_ENCODING_AES_KEY));
        return cfg;
    }

    static class WecomConfig {
        private final String corpid;
        private final String corpsecret;
        private String token;
        private String encodingAesKey;

        protected WecomConfig(String corpid, String corpsecret) {
            this.corpid = corpid;
            this.corpsecret = corpsecret;
        }

        public String getCorpid() {
            return corpid;
        }

        public String getCorpsecret() {
            return corpsecret;
        }

        public String getToken() {
            return token;
        }

        public void setToken(String token) {
            this.token = token;
        }

        public String getEncodingAesKey() {
            return encodingAesKey;
        }

        public void setEncodingAesKey(String encodingAesKey) {
            this.encodingAesKey = encodingAesKey;
        }
    }
}
