package com.elitescloud.cloudt.system.service.alert;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.crypto.digest.MD5;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.system.model.vo.resp.extend.AlertConfigBaseRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.AlertConfigBaseSaveVO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serial;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.List;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2025/10/22 周三
 */
abstract class AbstractAlertProvider<T extends AlertConfigBaseSaveVO, R extends AlertConfigBaseRespVO> implements AlertProvider<T, R> {
    private static final Logger logger = LoggerFactory.getLogger(AbstractAlertProvider.class);

    private final RedisUtils redisUtils;

    public AbstractAlertProvider(RedisUtils redisUtils) {
        this.redisUtils = redisUtils;
    }

    protected boolean isDuplicate(BaseCfg cfg, Map<String, Object> tmplParams, String category) {
        if (!Boolean.TRUE.equals(cfg.getDeduplicate())) {
            // 不需要去重
            return false;
        }

        // 拼接去重字段
        StringBuffer txt = new StringBuffer();
        for (String field : cfg.getDeduplicateFields()) {
            var key = this.normalizeTmplField(field);
            var value = ObjectUtil.defaultIfNull(tmplParams.get(key), "").toString();
            if (CharSequenceUtil.isBlank(value)) {
                // 空值忽略
                continue;
            }
            txt.append(value + ":");
        }

        if (txt.length() == 0) {
            // 没有有效值
            return false;
        }

        var digestHex = MD5.create().digestHex(ObjectUtil.defaultIfNull(category, "default") + ":" + txt, StandardCharsets.UTF_8);
        var exists = redisUtils.getRedisTemplate().opsForValue().setIfAbsent(digestHex, "1", Duration.ofMinutes(cfg.getDeduplicateIntervals()));
        logger.info("去重：{}，{}", txt, exists);
        return exists != null && !exists;
    }

    protected String normalizeTmplField(String tmplField) {
        return CharSequenceUtil.strip(tmplField, "{", "}");
    }

    static class BaseCfg implements Serializable {
        @Serial
        private static final long serialVersionUID = -3924675064627694599L;

        private String tmplContent;
        /**
         * 是否去重
         */
        private Boolean deduplicate;

        /**
         * 去重字段
         */
        private List<String> deduplicateFields;

        /**
         * 去重间隔
         * <p>
         * 单位分钟
         */
        private Integer deduplicateIntervals;

        public BaseCfg(String tmplContent) {
            this.tmplContent = tmplContent;
        }

        public String getTmplContent() {
            return tmplContent;
        }

        public void setTmplContent(String tmplContent) {
            this.tmplContent = tmplContent;
        }

        public Boolean getDeduplicate() {
            return deduplicate;
        }

        public void setDeduplicate(Boolean deduplicate) {
            this.deduplicate = deduplicate;
        }

        public List<String> getDeduplicateFields() {
            return deduplicateFields;
        }

        public void setDeduplicateFields(List<String> deduplicateFields) {
            this.deduplicateFields = deduplicateFields;
        }

        public Integer getDeduplicateIntervals() {
            return deduplicateIntervals;
        }

        public void setDeduplicateIntervals(Integer deduplicateIntervals) {
            this.deduplicateIntervals = deduplicateIntervals;
        }
    }
}
