package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitescloud.cloudt.system.constant.MsgSendTypeEnum;
import lombok.extern.slf4j.Slf4j;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * 消息模板.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
@Slf4j
public class MsgTemplateConfigInitProvider extends AbstractBasicDataInitProvider {

    @Override
    public String typeName() {
        return "消息模板配置";
    }

    @Override
    public String tableName() {
        return "sys_msg_template_config";
    }

    @Override
    public List<String> fields() {
        return List.of("template_id", "template_code", "template_name", "send_type_code", "send_type_switch", "mes_subject", "mes_title",
                "mes_rich_content_sign", "external_template_id", "mes_text", "external_sign_name", "std_version",
                "create_time", "modify_time");
    }

    @Override
    public Map<String, String> fieldTitles() {
        Map<String, String> titles = new LinkedHashMap<>(16);
        titles.put("template_code", "模板编码");
        titles.put("template_id", "模板ID");
        titles.put("send_type_code", "发送类型");
        titles.put("send_type_code_name", "发送类型名称");
        titles.put("send_type_switch", "是否启用");
        titles.put("mes_subject", "主题");
        titles.put("mes_title", "标题");
        titles.put("mes_rich_content_sign", "是否是富文本");
        titles.put("external_template_id", "外部模板ID");
        titles.put("mes_text", "内容");
        titles.put("external_sign_name", "外部签名");
        titles.put("create_time", "创建时间");
        titles.put("modify_time", "最后修改时间");
        return titles;
    }

    @Override
    public boolean isTenantData() {
        return true;
    }

    @Override
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.desc("create_time"));
    }

    @Override
    public String fieldAppCode() {
        return "";
    }

    @Override
    public Map<String, Object> convertForExport(Map<String, Object> data, List<Map<String, Object>> preData) {
        var type = MsgSendTypeEnum.parse(ObjectUtil.defaultIfNull(data.get("send_type_code"), "").toString());
        if (type != null) {
            data.put("send_type_code_name", type.getDescription());
        }
        return data;
    }

    @Override
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData, List<Map<String, Object>> lastStepData) {
        var tmplCode = getStringValue(data, "template_code", null);
        Assert.notBlank(tmplCode, "template_code为空");
        this.normalizeBooleanValue(data, "send_type_switch");
        this.normalizeBooleanValue(data, "mes_rich_content_sign");
        var send_type_code = this.getStringValue(data, "send_type_code", null);
        Assert.notBlank(send_type_code, "发送类型为空");

        Assert.notEmpty(lastStepData, "未获取到有效消息模板配置");
        var tmpl = lastStepData.parallelStream()
                .filter(t -> tmplCode.equals(t.get("template_code")))
                .findFirst()
                .orElse(null);
        Assert.notNull(tmpl, "未查询到模板" + tmplCode);

        var tmplId = tmpl.get("id");
        Assert.notNull(tmplId, "模板ID为空");
        data.put("template_id", tmplId);

        if (!currentData.isEmpty()) {
            var id = currentData.parallelStream()
                    .filter(t -> tmplCode.equals(t.get("template_code")) && send_type_code.equals(t.get("send_type_code")))
                    .findFirst()
                    .map(t -> t.get("id"))
                    .orElse(null);
            if (id != null) {
                data.put("id", id);
                return UpdateType.UPDATE;
            }
        }

        return UpdateType.ADD;
    }
}
