package com.elitescloud.cloudt.system.service.impl;

import com.elitescloud.boot.common.param.AreaVO;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.constant.SysOuType;
import com.elitescloud.cloudt.constant.SysRegion;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.system.convert.OuConvert;
import com.elitescloud.cloudt.system.model.vo.query.org.OuPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OuDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.OuPageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.org.OuSaveVO;
import com.elitescloud.cloudt.system.service.AreaQueryService;
import com.elitescloud.cloudt.system.service.OuMngService;
import com.elitescloud.cloudt.system.service.manager.OuMngManager;
import com.elitescloud.cloudt.system.service.model.entity.SysOuDO;
import com.elitescloud.cloudt.system.service.repo.OuRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/2/1
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class OuMngServiceImpl extends BaseServiceImpl implements OuMngService {
    private static final OuConvert CONVERT = OuConvert.INSTANCE;

    @Autowired
    private OuRepoProc repoProc;
    @Autowired
    private OuMngManager manager;

    @Autowired
    private AreaQueryService areaQueryService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(OuSaveVO saveVO) {
        // 参数转换
        var saveBO = CONVERT.saveVo2SaveBO(saveVO, saveVO.getArea());

        // 保存公司
        var ouId = manager.upsert(saveBO);

        return ApiResult.ok(ouId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        manager.delete(id);
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        // 获取当前启用状态
        Boolean enabled = repoProc.getEnabled(id);

        // 修改启用状态
        enabled = enabled == null || !enabled;
        manager.updateEnabled(id, enabled);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<PagingVO<OuPageRespVO>> page(OuPageQueryVO queryVO) {
        // 查询数据
        var pageData = repoProc.pageMng(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData.map(CONVERT::do2PageRespVO));
        }

        // 格式转换
        var ouTypeNames = super.udcMap(new SysOuType());
        var result = pageData.map(t -> {
            var vo = CONVERT.do2PageRespVO(t);
            vo.setOuTypeName(ouTypeNames.get(t.getOuType()));
            return vo;
        });
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<OuDetailRespVO> get(Long id) {
        return repoProc.getOptional(id)
                .map(t -> {
                    var vo = CONVERT.do2DetailRespVO(t);
                    // 公司类型
                    vo.setOuTypeName(super.udcValue(new SysOuType(t.getOuType())));
                    // 电话号码
                    if (StringUtils.hasText(t.getPhone())) {
                        var temp = t.getPhone().split("-");
                        vo.setPhonePrefix(temp[0]);
                        vo.setPhoneNum(temp[1]);
                    }
                    // 法人公司
                    if (t.getLegalOuId() != null) {
                        vo.setLegalOuName(repoProc.getOuName(t.getLegalOuId()));
                    }
                    // 所属区域
                    vo.setRegionName(super.udcValue(new SysRegion(t.getRegion())));
                    //所在地
                    var areaCodes = Stream.of(t.getProvinceCode(), t.getCityCode(), t.getCountyCode()).filter(StringUtils::hasText).collect(Collectors.toSet());
                    if (!areaCodes.isEmpty()) {
                        var areaNameMap = areaQueryService.queryNamesByAreaCode(areaCodes).getData();
                        if (areaNameMap != null && !areaNameMap.isEmpty()) {
                            vo.setArea(this.convertAreaVO(t, areaNameMap));
                        }
                    }

                    return vo;
                })
                .map(ApiResult::ok)
                .orElse(ApiResult.noData());
    }

    private AreaVO convertAreaVO(SysOuDO ouDO, Map<String, String> nameMap) {
        AreaVO areaVO = new AreaVO();
        areaVO.setCountryCode(ouDO.getCountryCode());
        areaVO.setCountryName(nameMap.get(ouDO.getCountryCode()));
        areaVO.setProvinceCode(ouDO.getProvinceCode());
        areaVO.setProvinceName(nameMap.get(ouDO.getProvinceCode()));
        areaVO.setCityCode(ouDO.getCityCode());
        areaVO.setCityName(nameMap.get(ouDO.getCityCode()));
        areaVO.setCountyCode(ouDO.getCountyCode());
        areaVO.setCountyName(nameMap.get(ouDO.getCountyCode()));

        return areaVO;
    }
}
