package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.mq.common.MessageRetryService;
import com.elitescloud.boot.mq.common.model.RetryMessageDTO;
import com.elitescloud.boot.mq.config.CloudtMqAutoConfiguration;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.MqMessageConvert;
import com.elitescloud.cloudt.system.model.entity.SysMqConsumeDO;
import com.elitescloud.cloudt.system.model.entity.SysMqMessageDO;
import com.elitescloud.cloudt.system.model.entity.SysMqRetryDO;
import com.elitescloud.cloudt.system.provider.dto.SysMqConsumeResultDTO;
import com.elitescloud.cloudt.system.provider.dto.SysMqMessageStorageDTO;
import com.elitescloud.cloudt.system.provider.dto.SysMqSendResultDTO;
import com.elitescloud.cloudt.system.service.SysMqMessageMngService;
import com.elitescloud.cloudt.system.service.repo.MqConsumeRepoProc;
import com.elitescloud.cloudt.system.service.repo.MqMessageRepoProc;
import com.elitescloud.cloudt.system.service.repo.MqMessageRetryRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/8/16
 */
@Slf4j
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
@ConditionalOnClass(CloudtMqAutoConfiguration.class)
public class SysMqMessageMngServiceImpl extends BaseServiceImpl implements SysMqMessageMngService {

    @Autowired
    private MqMessageRepoProc repoProc;
    @Autowired
    private MqMessageRetryRepoProc retryRepoProc;
    @Autowired
    private MqConsumeRepoProc consumeRepoProc;
    @Autowired(required = false)
    private MessageRetryService messageRetryService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> saveMessage(SysMqMessageStorageDTO messageDTO) {
        var messageDO = MqMessageConvert.INSTANCE.dto2Do(messageDTO);
        messageDO.setSysTenantId(super.currentTenantId());
        messageDO.setRetryTimes(0);
        messageDO.setRetried(false);
        messageDO.setConsumed(false);

        repoProc.save(messageDO);
        return ApiResult.ok(messageDO.getOriginalMessageId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> updateSendResult(SysMqSendResultDTO sendResult) {
        if (CharSequenceUtil.isBlank(sendResult.getMessageId())) {
            log.error("更新发送结果失败，消息ID为空");
            return ApiResult.fail("更新发送结果失败，消息ID为空");
        }

        // 更新发送结果
        var sendSuccess = Boolean.TRUE.equals(sendResult.getSuccess());
        repoProc.updateSendResult(sendResult.getMessageId(), sendSuccess, sendResult.getFailReason());
        var originalMessageId = repoProc.getOriginalMessageId(sendResult.getMessageId());

        if (sendSuccess) {
            // 发送成功
            if (StringUtils.hasText(originalMessageId)) {
                // 更新原始消息状态
                repoProc.updateSendResult(originalMessageId, true, null);
                // 删除重试记录
                retryRepoProc.deleteRetryByMessageId(originalMessageId, false);
            }
            return ApiResult.ok(sendResult.getMessageId());
        }

        // 发送失败时
        this.createRetryMessage(sendResult.getMessageId(), false);

        return ApiResult.ok(sendResult.getMessageId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> updateConsumeResult(SysMqConsumeResultDTO consumeResult) {
        String originalMessageId = repoProc.getOriginalMessageId(consumeResult.getMessageId());

        // 更新消费结果
        this.updateConsumeResult(consumeResult, consumeResult.getMessageId(), originalMessageId);
        // 更新原始消息
        if (StringUtils.hasText(originalMessageId)) {
            if (Boolean.TRUE.equals(consumeResult.getSuccess())) {
                // 消费成功，则不再重试
                retryRepoProc.deleteRetryByMessageId(originalMessageId);
            }
        }

        // 消费失败，则发起重试
        if (!Boolean.TRUE.equals(consumeResult.getSuccess())) {
            this.createRetryMessage(consumeResult.getMessageId(), true);
        }

        return ApiResult.ok(consumeResult.getMessageId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> deleteRetryMessage(String messageId, String reason) {
        var originalMessageId = repoProc.getOriginalMessageId(messageId);
        if (CharSequenceUtil.isBlank(originalMessageId)) {
            log.error("未查询到原始消息：{}", messageId);
            return ApiResult.fail("未查询到原始消息");
        }

        // 删除重试消息
        retryRepoProc.deleteRetryByMessageId(originalMessageId);
        repoProc.deleteByMessageId(messageId);

        repoProc.updateRetryFailResult(originalMessageId, reason);

        return ApiResult.ok(true);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> trySend(String messageId, Integer version) {
        var originalMessageId = repoProc.getOriginalMessageId(messageId);
        if (CharSequenceUtil.isBlank(originalMessageId)) {
            log.info("未查询到原始消息：{}", messageId);
            return ApiResult.ok(false);
        }
        var retryVersion = retryRepoProc.getVersionByMessageId(originalMessageId);

        var canSend = retryVersion != null && retryVersion.intValue() == ObjectUtil.defaultIfNull(version, 0);
        if (canSend) {
            // 更新消息的发送时间
            repoProc.updateRetrySendTime(messageId, LocalDateTime.now());
        }
        return ApiResult.ok(canSend);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> deleteExpiredMessage(LocalDateTime expiredTime) {
        repoProc.deleteExpiredMessage(expiredTime);
        retryRepoProc.deleteExpiredMessage(expiredTime);
        consumeRepoProc.deleteExpiredMessage(expiredTime);

        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<List<RetryMessageDTO>> queryRetryMessage(String lastMessageId, int size) {
        var retryList = retryRepoProc.queryRetry(lastMessageId, size);
        if (retryList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        var messageIds = retryList.stream().map(SysMqRetryDO::getMessageId).collect(Collectors.toSet());
        var messageMap = repoProc.listByMessageId(messageIds).stream().collect(Collectors.toMap(SysMqMessageDO::getMessageId, t -> t, (t1, t2) -> t1));

        var messageList = retryRepoProc.queryRetry(lastMessageId, size)
                .stream().filter(t -> messageMap.containsKey(t.getMessageId())).map(t -> {
                    var message = messageMap.get(t.getMessageId());

                    RetryMessageDTO messageDTO = new RetryMessageDTO();
                    messageDTO.setRetryTimes(t.getRetryTimes());
                    messageDTO.setRetryTime(t.getSendTimeNext());
                    messageDTO.setSysTenantId(message.getSysTenantId());
                    messageDTO.setTaskId(message.getLastRetryMessageId());
                    messageDTO.setChannel(message.getChannel());
                    messageDTO.setMessageContent(message.getMessageContent());
                    messageDTO.setSendTime(t.getSendTime());
                    messageDTO.setVersion(t.getAuditDataVersion());

                    return messageDTO;
                }).filter(t -> StringUtils.hasText(t.getTaskId())).collect(Collectors.toList());
        return ApiResult.ok(messageList);
    }

    private void updateConsumeResult(SysMqConsumeResultDTO consumeResult, String messageId, String originalMessageId) {
        var msgId = repoProc.getIdByMessageId(messageId);
        if (msgId == null) {
            log.error("未找到消息记录：{}", messageId);
            return;
        }

        if (Boolean.TRUE.equals(consumeResult.getSuccess())) {
            // 消费成功
            repoProc.updateConsumed(msgId, consumeResult.getConsumeTime());
            if (CharSequenceUtil.isNotBlank(originalMessageId)) {
                repoProc.updateConsumedByMessageId(originalMessageId, consumeResult.getConsumeTime());
            }
        }

        // 创建消费记录
        SysMqConsumeDO consumeDO = new SysMqConsumeDO();
        consumeDO.setMessageId(messageId);
        consumeDO.setRecordId(msgId);
        consumeDO.setAppCode(consumeResult.getAppCode());
        consumeDO.setConsumerName(consumeResult.getConsumerName());
        consumeDO.setConsumerIp(consumeResult.getConsumerIp());
        consumeDO.setConsumeTime(consumeResult.getConsumeTime());
        consumeDO.setSuccess(consumeResult.getSuccess());
        consumeDO.setFailReason(consumeResult.getFailReason());
        consumeDO.setOriginalMessageId(CharSequenceUtil.blankToDefault(originalMessageId, messageId));
        consumeDO.setCostTime(consumeResult.getCostTime());

        consumeRepoProc.save(consumeDO);
    }

    private void createRetryMessage(String messageId, Boolean consumer) {
        if (messageRetryService == null) {
            return;
        }

        var originalMessageId = repoProc.getOriginalMessageId(messageId);
        var originalMessageDO = repoProc.getByMessageId(CharSequenceUtil.blankToDefault(originalMessageId, messageId));
        if (originalMessageDO == null) {
            log.info("未查询到消息信息：{}，无法创建重试消息", messageId);
            return;
        }
        if (Boolean.TRUE.equals(originalMessageDO.getConsumed())) {
            // 已消费
            log.info("已成功消费过消息：{}，无需重试", originalMessageDO.getMessageId());
            return;
        }

        // 初始化重试消息
        var messageDO = this.createRetryMessage(originalMessageDO);
        // 保存重试记录
        var retryDO = this.upsertRetryRecord(messageDO, consumer);
        // 发起重试请求
        this.createRetryRequest(messageDO, retryDO);
    }

    private SysMqMessageDO createRetryMessage(SysMqMessageDO original) {
        SysMqMessageDO messageDO = new SysMqMessageDO();
        messageDO.setMessageId(messageRetryService.generateMessageId());
        messageDO.setSysTenantId(original.getSysTenantId());
        messageDO.setAppCode(original.getAppCode());
        messageDO.setChannel(original.getChannel());
        messageDO.setMessageContent(original.getMessageContent());
        messageDO.setBusinessKey(original.getBusinessKey());
        messageDO.setLocal(original.getLocal());
        messageDO.setRetried(true);
        messageDO.setOriginalMessageId(original.getMessageId());
        messageDO.setConsumed(false);
        repoProc.save(messageDO);

        repoProc.updateLastRetryMessageId(original.getMessageId(), messageDO.getMessageId());
        return messageDO;
    }

    private SysMqRetryDO upsertRetryRecord(SysMqMessageDO messageDO, Boolean consumer) {
        var messageId = CharSequenceUtil.blankToDefault(messageDO.getOriginalMessageId(), messageDO.getMessageId());
        var retryDO = retryRepoProc.getByMessageId(messageId, consumer);
        if (retryDO != null) {
            retryDO.setAuditDataVersion(retryDO.getAuditDataVersion() + 1);
            retryDO.setRetryTimes(retryDO.getRetryTimes() + 1);
            retryDO.setSendTime(LocalDateTime.now());
            retryDO.setSendTimeNext(messageRetryService.generateNextRetryTime(retryDO.getSendTime(), retryDO.getRetryTimes()));
            retryRepoProc.save(retryDO);
            return retryDO;
        }

        retryDO = new SysMqRetryDO();
        retryDO.setMessageId(messageId);
        retryDO.setConsumer(consumer);
        retryDO.setRetryTimes(0);
        retryDO.setSendTime(ObjectUtil.defaultIfNull(messageDO.getSendTime(), LocalDateTime.now()));
        retryDO.setSendTimeNext(messageRetryService.generateNextRetryTime(retryDO.getSendTime(), retryDO.getRetryTimes()));
        retryDO.setAuditDataVersion(0);
        retryRepoProc.save(retryDO);

        return retryDO;
    }

    private void createRetryRequest(SysMqMessageDO messageDO, SysMqRetryDO retryDO) {
        RetryMessageDTO failMessageDTO = new RetryMessageDTO();
        failMessageDTO.setSysTenantId(messageDO.getSysTenantId());
        failMessageDTO.setTaskId(messageDO.getMessageId());
        failMessageDTO.setChannel(messageDO.getChannel());
        failMessageDTO.setMessageContent(messageDO.getMessageContent());
        failMessageDTO.setSendTime(retryDO.getSendTime());
        failMessageDTO.setVersion(retryDO.getAuditDataVersion());
        failMessageDTO.setRetryTimes(retryDO.getRetryTimes());
        failMessageDTO.setRetryTime(retryDO.getSendTimeNext());

        messageRetryService.addRetryTask(failMessageDTO);
    }
}
