package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.lang.UUID;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.core.support.common.UnifyQueryClient;
import com.elitescloud.boot.core.support.common.param.UnifyQueryParam;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.openfeign.common.DynamicClientHelper;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.dto.req.UnifyQueryQueryDTO;
import com.elitescloud.cloudt.system.dto.req.UnifySqlQueryQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.UnifyQueryRespDTO;
import com.elitescloud.cloudt.system.model.bo.BusinessParamBO;
import com.elitescloud.cloudt.system.model.bo.UnifyQueryParamBO;
import com.elitescloud.cloudt.system.service.SysUnifyQueryQueryService;
import com.elitescloud.cloudt.system.service.common.constant.UnifyQueryTypeEnum;
import com.elitescloud.cloudt.system.service.model.entity.SysUnifyQueryRecordDO;
import com.elitescloud.cloudt.system.service.repo.AppRepoProc;
import com.elitescloud.cloudt.system.service.repo.BusinessObjectRepoProc;
import com.elitescloud.cloudt.system.service.repo.BusinessParamRepoProc;
import com.elitescloud.cloudt.system.service.repo.UnifyQueryRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.encrypt.TextEncryptor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/6/6
 */
@Slf4j
@Service
public class SysUnifyQueryQueryServiceImpl implements SysUnifyQueryQueryService {

    @Autowired
    private UnifyQueryRepoProc repoProc;
    @Autowired
    private BusinessObjectRepoProc businessObjectRepoProc;
    @Autowired
    private BusinessParamRepoProc businessParamRepoProc;
    @Autowired
    private AppRepoProc appRepoProc;

    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;
    @Autowired
    private TextEncryptor textEncryptor;
    @Autowired
    private RedisUtils redisUtils;
    @Autowired
    private SystemProperties systemProperties;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<UnifyQueryRespDTO> query(UnifyQueryQueryDTO paramDTO) {
        UnifyQueryParamBO paramBO = null;
        try {
            paramBO = this.convertAndCheckParam(paramDTO);
        } catch (Exception e) {
            return ApiResult.fail("查询失败，" + e.getMessage());
        }

        // 保存记录信息
        var recordDO = this.convertParam2DO(paramDTO);
        paramBO.getConditionFileTypes().putAll(paramDTO.getConditionFieldTypes());
        recordDO.setQueryTime(LocalDateTime.now());
        repoProc.save(recordDO);

        // 查询数据
        try {
            var queryResult = this.queryByRpc(recordDO, paramBO);
            var finishTime = LocalDateTime.now();
            String queryResultStr = Boolean.FALSE.equals(systemProperties.getUnifyQuery().getSaveQueryResult()) ? null : JSONUtil.toJsonString(queryResult);

            repoProc.updateQueryResult(recordDO.getId(), true, queryResultStr, null, finishTime, Duration.between(recordDO.getQueryTime(), finishTime).toMillis());
            return ApiResult.ok(new UnifyQueryRespDTO(true, recordDO.getId().toString(), queryResult, null));
        } catch (Exception e) {
            log.error("统一查询异常：{}", recordDO.getId(), e);
            var finishTime = LocalDateTime.now();
            repoProc.updateQueryResult(recordDO.getId(), false, null, e.getMessage(), finishTime, Duration.between(recordDO.getQueryTime(), finishTime).toMillis());
            return ApiResult.ok(new UnifyQueryRespDTO(false, recordDO.getId().toString(), null, e.getMessage()));
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<UnifyQueryRespDTO> queryBySql(UnifySqlQueryQueryDTO paramDTO) {
        UnifyQueryParamBO paramBO = null;
        try {
            paramBO = this.convertAndCheckParam(paramDTO);
        } catch (Exception e) {
            return ApiResult.fail("查询失败，" + e.getMessage());
        }

        // 保存记录信息
        var recordDO = this.convertParam2DO(paramDTO);
        recordDO.setQueryTime(LocalDateTime.now());
        repoProc.save(recordDO);

        // 查询数据
        try {
            var queryResult = this.queryByRpc(recordDO, paramBO);
            var finishTime = LocalDateTime.now();
            String queryResultStr = Boolean.FALSE.equals(systemProperties.getUnifyQuery().getSaveQueryResult()) ? null : JSONUtil.toJsonString(queryResult);
            repoProc.updateQueryResult(recordDO.getId(), true, queryResultStr, null, finishTime, Duration.between(recordDO.getQueryTime(), finishTime).toMillis());
            return ApiResult.ok(new UnifyQueryRespDTO(true, recordDO.getId().toString(), queryResult, null));
        } catch (Exception e) {
            log.error("统一查询异常：{}", recordDO.getId(), e);
            var finishTime = LocalDateTime.now();
            repoProc.updateQueryResult(recordDO.getId(), false, null, e.getMessage(), finishTime, Duration.between(recordDO.getQueryTime(), finishTime).toMillis());
            return ApiResult.ok(new UnifyQueryRespDTO(false, recordDO.getId().toString(), null, e.getMessage()));
        }
    }

    private List<Map<String, Object>> queryByRpc(SysUnifyQueryRecordDO recordDO, UnifyQueryParamBO queryParamBO) {
        var client = DynamicClientHelper.getClient(recordDO.getAppCode(), UnifyQueryClient.class, UnifyQueryClient.URI);
        UnifyQueryParam queryParam = new UnifyQueryParam();
        queryParam.setUnifyId(recordDO.getId().toString());
        queryParam.setUnifyVerifier(UUID.fastUUID().toString());

        if (queryParamBO.getQueryType() == UnifyQueryTypeEnum.JPQL) {
            queryParam.setFields(queryParamBO.getBusinessObjectQuery().getQueryFields());
            queryParam.setJpql(recordDO.getSqlTxt());
        } else if (queryParamBO.getQueryType() == UnifyQueryTypeEnum.SQL) {
            queryParam.setSql(recordDO.getSqlTxt());
        } else {
            throw new BusinessException("暂不支持的查询类型");
        }
        queryParam.setParams(queryParamBO.getConditions());
        queryParam.setParamsFieldTypes(queryParamBO.getConditionFileTypes());

        redisUtils.set(queryParam.getUnifyId(), textEncryptor.encrypt(queryParam.getUnifyVerifier()), 2, TimeUnit.MINUTES);

        ApiResult<List<Map<String, Object>>> rpcResult = client.query(queryParam);
        return rpcResult.computeData();
    }

    private UnifyQueryParamBO convertAndCheckParam(UnifySqlQueryQueryDTO paramDTO) {
        Assert.notNull(paramDTO, "查询参数为空");
        Assert.notBlank(paramDTO.getSql(), "SQL为空");
        Assert.notBlank(paramDTO.getAppCode(), "appCode为空");

        var appCode = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.getCode(paramDTO.getAppCode()));
        Assert.notBlank(appCode, "应用" + paramDTO.getAppCode() + "不存在");

        UnifyQueryParamBO paramBO = new UnifyQueryParamBO();
        paramBO.setAppCode(paramDTO.getAppCode());
        paramBO.setQueryType(UnifyQueryTypeEnum.SQL);
        paramBO.setSqlQuery(new UnifyQueryParamBO.SQL(paramDTO.getSql()));
        paramBO.setConditions(paramDTO.getConditions());
        return paramBO;
    }

    private SysUnifyQueryRecordDO convertParam2DO(UnifySqlQueryQueryDTO paramDTO) {
        SysUnifyQueryRecordDO recordDO = new SysUnifyQueryRecordDO();
        recordDO.setUserId(SecurityContextUtil.currentUserId());
        recordDO.setSuccess(false);

        recordDO.setAppCode(paramDTO.getAppCode());
        recordDO.setBusinessParams(JSONUtil.toJsonString(paramDTO.getConditions()));
        recordDO.setQueryType(UnifyQueryTypeEnum.SQL.name());
        recordDO.setSqlTxt(paramDTO.getSql());
        recordDO.setSqlParams(recordDO.getBusinessParams());

        return recordDO;
    }

    private UnifyQueryParamBO convertAndCheckParam(UnifyQueryQueryDTO paramDTO) {
        Assert.notNull(paramDTO, "查询参数为空");
        Assert.notBlank(paramDTO.getBusinessObjectCode(), "业务对象编码为空");

        List<String> businessParams = CollUtil.isEmpty(paramDTO.getQueryFields()) ? Collections.emptyList() :
                paramDTO.getQueryFields().stream().filter(StringUtils::hasText).collect(Collectors.toList());

        UnifyQueryParamBO paramBO = new UnifyQueryParamBO();
        var appCode = tenantDataIsolateProvider.byDefaultDirectly(() -> businessObjectRepoProc.getAppCode(paramDTO.getBusinessObjectCode()));
        Assert.notBlank(appCode, "业务对象" + paramDTO.getBusinessObjectCode() + "不存在或所属应用不存在");
        paramBO.setAppCode(appCode);
        paramBO.setQueryType(UnifyQueryTypeEnum.JPQL);
        paramBO.setBusinessObjectQuery(new UnifyQueryParamBO.BusinessObject(paramDTO.getBusinessObjectCode(), businessParams));
        paramBO.setConditions(paramDTO.getConditions());
        paramBO.setConditionFileTypes(ObjUtil.defaultIfNull(paramDTO.getConditionFieldTypes(), new HashMap<>(0)));
        return paramBO;
    }

    private SysUnifyQueryRecordDO convertParam2DO(UnifyQueryQueryDTO paramDTO) {
        SysUnifyQueryRecordDO recordDO = new SysUnifyQueryRecordDO();
        recordDO.setUserId(SecurityContextUtil.currentUserId());
        recordDO.setSuccess(false);

        // 业务对象
        recordDO.setBusinessObjectCode(paramDTO.getBusinessObjectCode());
        var paramAll = tenantDataIsolateProvider.byDefaultDirectly(() -> {
                    var businessObject = businessObjectRepoProc.getSimple(paramDTO.getBusinessObjectCode());
                    if (businessObject == null || Boolean.FALSE.equals(businessObject.getEnabled())) {
                        throw new BusinessException("业务对象不存在或已禁用");
                    }

                    recordDO.setAppCode(businessObject.getAppCode());
                    if (CharSequenceUtil.isBlank(businessObject.getAppCode())) {
                        throw new BusinessException("未知业务对象的所属应用");
                    }

                    return businessParamRepoProc.listSimpleBoByBusinessObjectCode(paramDTO.getBusinessObjectCode());
                }).stream()
                .collect(Collectors.toMap(BusinessParamBO::getFieldName, Function.identity(), (t1, t2) -> t1));
        if (paramAll.isEmpty()) {
            throw new BusinessException("业务对象的参数为空");
        }

        // 查询字段
        for (String queryField : paramDTO.getQueryFields()) {
            if (!paramAll.containsKey(queryField)) {
                throw new BusinessException("业务对象" + paramDTO.getBusinessObjectCode() + "不存在参数: " + queryField);
            }
        }
        recordDO.setBusinessFields(String.join(",", paramDTO.getQueryFields()));

        if (CollUtil.isNotEmpty(paramDTO.getConditions())) {
            if (paramDTO.getConditionFieldTypes() == null) {
                paramDTO.setConditionFieldTypes(new HashMap<>(paramDTO.getConditions().size()));
            }
            for (String s : paramDTO.getConditions().keySet()) {
                if (!paramAll.containsKey(s)) {
                    throw new BusinessException("业务对象" + paramDTO.getBusinessObjectCode() + "不存在参数: " + s);
                }
                paramDTO.getConditionFieldTypes().putIfAbsent(s, paramAll.get(s).getFieldJavaType());
            }
        }
        recordDO.setBusinessParams(JSONUtil.toJsonString(paramDTO.getConditions()));
        recordDO.setQueryType(UnifyQueryTypeEnum.JPQL.name());
        recordDO.setSqlTxt(this.buildSQL(paramDTO, paramAll));
        recordDO.setSqlParams(recordDO.getBusinessParams());

        return recordDO;
    }

    private String buildSQL(UnifyQueryQueryDTO paramDTO, Map<String, BusinessParamBO> businessParamMap) {
        StringBuilder sql = new StringBuilder();
        String tableAlias = "t";
        String entityName = this.obtainEntityName(new ArrayList<>(businessParamMap.values()).get(0));

        // 查询字段
        sql.append("select ");
        int i = 0;
        for (String queryField : paramDTO.getQueryFields()) {
            if (i == 0) {
                sql.append(tableAlias).append(".").append(queryField);
            } else {
                sql.append(", ").append(tableAlias).append(".").append(queryField);
            }

            i++;
        }
        sql.append(" from ").append(entityName).append(" ").append(tableAlias).append(" ");

        // 查询条件
        if (CollUtil.isNotEmpty(paramDTO.getConditions())) {
            sql.append(" where ");
            i = 0;
            String conditionRelation = paramDTO.getAnd() == null || paramDTO.getAnd() ? "and" : "or";
            boolean valueEquals = paramDTO.getEquals() == null || paramDTO.getEquals();
            String valueRelation = "=";
            Class<?> valueType = null;
            for (Map.Entry<String, Object> entry : paramDTO.getConditions().entrySet()) {
                valueType = entry.getValue() == null ? null : entry.getValue().getClass();
                valueRelation = (valueType != null && valueType.isArray() || entry.getValue() instanceof Iterable) ? (valueEquals ? "in" : "not in") : (valueEquals ? "=" : "<>");
                // 与上一个条件的关系
                if (i != 0) {
                    sql.append(" ").append(conditionRelation).append(" ");
                }

                // 条件的字段名
                sql.append(entry.getKey()).append(" ");

                // 条件的字段值
                if (entry.getValue() == null) {
                    sql.append("is null");
                } else {
                    sql.append(valueRelation).append(" :").append(entry.getKey());
                }

                i++;
            }
        }

        return sql.toString();
    }

    private String obtainEntityName(BusinessParamBO businessParam) {
        if (CharSequenceUtil.isBlank(businessParam.getEntityClassName())) {
            return null;
        }

        return businessParam.getEntityClassName().substring(businessParam.getEntityClassName().lastIndexOf(".") + 1);
    }
}
