package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.auth.cas.common.PwdStrategyTypeEnum;
import com.elitescloud.boot.auth.cas.provider.PwdStrategyTransferHelper;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.platform.service.repo.SysPlatformAppRepoProc;
import com.elitescloud.cloudt.system.common.OuterAppAuthProvider;
import com.elitescloud.cloudt.system.model.vo.CommonTenantAppVO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonAppPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonDataRelationQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.extend.CurrencyRateQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.udc.UdcPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonDataRelationRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonDataSelectorListRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonTaxRateRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.OuterAppSettingRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.PwdStrategyRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.udc.UdcWithValuesPageRespVO;
import com.elitescloud.cloudt.system.service.*;
import com.elitescloud.cloudt.system.service.common.constant.AppAuthTypeEnum;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformAppDO;
import com.elitescloud.cloudt.system.service.model.vo.CommonAreaQueryVO;
import com.elitescloud.cloudt.system.service.model.vo.CommonAreaTreeQueryVO;
import com.elitescloud.cloudt.system.service.model.vo.CommonAreaTreeRespVO;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/11
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class SystemQueryServiceImpl extends BaseServiceImpl implements SystemQueryService {

    @Autowired
    private SysPlatformAppRepoProc appRepoProc;
    @Autowired
    private AreaQueryService areaQueryService;
    @Autowired
    private CurrencyQueryService currencyQueryService;
    @Autowired
    private TaxRateQueryService taxRateQueryService;
    @Autowired
    private CurrencyRateQueryService currencyRateQueryService;
    @Autowired(required = false)
    private PwdStrategyTransferHelper pwdStrategyTransferHelper;
    @Autowired
    private ObjectProvider<OuterAppAuthProvider> outerAppAuthProviders;
    @Autowired
    private DataSelectorQueryService dataSelectorQueryService;
    @Autowired
    private DataRelationInstQueryService dataRelationInstQueryService;
    @Autowired
    private UdcQueryService udcQueryService;

    @Override
    public ApiResult<List<CommonTenantAppVO>> listApp(Boolean outerApp, Boolean authed) {
        Set<String> filterAppCodes = new HashSet<>(64);
        if (super.isTenantUser()) {
            // 如果是租户下的用户，则返回租户的应用列表，否则返回所有
            // 获取租户的应用
            var tenant = super.currentTenant();
            if (tenant == null || CollectionUtils.isEmpty(tenant.getAppCodes())) {
                return ApiResult.ok(Collections.emptyList());
            }
            filterAppCodes.addAll(tenant.getAppCodes());
        }

        // 获取所有
        var appList = tenantDataIsolateProvider.byDefaultDirectly(() -> appRepoProc.allEnabled(outerApp, authed))
                .stream()
                .filter(t -> filterAppCodes.isEmpty() || filterAppCodes.contains(t.getAppCode()))
                .map(t -> new CommonTenantAppVO(t.getAppCode(), t.getAppName(), t.getIcon(), t.getAppOrder(),
                        t.getAdaptedTerminal()))
                .collect(Collectors.toList());
        return ApiResult.ok(appList);
    }

    @Override
    public ApiResult<PagingVO<CommonTenantAppVO>> pageApp(CommonAppPageQueryVO queryVO) {
        var listData = listApp(queryVO.getOuterApp(), queryVO.getAuthed()).getData();
        if (CollectionUtils.isEmpty(listData)) {
            return ApiResult.ok(PagingVO.empty());
        }

        // 转为分页信息
        var pageRequest = queryVO.getPageRequest();
        int offset = (int) pageRequest.getOffset();
        var recordList = CollUtil.sub(listData, offset, offset + pageRequest.getPageSize());
        var pageData = PagingVO.<CommonTenantAppVO>builder()
                .total(listData.size()).records(recordList)
                .build();
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<List<CommonAreaTreeRespVO>> listArea(CommonAreaQueryVO queryVO) {
        return areaQueryService.listArea(queryVO);
    }

    @Override
    public ApiResult<List<CommonAreaTreeRespVO>> treeArea(CommonAreaTreeQueryVO queryVO) {
        return areaQueryService.treeArea(queryVO);
    }

    @Override
    public ApiResult<List<CodeNameParam>> listCurrency() {
        return currencyQueryService.listCurrency();
    }

    @Override
    public ApiResult<List<CommonTaxRateRespVO>> listTaxRate(String taxRateNo, String taxRateType) {
        return taxRateQueryService.queryList(taxRateNo, taxRateType);
    }

    @Override
    public ApiResult<Double> queryCurrentRate(CurrencyRateQueryVO queryVO) {
        return currencyRateQueryService.findRatio(queryVO);
    }

    @Override
    public ApiResult<List<PwdStrategyRespVO>> listPwdStrategyForCreateAccount() {
        if (pwdStrategyTransferHelper == null) {
            return ApiResult.fail("系统异常，请稍后再试");
        }

        var queryResult = pwdStrategyTransferHelper.listByType(PwdStrategyTypeEnum.PWD_UPDATE);
        if (Boolean.FALSE.equals(queryResult.getSuccess())) {
            return ApiResult.fail(queryResult.getMsg());
        }
        var strategyList = queryResult.getData().stream().map(t -> {
            PwdStrategyRespVO respVO = new PwdStrategyRespVO();
            respVO.setStrategyCode(t.getCode());
            respVO.setStrategyName(t.getName());
            respVO.setExpression(t.getExpression());
            respVO.setParams(ObjectUtil.defaultIfNull(t.getParameters(), () -> new String[0]));

            return respVO;
        }).collect(Collectors.toList());

        return ApiResult.ok(strategyList);
    }

    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Override
    public ApiResult<List<OuterAppSettingRespVO>> listOuterAppSettings() {
        var currentTenant = super.currentTenant();
        // 获取外部应用
        var outerAppList = appRepoProc.listOuterApp(currentTenant == null ? null : currentTenant.getId());
        if (outerAppList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 过滤自定义授权的
        Set<String> authedCustomAppCodes = outerAppList.stream().filter(t -> AppAuthTypeEnum.CUSTOM.name().equals(t.getAuthType()))
                .map(SysPlatformAppDO::getAppCode).collect(Collectors.toSet());
        if (!authedCustomAppCodes.isEmpty()) {
            var authProvider = outerAppAuthProviders.getIfUnique();
            authedCustomAppCodes = authProvider == null ? Collections.emptySet() :
                    ObjectUtil.defaultIfNull(authProvider.filterAuthed(authedCustomAppCodes), Collections.emptySet());
        }

        Set<String> finalAuthedCustomAppCodes = authedCustomAppCodes;
        var appList = outerAppList.stream().map(t -> {
                    OuterAppSettingRespVO respVO = new OuterAppSettingRespVO();
                    respVO.setAppCode(t.getAppCode());
                    respVO.setAppName(t.getAppName());
                    respVO.setIcon(t.getIcon());
                    respVO.setShow(AppAuthTypeEnum.UNNECESSARY.name().equals(t.getAuthType()) || finalAuthedCustomAppCodes.contains(t.getAppCode()));
                    respVO.setUrl(t.getUrl());
                    respVO.setTokenUrl(t.getTokenUrl());

                    return respVO;
                })
                .collect(Collectors.toList());
        return ApiResult.ok(appList);
    }

    @Override
    public ApiResult<List<CommonDataSelectorListRespVO>> listTree(Boolean tree) {
        return dataSelectorQueryService.listTree(tree);
    }

    @Override
    public ApiResult<List<CommonDataRelationRespVO>> queryRelation(CommonDataRelationQueryVO queryVO) {
        return dataRelationInstQueryService.query(queryVO);
    }

    @Override
    public ApiResult<List<CommonDataRelationRespVO>> queryRelationByRefData(CommonDataRelationQueryVO queryVO) {
        return dataRelationInstQueryService.queryByRefData(queryVO);
    }

    @Override
    public ApiResult<PagingVO<UdcWithValuesPageRespVO>> pageQueryUdc(UdcPageQueryVO queryVO) {
        return udcQueryService.pageQuery(queryVO);
    }
}
