package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.TaxRateType;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.system.cacheable.SysCacheTaxRateRpcService;
import com.elitescloud.cloudt.system.convert.TaxRateConvert;
import com.elitescloud.cloudt.system.model.vo.query.extend.TaxRatePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.TaxRateDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.TaxRatePageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.TaxRateSaveVO;
import com.elitescloud.cloudt.system.service.TaxRateMngService;
import com.elitescloud.cloudt.system.service.model.entity.SysTaxRateDO;
import com.elitescloud.cloudt.system.service.repo.TaxRateRepo;
import com.elitescloud.cloudt.system.service.repo.TaxRateRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.util.Collections;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/2/1
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class TaxRateMngServiceImpl extends BaseServiceImpl implements TaxRateMngService {
    private static final TaxRateConvert CONVERT = TaxRateConvert.INSTANCE;

    @Autowired
    private TaxRateRepo repo;
    @Autowired
    private TaxRateRepoProc repoProc;

    @Autowired
    private SysCacheTaxRateRpcService cacheTaxRateRpcService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(TaxRateSaveVO saveVO) {
        // 保存参数转换与校验
        SysTaxRateDO taxRateDO;
        try {
            taxRateDO = saveVO.getId() == null ? this.convertForInsert(saveVO) : this.convertForUpdate(saveVO);
        } catch (IllegalArgumentException e) {
            return ApiResult.fail("保存失败，" + e.getMessage());
        }

        // 保存数据
        repo.save(taxRateDO);

        // 清空缓存
        clearCache();

        return ApiResult.ok(taxRateDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        repoProc.delete(id);

        // 清空缓存
        clearCache();
        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        // 获取当前启用状态
        Boolean enabled = repoProc.getEnabled(id);

        // 修改启用状态
        enabled = enabled == null || !enabled;
        repoProc.updateEnabled(id, enabled);

        // 清空缓存
        clearCache();
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<PagingVO<TaxRatePageRespVO>> page(TaxRatePageQueryVO queryVO) {
        var pageDO = repoProc.pageMng(queryVO);
        if (pageDO.isEmpty()) {
            // 未查询到数据
            return ApiResult.ok(new PagingVO<>(pageDO.getTotal(), Collections.emptyList()));
        }

        var udcMap = super.udcMap(TaxRateType.OUTPUT);
        var pageData = repoProc.pageMng(queryVO)
                .map(t -> {
                    var respVO = CONVERT.do2PageRespVO(t);
                    respVO.setTaxRateTypeName(udcMap.get(t.getTaxRateType()));
                    return respVO;
                });
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<TaxRateDetailRespVO> get(Long id) {
        return repoProc.getOptional(id)
                .map(t -> {
                    var respVO = CONVERT.do2DetailRespVO(t);
                    if (CharSequenceUtil.hasBlank(t.getTaxRateType())) {
                        respVO.setTaxRateTypeName(super.udcValue(new TaxRateType(t.getTaxRateType())));
                    }
                    return respVO;
                })
                .map(ApiResult::ok)
                .orElse(ApiResult.noData());
    }

    private void clearCache() {
        cacheTaxRateRpcService.clearCache();
    }

    private SysTaxRateDO convertForInsert(TaxRateSaveVO saveVO) {
        // 税率编码
        var exists = repoProc.existsRateNo(saveVO.getTaxRateNo(), null);
        Assert.isTrue(!exists, "税率编码已存在");

        // 税率值
        Assert.isTrue(saveVO.getTaxRateValue().compareTo(BigDecimal.ZERO) >= 0, "税率值不能小于0");

        if (saveVO.getEnabled() == null) {
            saveVO.setEnabled(true);
        }

        return CONVERT.saveVo2Do(saveVO);
    }

    private SysTaxRateDO convertForUpdate(TaxRateSaveVO saveVO) {
        var taxRateDO = repoProc.get(saveVO.getId());
        Assert.notNull(taxRateDO, "修改的数据不存在");

        // 税率编码
        if (!taxRateDO.getTaxRateNo().equals(saveVO.getTaxRateNo())) {
            var exists = repoProc.existsRateNo(saveVO.getTaxRateNo(), null);
            Assert.isTrue(!exists, "税率编码已存在");
        }

        // 税率值
        Assert.isTrue(saveVO.getTaxRateValue().compareTo(BigDecimal.ZERO) >= 0, "税率值不能小于0");

        if (saveVO.getEnabled() == null) {
            saveVO.setEnabled(true);
        }

        CONVERT.copySaveVo2Do(saveVO, taxRateDO);
        return taxRateDO;
    }
}
