package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.constant.SysThirdPartyAccountBusinessType;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.model.vo.resp.extend.*;
import com.elitescloud.cloudt.system.model.vo.save.extend.*;
import com.elitescloud.cloudt.system.service.ThirdPartAccountMngService;
import com.elitescloud.cloudt.system.service.callback.ThirdPartyAccountChangedCallback;
import com.elitescloud.cloudt.system.service.common.constant.ThirdPartAccountType;
import com.elitescloud.cloudt.system.service.common.constant.ThirdPartyConstant;
import com.elitescloud.cloudt.system.service.model.entity.SysThirdPartyAccountDO;
import com.elitescloud.cloudt.system.service.repo.ThirdPartyAccountRepo;
import com.elitescloud.cloudt.system.service.repo.ThirdPartyAccountRepoProc;
import com.fasterxml.jackson.core.type.TypeReference;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.HashMap;
import java.util.Map;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2/10/2023
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class ThirdPartAccountMngServiceImpl extends BaseServiceImpl implements ThirdPartAccountMngService {

    @Autowired
    private ThirdPartyAccountRepo repo;
    @Autowired
    private ThirdPartyAccountRepoProc repoProc;

    @Autowired
    private ObjectProvider<ThirdPartyAccountChangedCallback> changedCallbacks;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveWechatMiniApp(WechatMiniProgramSaveVO saveVO, String businessType) {
        // 转为do
        var accountDO = this.convertWechatSaveVO(saveVO, ThirdPartAccountType.WECHAT_MINI_APP, businessType);

        // 保存账号
        this.saveAccount(accountDO, saveVO.getId() == null);

        return ApiResult.ok(accountDO.getId());
    }

    @Override
    public ApiResult<WechatMiniProgramRespVO> getWechatMiniApp(String businessType) {
        var respVO = this.fillWechatRespVO(new WechatMiniProgramRespVO(), ThirdPartAccountType.WECHAT_MINI_APP, businessType);
        return ApiResult.ok(respVO);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveWechatService(WechatServiceSaveVO saveVO, String businessType) {
        // 转为do
        var accountDO = this.convertWechatSaveVO(saveVO, ThirdPartAccountType.WECHAT_SERVICE, businessType);

        // 保存账号
        this.saveAccount(accountDO, saveVO.getId() == null);

        return ApiResult.ok(accountDO.getId());
    }

    @Override
    public ApiResult<WechatServiceRespVO> getWechatService(String businessType) {
        var respVO = this.fillWechatRespVO(new WechatServiceRespVO(), ThirdPartAccountType.WECHAT_SERVICE, businessType);
        return ApiResult.ok(respVO);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveWechatSubscription(WechatSubscriptionSaveVO saveVO, String businessType) {
        // 转为do
        var accountDO = this.convertWechatSaveVO(saveVO, ThirdPartAccountType.WECHAT_SUBSCRIPTION, businessType);

        // 保存账号
        this.saveAccount(accountDO, saveVO.getId() == null);

        return ApiResult.ok(accountDO.getId());
    }

    @Override
    public ApiResult<WechatSubscriptionRespVO> getWechatSubscription(String businessType) {
        var respVO = this.fillWechatRespVO(new WechatSubscriptionRespVO(), ThirdPartAccountType.WECHAT_SUBSCRIPTION, businessType);
        return ApiResult.ok(respVO);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> saveWecom(WecomSaveVO saveVO, String businessType) {
        // 转为do
        var accountDO = this.convertWecomSaveVO(saveVO, ThirdPartAccountType.WECOM, businessType);

        // 保存账号
        this.saveAccount(accountDO, saveVO.getId() == null);

        return ApiResult.ok(accountDO.getId());
    }

    @Override
    public ApiResult<WecomRespVO> getWecom(String businessType) {
        var respVO = this.fillWecomRespVO(new WecomRespVO(), ThirdPartAccountType.WECOM, businessType);
        return ApiResult.ok(respVO);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateCasId(Long id, Long casId) {
        repoProc.updateCasId(id, casId);
        return ApiResult.ok(id);
    }

    private void saveAccount(SysThirdPartyAccountDO accountDO, boolean isAdd) {
        // 保存
        repo.save(accountDO);

        // 保存后的回调
        changedCallbacks.forEach(t -> t.onUpsert(isAdd, accountDO));
    }

    private <T extends BaseWecomAppSaveVO> SysThirdPartyAccountDO convertWecomSaveVO(T saveVO, ThirdPartAccountType accountType, String businessType) {
        var tenantId = super.currentTenantId();
        if (CharSequenceUtil.isBlank(businessType)) {
            businessType = SysThirdPartyAccountBusinessType.DEFAULT.getValue();
        }
        // 判断是否已存在
        var existsId = repoProc.getIdByType(accountType, businessType, tenantId);

        SysThirdPartyAccountDO accountDO = null;
        String account = saveVO.getCorpid() + ":" + CharSequenceUtil.blankToDefault(saveVO.getAgentId(), "");
        if (saveVO.getId() == null) {
            // 新增
            Assert.isNull(existsId, "数据已存在，请刷新后再试");
            accountDO = new SysThirdPartyAccountDO();
            accountDO.setAccountType(accountType.getValue());

            boolean exists = repoProc.existsAccount(account);
            Assert.isTrue(!exists, "应用已存在");
        } else {
            // 修改
            if (existsId != null && existsId.longValue() != saveVO.getId()) {
                // 删除别的
                repoProc.delete(existsId);
            }
            accountDO = repoProc.get(saveVO.getId());

            if (!account.equals(accountDO.getAccount())) {
                // 发送变更
                boolean exists = repoProc.existsAccount(account);
                Assert.isTrue(!exists, "应用已存在");
            }
        }

        accountDO.setBusinessType(businessType);
        accountDO.setSysTenantId(tenantId);
        accountDO.setName(saveVO.getName());
        accountDO.setAccount(account);
        accountDO.setAccountType(accountType.getValue());
        accountDO.setEnabled(ObjectUtil.defaultIfNull(saveVO.getEnabled(), true));

        Map<String, String> config = new HashMap<>(4);
        config.put(ThirdPartyConstant.CFG_WECOM_CORPID, saveVO.getCorpid());
        config.put(ThirdPartyConstant.CFG_WECOM_AGENTID, saveVO.getAgentId());
        config.put(ThirdPartyConstant.CFG_WECOM_CORPSECRET, saveVO.getCorpsecret());
        accountDO.setConfigJson(super.obj2Json(config));

        accountDO.setRemark(saveVO.getRemark());

        return accountDO;
    }

    private <T extends BaseWecomAppRespVO> T fillWecomRespVO(T respVO, ThirdPartAccountType accountType, String businessType) {
        var tenantId = super.currentTenantId();

        var accountDO = repoProc.getOneByType(accountType, businessType, tenantId);
        if (accountDO == null) {
            return null;
        }

        respVO.setId(accountDO.getId());
        respVO.setName(accountDO.getName());

        Map<String, String> config = super.json2Obj(accountDO.getConfigJson(), new TypeReference<>() {
        });
        respVO.setCorpid(config.get(ThirdPartyConstant.CFG_WECOM_CORPID));
        respVO.setCorpsecret(config.get(ThirdPartyConstant.CFG_WECOM_CORPSECRET));
        respVO.setEnabled(accountDO.getEnabled());
        respVO.setRemark(accountDO.getRemark());

        return respVO;
    }

    private <T extends BaseWechatAppSaveVo> SysThirdPartyAccountDO convertWechatSaveVO(T saveVO, ThirdPartAccountType accountType, String businessType) {
        var tenantId = super.currentTenantId();
        if (CharSequenceUtil.isBlank(businessType)) {
            businessType = SysThirdPartyAccountBusinessType.DEFAULT.getValue();
        }
        // 判断是否已存在
        var existsId = repoProc.getIdByType(accountType, businessType, tenantId);

        SysThirdPartyAccountDO accountDO = null;
        if (saveVO.getId() == null) {
            // 新增
            Assert.isNull(existsId, "数据已存在，请刷新后再试");
            accountDO = new SysThirdPartyAccountDO();
            accountDO.setAccountType(accountType.getValue());

            boolean exists = repoProc.existsAccount(saveVO.getAppId());
            Assert.isTrue(!exists, "AppId已存在");
        } else {
            // 修改
            if (existsId != null && existsId.longValue() != saveVO.getId()) {
                // 删除别的
                repoProc.delete(existsId);
            }
            accountDO = repoProc.get(saveVO.getId());

            if (!saveVO.getAppId().equals(accountDO.getAccount())) {
                // 发送变更
                boolean exists = repoProc.existsAccount(saveVO.getAppId());
                Assert.isTrue(!exists, "AppId已存在");
            }
        }

        accountDO.setBusinessType(businessType);
        accountDO.setSysTenantId(tenantId);
        accountDO.setName(saveVO.getName());
        accountDO.setAccount(saveVO.getAppId());
        accountDO.setAccountType(accountType.getValue());
        accountDO.setEnabled(ObjectUtil.defaultIfNull(saveVO.getEnabled(), true));

        Map<String, String> config = new HashMap<>(4);
        config.put(ThirdPartyConstant.CFG_WECHAT_APP_ID, saveVO.getAppId());
        config.put(ThirdPartyConstant.CFG_WECHAT_APP_SECRET, saveVO.getAppSecret());
        accountDO.setConfigJson(super.obj2Json(config));

        accountDO.setRemark(saveVO.getRemark());

        return accountDO;
    }

    private <T extends BaseWechatAppRespVO> T fillWechatRespVO(T respVO, ThirdPartAccountType accountType, String businessType) {
        var tenantId = super.currentTenantId();

        var accountDO = repoProc.getOneByType(accountType, businessType, tenantId);
        if (accountDO == null) {
            return null;
        }

        respVO.setId(accountDO.getId());
        respVO.setName(accountDO.getName());
        respVO.setAppId(accountDO.getAccount());

        Map<String, String> config = super.json2Obj(accountDO.getConfigJson(), new TypeReference<>() {
        });
        respVO.setAppSecret(config.get(ThirdPartyConstant.CFG_WECHAT_APP_SECRET));
        respVO.setEnabled(accountDO.getEnabled());
        respVO.setRemark(accountDO.getRemark());

        return respVO;
    }
}
