package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.common.param.IdNameParam;
import com.elitescloud.boot.constant.Gender;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.spi.common.ServiceProviderLoader;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.constant.OrgType;
import com.elitescloud.cloudt.system.constant.UserSourceType;
import com.elitescloud.cloudt.system.convert.UserConvert;
import com.elitescloud.cloudt.system.convert.old.SysUserConvert;
import com.elitescloud.cloudt.system.dto.SysUserBasicDTO;
import com.elitescloud.cloudt.system.dto.SysUserType;
import com.elitescloud.cloudt.system.dto.SysUserTypeDTO;
import com.elitescloud.cloudt.system.dto.req.UserQueryDTO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserListQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.user.UserQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpOrgRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.UserTypeRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserListRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.user.UserQueryRespVO;
import com.elitescloud.cloudt.system.service.EmployeeQueryService;
import com.elitescloud.cloudt.system.service.SysUserTerminalService;
import com.elitescloud.cloudt.system.service.UserQueryService;
import com.elitescloud.cloudt.system.service.manager.OuQueryManager;
import com.elitescloud.cloudt.system.service.manager.PermissionQueryManager;
import com.elitescloud.cloudt.system.service.model.entity.SysTenantUserDO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserDO;
import com.elitescloud.cloudt.system.spi.SysUserLoginSpi;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/9/28
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT, supportOperation = true)
@TenantOrgTransaction(useTenantOrg = false)
@Log4j2
public class UserQueryServiceImpl extends UserBaseServiceImpl implements UserQueryService {
    private static final UserConvert CONVERT = UserConvert.INSTANCE;

    @Autowired
    private PermissionQueryManager permissionQueryManager;
    @Autowired
    private OuQueryManager ouQueryManager;
    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private EmployeeQueryService employeeQueryService;
    @Autowired
    private SysUserTerminalService userTerminalService;

    private AtomicBoolean userLoginSpiLoaded = new AtomicBoolean(false);
    private List<SysUserLoginSpi> userLoginSpiList = new ArrayList<>();

    @Override
    public ApiResult<Long> getIdByUsername(String username) {
        if (StrUtil.isBlank(username)) {
            return ApiResult.fail("账号为空");
        }

        Long id = userRepoProc.getIdByUsername(username);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<Boolean> existsUsername(String username) {
        if (StrUtil.isBlank(username)) {
            return ApiResult.fail("账号为空");
        }

        boolean exists = userRepoProc.existsUsername(username);
        return ApiResult.ok(exists);
    }

    @Override
    public ApiResult<Boolean> existsMobile(String mobile) {
        if (StrUtil.isBlank(mobile)) {
            return ApiResult.fail("手机号为空");
        }

        boolean exists = userRepoProc.existsMobile(mobile);
        return ApiResult.ok(exists);
    }

    @Override
    public ApiResult<Boolean> existsEmail(String email) {
        if (StrUtil.isBlank(email)) {
            return ApiResult.fail("邮箱为空");
        }

        boolean exists = userRepoProc.existsEmail(email);
        return ApiResult.ok(exists);
    }

    @Override
    public ApiResult<Boolean> existsOuterKey(String outerKey) {
        if (StrUtil.isBlank(outerKey)) {
            return ApiResult.fail("外部标识为空");
        }

        boolean exists = userRepoProc.existsOuterKey(outerKey);
        return ApiResult.ok(exists);
    }

    @Override
    public ApiResult<SysUserBasicDTO> getBasicById(Long id) {
        if (id == null) {
            return ApiResult.fail("用户ID为空");
        }

        var userDO = userRepoProc.get(id);
        if (userDO == null) {
            return ApiResult.ok();
        }
        var basic = CONVERT.do2Basic(userDO);
        super.fillTenantUser(basic);

        return ApiResult.ok(basic);
    }

    @Override
    public ApiResult<SysUserBasicDTO> getBasicByUsername(String username) {
        if (CharSequenceUtil.isBlank(username)) {
            return ApiResult.fail("登录号为空");
        }

        var userDO = userRepoProc.getByUsername(username);
        if (userDO == null) {
            return ApiResult.ok();
        }
        var basic = CONVERT.do2Basic(userDO);
        super.fillTenantUser(basic);

        return ApiResult.ok(basic);
    }

    @Override
    public ApiResult<List<SysUserBasicDTO>> getBasicById(List<Long> ids) {
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.ok(Collections.emptyList());
        }

        var userList = userRepoProc.get(ids).stream()
                .map(CONVERT::do2Basic)
                .collect(Collectors.toList());
        super.fillTenantUser(userList);
        return ApiResult.ok(userList);
    }

    @Override
    public ApiResult<List<SysUserBasicDTO>> queryUser(UserQueryDTO queryParam) {
        var userList = userRepoProc.queryBasicDto(queryParam);
        super.fillTenantUser(userList);
        return ApiResult.ok(userList);
    }

    @Override
    public ApiResult<PagingVO<UserQueryRespVO>> pageQuery(UserQueryVO queryVO) {
        GeneralUserDetails currentUser = super.currentUser(true);
        if (queryVO.getRoleId() != null) {
            // 根据角色查询用户
            var roleUserIds = tenantDataIsolateProvider.byTenantAuto(() -> permissionQueryManager.queryUserIdOfRole(queryVO.getRoleId()));
            if (roleUserIds.isEmpty()) {
                return ApiResult.ok(PagingVO.empty());
            }
            queryVO.setIdSet(new LinkedHashSet<>(roleUserIds));
        }

        var tenantId = super.currentTenantId();
        // 先查询基本信息
        UserPageQueryVO pageQueryVO = convertPageQueryVO(queryVO);
        var userPage = userRepoProc.pageQuery(currentUser, pageQueryVO);
        if (userPage.isEmpty()) {
            return ApiResult.ok(PagingVO.empty());
        }
        var userIds = userPage.getRecords().stream().map(SysUserDO::getId).collect(Collectors.toSet());
        // 设置特殊字段
        var userTypeMap = convertUserType(currentUser, userIds);
        Map<Long, List<IdCodeNameParam>> userOrgMap = Boolean.TRUE.equals(queryVO.getWithOrg()) ?
                tenantDataIsolateProvider.byTenantAuto(() -> employeeQueryManager.getEmployeeOrgsByUserIds(userIds)) : Collections.emptyMap();
        var userVoPage = userPage.map(t -> {
            var vo = CONVERT.doToQueryRespVo(t);
            vo.setFullName(t.getFullName());
            vo.setGenderName(new Gender(t.getGender()).getDescription());
            vo.setSourceName(new UserSourceType(t.getSourceType()).getDescription());
            // 账号类型
            var userTypes = userTypeMap.get(t.getId());
            if (CollUtil.isNotEmpty(userTypes)) {
                vo.setUserTypes(userTypes.stream().map(UserTypeRespVO::getType).collect(Collectors.toList()));
                vo.setUserTypeNames(userTypes.stream().map(UserTypeRespVO::getTypeName).collect(Collectors.toList()));
            }
            // 所属组织
            var userOrgs = userOrgMap.get(t.getId());
            vo.setOrgList(userOrgs == null ? Collections.emptyList() : userOrgs.stream().map(o -> {
                var org = new EmpOrgRespVO();
                org.setDetails(userOrgs);

                return org;
            }).collect(Collectors.toList()));
            return vo;
        });

        if (!ObjectUtil.defaultIfNull(queryVO.getAllTenant(), false) && enabledTenant()) {
            // 启用租户时返回租户相关信息
            var tenantUserMap = tenantUserRepoProc.listByUserIds(tenantId, userIds).stream().collect(Collectors.toMap(SysTenantUserDO::getSysUserId, t -> t, (t1, t2) -> t1));
            userVoPage.each(t -> {
                var tenantUser = tenantUserMap.get(t.getId());
                if (tenantUser == null) {
                    return;
                }
                t.setLastLoginTime(ObjectUtil.defaultIfNull(tenantUser.getLastLoginTime(), t.getLastLoginTime()));
                t.setExpiredTime(ObjectUtil.defaultIfNull(tenantUser.getExpiredTime(), t.getExpiredTime()));
                t.setEnabled(Boolean.TRUE.equals(t.getEnabled()) && Boolean.TRUE.equals(tenantUser.getEnabled()));
                t.setCreateTime(tenantUser.getBindTime());
            });
        }

        return ApiResult.ok(userVoPage);
    }

    @Override
    public ApiResult<List<UserListRespVO>> listUser(UserListQueryVO queryVO) {
        var userList = userRepoProc.queryUser(queryVO);
        return ApiResult.ok(userList);
    }

    @Override
    public ApiResult<List<IdNameParam>> queryUserName(Set<Long> ids) {
        if (CollUtil.isEmpty(ids)) {
            return ApiResult.fail("ID为空");
        }

        var userNames = userRepoProc.queryUserName(ids);
        return ApiResult.ok(userNames);
    }

    @Override
    public ApiResult<List<SysUserTypeDTO>> getUserTypes(Long id) {
        if (id == null) {
            return ApiResult.fail("账号ID为空");
        }

        var tenantId = super.currentTenantId();
        var userTypeList = userTypeRepoProc.getUserTypeList(id, tenantId)
                .stream()
                .map(t -> {
                    SysUserTypeDTO dto = new SysUserTypeDTO();
                    dto.setUserType(t.getType());
                    dto.setIdentityId(t.getIdentityId());

                    return dto;
                }).collect(Collectors.toList());

        return ApiResult.ok(userTypeList);
    }

    @Override
    public SysUserDTO getUserByUsername(String username) {
        return queryUserDtoOne(() -> {
            var user = userRepoProc.getByUsername(username);
            return user == null ? Collections.emptyList() : List.of(user);
        }, users -> {
            throw new BusinessException("存在重复的用户名");
        });
    }

    @Override
    public SysUserDTO getUserByMobile(String mobile) {
        return queryUserDtoOne(() -> userRepoProc.queryByMobile(mobile), users -> {
            throw new BusinessException("存在重复的手机号");
        });
    }

    @Override
    public SysUserDTO getUserByEmail(String email) {
        return queryUserDtoOne(() -> userRepoProc.queryByEmail(email), users -> {
            throw new BusinessException("存在重复的邮箱");
        });
    }

    @Override
    public SysUserDTO getUserById(Long id) {
        return queryUserDtoOne(() -> {
            var user = userRepoProc.get(id);
            return user == null ? Collections.emptyList() : List.of(user);
        }, users -> {
            throw new BusinessException("存在重复的用户ID");
        });
    }

    @Override
    public SysUserDTO getUserByAccount(String account) {
        return queryUserDtoOne(() -> userRepoProc.queryByAccount(account), users -> {
            throw new BusinessException("存在重复的账号");
        });
    }

    @Override
    public SysUserDTO getUserByWechatOpenid(String wechatOpenid) {
        return getUserByOpenId(null, wechatOpenid);
    }

    @Override
    public SysUserDTO getUserByOpenId(String appId, String openId) {
        return queryUserDtoOne(() -> userRepoProc.queryByOpenId(appId, openId), users -> {
            throw new BusinessException("存在重复的账号");
        });
    }

    private SysUserDTO queryUserDtoOne(Supplier<List<SysUserDO>> usersSupplier, Function<List<SysUserDO>, SysUserDO> duplicateStrategy) {
        // 获取用户列表
        var users = usersSupplier.get().stream()
                .filter(p -> p.getDeleteFlag() == null || p.getDeleteFlag() == 0)
                .collect(Collectors.toList());
        if (users.isEmpty()) {
            return null;
        }

        // 如果只查出来一个用户
        if (users.size() == 1) {
            return wrapUserDetail(users.get(0));
        }

        // 如果查出来多个用户
        var userDto = duplicateStrategy.apply(users);
        return wrapUserDetail(userDto);
    }

    private SysUserDTO wrapUserDetail(SysUserDO sysUserDO) {
        if (sysUserDO == null) {
            return null;
        }
        var user = SysUserConvert.INSTANCE.doToDto(sysUserDO);

        // SPI扩展信息填充
        expendUserInfoBySpi(user);

        // 账号类型
        var tenantId = ObjectUtil.defaultIfNull(user.getTenantId(), TenantConstant.DEFAULT_TENANT_ID);
        var userTypeList = userTypeRepoProc.getUserTypeBO(user.getId(), tenantId).stream().map(t -> new SysUserType(t.getUserType(), t.getIdentityId())).collect(Collectors.toList());
        user.setUserTypeList(userTypeList);

        // 设置其他一些隔离的数据
        tenantDataIsolateProvider.byTenantUser(() -> {
            wrapIsolatedData(user);
            return null;
        }, user);

        log.info("登录用户：{}", user);
        return user;
    }

    /**
     * 设置租户相互隔离的数据
     *
     * @param user 用户
     */
    private void wrapIsolatedData(SysUserDTO user) {
        // 获取组织
        fillUserOrg(user);

        // 用户公司
        if (user.getOrg() != null) {
            if (Boolean.TRUE.equals(systemProperties.getUseCompanyOrgAsOu())) {
                // 查询组织的所属公司
                var company = orgRepoProc.queryParentNameForType(List.of(user.getOrg().getId()), OrgType.COMPANY.getValue(), true).get(user.getOrg().getId());
                if (company != null) {
                    user.setOuId(company.getId());
                    user.setOuCode(company.getCode());
                    user.setOuName(company.getName());
                }
            } else if (user.getOrg().getOuId() != null) {
                var ou = ouQueryManager.getOuBasicDTO(user.getOrg().getOuId());
                if (ou != null && Boolean.TRUE.equals(ou.getEnabled())) {
                    user.setOuId(ou.getId());
                    user.setOuCode(ou.getOuCode());
                    user.setOuName(ou.getOuName());
                }
            }
        }

        // 员工信息
        var employee = employeeRepoProc.getEmployeeNameByUserId(user.getId(), true);
        if (employee != null) {
            user.setEmployeeId(employee.getId());
            user.setEmployeeCode(employee.getCode());
        }

        // 用户角色
        var roleCodes = permissionQueryManager.queryRoleByUser(user);
        user.setRoleCodes(roleCodes);

        // 下属
        if (employee != null) {
            var underlingList = employeeQueryService.getUnderlingId(employee.getId()).computeData();
            user.setUnderlingList(underlingList);
        }

        // 获取用户终端
        var userTerminal = userTerminalService.getByUser(user.getId());
        if (CollUtil.isNotEmpty(userTerminal.getData())) {
            user.setTerminals(userTerminal.getData());
        }
    }

    private void fillUserOrg(SysUserDTO user) {
        var userOrg = employeeOrgManager.queryOrgByUser(user.getId());
        user.setOrgList(userOrg.getOrgList());
        user.setOrg(userOrg.getOrg());
        user.setTenantOrg(userOrg.getTenantOrg());
        user.setTenantOrgAdminId(userOrg.getTenantOrgAdminId());

        if (user.getOrg() != null) {
            var position = employeeOrgManager.getPositionOfUser(user.getId(), user.getOrg().getId());
            user.setPosition(position);
        }
    }

    private void expendUserInfoBySpi(SysUserDTO userDetails) {
        if (this.userLoginSpiLoaded.compareAndSet(false, true)) {
            userLoginSpiList = ServiceProviderLoader.loadProviderInstances(SysUserLoginSpi.class);
        }

        if (userLoginSpiList.isEmpty()) {
            log.info("暂无登录扩展实现");
            return;
        }

        for (SysUserLoginSpi spi : userLoginSpiList) {
            log.info("【{}】SPI调用...", spi.getServiceName());
            spi.expendLoginUserInfo(userDetails);
        }
    }

    private UserPageQueryVO convertPageQueryVO(UserQueryVO queryVO) {
        UserPageQueryVO pageQueryVO = new UserPageQueryVO();
        pageQueryVO.setAllTenant(queryVO.getAllTenant());
        pageQueryVO.setUsername(queryVO.getUsername());
        pageQueryVO.setFullName(queryVO.getFullName());
        pageQueryVO.setMobile(queryVO.getMobile());
        pageQueryVO.setEmail(queryVO.getEmail());
        pageQueryVO.setIncludeSelf(queryVO.getIncludeSelf());
        pageQueryVO.setUserType(queryVO.getUserType());
        pageQueryVO.setUserTypeList(queryVO.getUserTypeList());
        pageQueryVO.setOrders(queryVO.getOrders());
        pageQueryVO.setKeyword(queryVO.getKeyword());
        pageQueryVO.setEnabled(queryVO.getEnabled());
        pageQueryVO.setCurrent(queryVO.getCurrent() + 1);
        pageQueryVO.setSize(queryVO.getSize());
        pageQueryVO.setIdSet(queryVO.getIdSet());

        return pageQueryVO;
    }
}
