package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.common.BusinessObjectsConstant;
import com.elitescloud.cloudt.system.model.bo.BusinessOperationParamBO;
import com.elitescloud.cloudt.system.service.repo.BusinessOperationParamRepoProc;
import com.google.common.base.Functions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/20
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class BusinessObjectManager {

    /**
     * 转换参数
     *
     * @param paramBoList 参数列表
     * @param predicate   参数过滤条件
     * @return 参数列表
     */
    public List<BusinessOperationParamBO> convertParam(@NotEmpty List<BusinessOperationParamBO> paramBoList, Predicate<BusinessOperationParamBO> predicate) {
        return convertParam(paramBoList, predicate, Function.identity(), BusinessOperationParamBO::setChildrenParams);
    }

    /**
     * 转换参数
     *
     * @param paramBoList      参数列表
     * @param predicate        参数过滤条件
     * @param paramConvert     参数转换
     * @param childrenConsumer 子参数设置
     * @param <T>              参数类型
     * @return 参数列表
     */
    public <T> List<T> convertParam(@NotEmpty List<BusinessOperationParamBO> paramBoList, Predicate<BusinessOperationParamBO> predicate,
                                    @NotNull Function<BusinessOperationParamBO, T> paramConvert, @NotNull BiConsumer<T, List<T>> childrenConsumer) {
        if (CollUtil.isEmpty(paramBoList)) {
            return Collections.emptyList();
        }

        var paramWrappers = paramBoList.stream().filter(t -> {
                    if (CharSequenceUtil.isBlank(t.getFieldName())) {
                        return false;
                    }

                    return predicate == null || predicate.test(t);
                }).map(BusinessOperationParamWrapper::new)
                .collect(Collectors.groupingBy(BusinessOperationParamWrapper::getParentPath));
        if (paramWrappers.isEmpty()) {
            return Collections.emptyList();
        }

        var paramsRootList = paramWrappers.get(BusinessObjectsConstant.PARAM_ROOT_PATH);
        return this.convertParamWithChildren(paramsRootList, paramWrappers, paramConvert, childrenConsumer);
    }

    /**
     * 尝试获取其中的分页数据
     *
     * @param paramList 数据列表
     * @return 分页数据
     */
    public List<BusinessOperationParamBO> attemptObtainPagingData(@NotEmpty List<BusinessOperationParamBO> paramList) {
        if (CollUtil.isEmpty(paramList)) {
            return Collections.emptyList();
        }

        var paramNames = paramList.stream().collect(Collectors.toMap(BusinessOperationParamBO::getFieldName, Functions.identity(), (t1, t2) -> t1));
        if (paramNames.containsKey("total") && paramNames.containsKey("records")) {
            return ObjUtil.defaultIfNull(paramNames.get("records").getChildrenParams(), Collections.emptyList());
        }

        return Collections.emptyList();
    }

    private <T> List<T> convertParamWithChildren(List<BusinessOperationParamWrapper> paramWrappers,
                                                 Map<String, List<BusinessOperationParamWrapper>> childrenMap,
                                                 Function<BusinessOperationParamBO, T> paramConvert,
                                                 BiConsumer<T, List<T>> childrenConsumer) {
        if (CollUtil.isEmpty(paramWrappers)) {
            return Collections.emptyList();
        }

        return paramWrappers.stream().map(t -> {
            var param = paramConvert.apply(t.paramBO);
            childrenConsumer.accept(param, this.convertParamWithChildren(childrenMap.get(t.paramBO.getParamPath()), childrenMap, paramConvert, childrenConsumer));

            return param;
        }).collect(Collectors.toList());
    }

    static class BusinessOperationParamWrapper {
        private final BusinessOperationParamBO paramBO;

        private final String parentPath;

        public BusinessOperationParamWrapper(BusinessOperationParamBO paramBO) {
            this.paramBO = paramBO;
            this.parentPath = this.obtainParentPath(paramBO.getParamPath());
        }

        private String obtainParentPath(String paramPath) {
            if (CharSequenceUtil.isBlank(paramPath)) {
                return BusinessObjectsConstant.PARAM_ROOT_PATH;
            }

            int index = paramPath.lastIndexOf(":");
            if (index < 0) {
                return BusinessObjectsConstant.PARAM_ROOT_PATH;
            }

            String parent = paramPath.substring(0, index);
            while (parent.endsWith(":")) {
                parent = parent.substring(0, parent.length() - 1);
            }
            return parent;
        }

        public String getParentPath() {
            return parentPath;
        }
    }
}
