package com.elitescloud.cloudt.system.service.manager;

import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.dto.SysOrgBasicDTO;
import com.elitescloud.cloudt.system.model.bo.UserOrgBO;
import com.elitescloud.cloudt.system.service.repo.EmployeeOrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.EmployeeRepoProc;
import com.elitescloud.cloudt.system.service.repo.OrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.TenantOrgRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.validation.constraints.NotEmpty;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 员工与组织.
 *
 * @author Kaiser（wang shao）
 * 2022/10/16
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class EmployeeOrgManager {

    @Autowired
    private EmployeeOrgRepoProc employeeOrgRepoProc;
    @Autowired
    private EmployeeRepoProc employeeRepoProc;
    @Autowired
    private OrgRepoProc orgRepoProc;
    @Autowired
    private TenantOrgRepoProc tenantOrgRepoProc;


    /**
     * 获取用户的组织列表
     * <p>
     * 包含上级
     *
     * @param userId 用户ID
     * @return 组织及上级
     */
    public List<List<SysOrgBasicDTO>> queryOrgWithParentsByUser(Long userId) {
        // 查询用户的组织ID列表
        var orgIds = employeeOrgRepoProc.getOrgIdByUserId(userId);
        if (orgIds.isEmpty()) {
            return Collections.emptyList();
        }

        List<List<SysOrgBasicDTO>> orgList = new ArrayList<>(64);
        for (Long orgId : orgIds) {
            // 获取组织及上级组织
            var orgs = orgRepoProc.getBasicDtoOfParents(orgId);
            boolean existsDisabled = orgs.stream().anyMatch(t -> !t.getEnabled());
            if (existsDisabled) {
                // 存在禁用的
                continue;
            }
            orgList.add(orgs);
        }

        // 查询组织信息
        return orgList;
    }

    /**
     * 获取组织的租户组织
     *
     * @param orgId 组织ID
     * @return 租户组织ID 与 管理员ID
     */
    public Map<SysOrgBasicDTO, Long> getTenantOrg(Long orgId) {
        // 获取组织及上级组织
        var orgs = orgRepoProc.getBasicDtoOfParents(orgId);
        if (orgs.isEmpty()) {
            return Collections.emptyMap();
        }
        boolean existsDisabled = orgs.stream().anyMatch(t -> !t.getEnabled());
        if (existsDisabled) {
            // 存在禁用的
            return Collections.emptyMap();
        }

        // 查询租户组织
        var orgIds = orgs.stream().map(SysOrgBasicDTO::getId).collect(Collectors.toSet());
        var tenantAdminIdMap = queryTenantOrgAdmin(orgIds);
        SysOrgBasicDTO orgTemp = null;
        Long adminId = null;
        for (int i = orgs.size() - 1; i >= 0; i--) {
            orgTemp = orgs.get(i);
            adminId = tenantAdminIdMap.get(orgTemp.getId());
            if (tenantAdminIdMap.containsKey(orgTemp.getId())) {
                Map<SysOrgBasicDTO, Long> result = new HashMap<>(4);
                result.put(orgTemp, adminId);
                return result;
            }
        }
        return Collections.emptyMap();
    }

    /**
     * 获取用户的组织信息
     * <p>
     * 第一个组织将作为当前组织，常用以登录
     *
     * @param userId 用户ID
     * @return 组织列表
     */
    public UserOrgBO queryOrgByUser(Long userId) {
        UserOrgBO userOrgBO = new UserOrgBO();
        userOrgBO.setUserId(userId);

        // 判断是否启用
        var enabled = employeeRepoProc.getEnabledByUserId(userId);
        if (!Boolean.TRUE.equals(enabled)) {
            // 未启用
            return userOrgBO;
        }

        // 查询用户的组织
        var orgWithParents = this.queryOrgWithParentsByUser(userId);
        if (orgWithParents.isEmpty()) {
            userOrgBO.setOrgList(Collections.emptyList());
            return userOrgBO;
        }
        var defOrgId = employeeOrgRepoProc.getDefOrgByUserId(userId);

        var orgList = orgWithParents.stream().map(t -> t.get(t.size() - 1)).collect(Collectors.toList());
        userOrgBO.setOrgList(orgList);
        // 默认第一个组织
        if (defOrgId == null) {
            userOrgBO.setOrg(orgList.get(0));
        } else {
            var defOrg = orgList.stream().filter(t -> defOrgId.longValue() == t.getId()).findFirst().orElse(orgList.get(0));
            userOrgBO.setOrg(defOrg);
        }

        // 获取租户组织
        var firstOrgs = orgWithParents.get(0);
        var firstOrgIds = firstOrgs.stream().map(SysOrgBasicDTO::getId).collect(Collectors.toSet());
        var tenantAdminIdMap = this.queryTenantOrgAdmin(firstOrgIds);
        if (!tenantAdminIdMap.isEmpty()) {
            Long adminId = null;
            SysOrgBasicDTO orgTemp = null;
            for (int i = firstOrgs.size() - 1; i >= 0; i--) {
                orgTemp = firstOrgs.get(i);
                adminId = tenantAdminIdMap.get(orgTemp.getId());
                if (tenantAdminIdMap.containsKey(orgTemp.getId())) {
                    userOrgBO.setTenantOrg(orgTemp);
                    userOrgBO.setTenantOrgAdminId(adminId);
                    break;
                }
            }
        }
        return userOrgBO;
    }

    /**
     * 获取用户的组织对应的岗位
     *
     * @param userId
     * @param orgId
     * @return
     */
    public IdCodeNameParam getPositionOfUser(Long userId, Long orgId) {
        return employeeOrgRepoProc.getPositionByUserIdAndOrg(userId, orgId);
    }

    /**
     * 获取租户组织的管理员
     *
     * @param orgIds 组织ID
     * @return 组织ID与管理员ID
     */
    public Map<Long, Long> queryTenantOrgAdmin(Set<Long> orgIds) {
        return tenantOrgRepoProc.queryAdmin(orgIds);
    }

    /**
     * 获取下属组织
     *
     * @param currentUser 当前用户
     * @return 组织D
     */
    public Set<Long> queryChildrenOrgId(GeneralUserDetails currentUser) {
        if (currentUser == null) {
            currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        }
        if (currentUser.isSystemAdmin()) {
            // 系统管理员
            return Collections.emptySet();
        }

        if (currentUser.isTenantAdmin()) {
            // 租户管理员
            return orgRepoProc.queryAllOrgIds(currentUser.getTenantId(), true);
        }

        if (currentUser.isTenantOrgAdmin()) {
            return orgRepoProc.queryOrgIdsByBelong(currentUser.getTenantOrgId(), true);
        }

        return currentUser.getOrgId() == null ? Collections.emptySet() : orgRepoProc.queryOrgIdsByBelong(currentUser.getOrgId(), true);
    }

    /**
     * 用户是否有操作对应组织的权限
     *
     * @param currentUser 当前用户
     * @param orgIds       待判断的组织ID
     * @return
     */
    public boolean hasPermissionOfMng(GeneralUserDetails currentUser, @NotEmpty Collection<Long> orgIds) {
        if (CollectionUtils.isEmpty(orgIds)) {
            return true;
        }
        if (currentUser == null) {
            currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();
        }
        if (currentUser.isSystemAdmin()) {
            // 系统管理员有任何权限
            return true;
        }

        var permissionOrgIds = queryChildrenOrgId(currentUser);
        if (permissionOrgIds.isEmpty()) {
            return false;
        }
        for (Long orgId : orgIds) {
            if (!permissionOrgIds.contains(orgId)) {
                return false;
            }
        }
        return true;
    }
}
