package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.extra.pinyin.PinyinUtil;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.OuConvert;
import com.elitescloud.cloudt.system.service.callback.OuChangedCallback;
import com.elitescloud.cloudt.system.service.model.bo.SysOuSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysOuDO;
import com.elitescloud.cloudt.system.service.repo.OuRepo;
import com.elitescloud.cloudt.system.service.repo.OuRepoProc;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotNull;

/**
 * 公司管理.
 *
 * @author Kaiser（wang shao）
 * @date 2023/2/2
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class OuMngManager {
    private static final OuConvert CONVERT = OuConvert.INSTANCE;

    @Autowired
    private OuRepo repo;
    @Autowired
    private OuRepoProc repoProc;
    @Autowired
    private ObjectProvider<OuChangedCallback> ouChangedCallbacks;

    /**
     * 保存公司
     *
     * @param saveBO 公司参数
     * @return 公司ID
     */
    public Long upsert(@NotNull SysOuSaveBO saveBO) {
        // 保存参数转换与校验
        SysOuDO ouDO;
        try {
            ouDO = this.convertForUpsert(saveBO);
        } catch (IllegalArgumentException e) {
            throw new BusinessException("保存失败，" + e.getMessage());
        }

        // 保存数据
        repo.save(ouDO);

        // 保存后的回调
        ouChangedCallbacks.forEach(t -> t.onUpsert(saveBO.getId() == null, saveBO, ouDO));

        return ouDO.getId();
    }

    /**
     * 修改启用状态
     *
     * @param ouId    公司ID
     * @param enabled 启用状态
     * @return 公司ID
     */
    public Long updateEnabled(long ouId, boolean enabled) {
        // 修改启用状态
        repoProc.updateEnabled(ouId, enabled);
        // 回调
        ouChangedCallbacks.forEach(t -> t.onEnabled(ouId, enabled));

        return ouId;
    }

    /**
     * 删除组织数据
     *
     * @param ouId 公司ID
     */
    public void delete(long ouId) {
        repoProc.delete(ouId);

        // 删除后的回调
        ouChangedCallbacks.forEach(t -> t.onDelete(ouId));
    }

    private SysOuDO convertForUpsert(SysOuSaveBO saveBO) {
        var isAdd = saveBO.getId() == null;
        SysOuDO ouDO = null;
        if (isAdd) {
            // 新增时
            var exists = repoProc.existsOuCode(saveBO.getOuCode(), null);
            Assert.isTrue(!exists, "公司编码已存在");
        } else {
            // 更新时
            ouDO = repoProc.get(saveBO.getId());
            Assert.notNull(ouDO, "修改的数据不存在");
            // 公司编码
            if (!ouDO.getOuCode().equals(saveBO.getOuCode())) {
                var exists = repoProc.existsOuCode(saveBO.getOuCode(), null);
                Assert.isTrue(!exists, "公司编码已存在");
            }
        }
        // 默认值
        if (saveBO.getSortNo() == null) {
            saveBO.setSortNo(0);
        }
        if (saveBO.getEnabled() == null) {
            saveBO.setEnabled(true);
        }
        // 电话号码
        if (StringUtils.hasText(saveBO.getPhoneNum())) {
            Assert.hasText(saveBO.getPhonePrefix(), "电话号码的区号为空");
        } else {
            saveBO.setPhonePrefix(null);
        }

        // 转为do
        ouDO = isAdd ? CONVERT.saveBo2DO(saveBO) : CONVERT.copySaveBo2DO(saveBO, ouDO);
        ouDO.setPinyin(PinyinUtil.getPinyin(ouDO.getOuName()));
        if (StringUtils.hasText(saveBO.getPhoneNum())) {
            ouDO.setPhone(saveBO.getPhonePrefix() + "-" + saveBO.getPhoneNum());
        }
        return ouDO;
    }

}
