package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformCurrencyDO;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformCurrencyDO;
import com.elitescloud.cloudt.platform.model.vo.extend.query.CurrencyPageQueryVO;
import com.elitescloud.cloudt.system.dto.req.SysCurrencyQueryDTO;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/1/29
 */
@Repository
public class CurrencyRepoProc extends BaseRepoProc<SysPlatformCurrencyDO> {
    private static final QSysPlatformCurrencyDO QDO = QSysPlatformCurrencyDO.sysPlatformCurrencyDO;

    public CurrencyRepoProc() {
        super(QDO);
    }

    /**
     * 分页查询管理
     *
     * @param queryVO 查询参数
     * @return 货币列表
     */
    public PagingVO<SysPlatformCurrencyDO> pageMng(CurrencyPageQueryVO queryVO, Collection<String> currCodes) {
        var predicate = PredicateBuilder.builder()
                .andLike(StringUtils.hasText(queryVO.getCurrCode()), QDO.currCode, queryVO.getCurrCode())
                .andLike(StringUtils.hasText(queryVO.getCurrName()), QDO.currName, queryVO.getCurrName())
                // 租户侧查询到的都是启用的
                .andEq(true, QDO.enabled, true)
                .andIn(CollUtil.isNotEmpty(currCodes), QDO.currCode, currCodes)
                .build();
        return super.queryByPage(predicate, queryVO.getPageRequest(), QDO.createTime.desc());
    }

    /**
     * 根据货币码获取名称
     *
     * @param currCodes 货币码
     * @return 货币码与货币名称映射
     */
    public Map<String, String> getNameByCurrCode(@NotEmpty Collection<String> currCodes) {
        return super.jpaQueryFactory.select(QDO.currName, QDO.currCode)
                .from(QDO)
                .where(QDO.currCode.in(currCodes))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.currCode), t -> t.get(QDO.currName), (t1, t2) -> t1));
    }

    /**
     * 根据货币编码获取货币
     *
     * @param code 货币编码
     * @return 货币
     */
    public SysPlatformCurrencyDO getByCode(@NotBlank String code) {
        return super.getOneByValue(QDO.currCode, code);
    }

    /**
     * 根据货币编码获取货币
     *
     * @param codes 货币编码
     * @return 货币
     */
    public List<SysPlatformCurrencyDO> getByCodes(@NotEmpty Collection<String> codes) {
        return super.getListByValue(QDO.currCode, codes);
    }

    /**
     * 查询货币列表
     *
     * @param queryDTO 查询参数
     * @return 货币列表
     */
    public List<SysPlatformCurrencyDO> queryList(SysCurrencyQueryDTO queryDTO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.currCode, queryDTO.getCurrCode())
                .andIn(QDO.currCode, queryDTO.getCurrCodes())
                .andEq(QDO.enabled, queryDTO.getEnabled())
                .andLike(QDO.currName, queryDTO.getCurrName())
                .build();
        return super.getList(predicate).stream()
                .sorted(Comparator.comparing(SysPlatformCurrencyDO::getCreateTime, Comparator.nullsLast(LocalDateTime::compareTo)))
                .collect(Collectors.toList());
    }

}
