package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.core.support.customfield.service.impl.CustomFieldJpaServiceUtil;
import com.elitescloud.boot.datasecurity.common.DataSecurityUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.constant.SysConstant;
import com.elitescloud.cloudt.system.dto.SysEmployeeBasicDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeePageQueryDTO;
import com.elitescloud.cloudt.system.dto.req.EmployeeQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.EmployeePageRespDTO;
import com.elitescloud.cloudt.system.model.entity.QSysPositionDO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonEmployeePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.org.EmployeeListQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.org.EmployeePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeeListRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeePagedRespVO;
import com.elitescloud.cloudt.system.modules.orgtree.model.entity.QOrgBuTreeEmployeeDO;
import com.elitescloud.cloudt.system.service.model.entity.*;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.*;
import com.querydsl.core.types.dsl.*;
import com.querydsl.jpa.JPAExpressions;
import com.querydsl.jpa.impl.JPAUpdateClause;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.util.*;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * 2022/10/8
 */
@Repository
public class EmployeeRepoProc extends BaseRepoProc<SysEmployeeDO> {
    private static final QSysEmployeeDO QDO = QSysEmployeeDO.sysEmployeeDO;
    private static final QSysEmployeeOrgDO QDO_EMPLOYEE_ORG = QSysEmployeeOrgDO.sysEmployeeOrgDO;
    private static final QSysOrgDO QDO_ORG = QSysOrgDO.sysOrgDO;
    private static final QSysUserRoleDO QDO_USER_ROLE = QSysUserRoleDO.sysUserRoleDO;
    private static final QSysRoleDO QDO_ROLE = QSysRoleDO.sysRoleDO;
    private static final QOrgBuTreeEmployeeDO QDO_TREE_EMP = QOrgBuTreeEmployeeDO.orgBuTreeEmployeeDO;
    private static final QSysPositionDO QDO_POSITION = QSysPositionDO.sysPositionDO;
    private static final QSysEmpGroupDO QDO_EMP_GROUP = QSysEmpGroupDO.sysEmpGroupDO;
    private static final QSysEmpGroupEmpDO QDO_EMP_GROUP_EMP = QSysEmpGroupEmpDO.sysEmpGroupEmpDO;

    @Autowired
    private OrgRepoProc orgRepoProc;

    public EmployeeRepoProc() {
        super(QDO);
    }

    public void updateJsonField(EntityPathBase<?> entity, MapPath<String, String, StringPath> jsonField, Map<String, String> updates,
                                Predicate predicate) {
        // 开始一个更新操作
        JPAUpdateClause update = jpaQueryFactory.update(entity);

        // 构建 JSON_SET 表达式
        String jsonSetExpression = "extension_info";  // 假设你的实体字段名为 extension_info
        for (Map.Entry<String, String> entry : updates.entrySet()) {
            jsonSetExpression = String.format("JSON_SET(%s, '$.%s', '%s')", jsonSetExpression, entry.getKey(), entry.getValue());
        }

        // 使用 Expressions.stringTemplate 进行更新
        update.set(Expressions.stringPath("extension_info"), Expressions.stringTemplate("JSON_SET(extension_info, '$.test5', '33333')"))
                .where(predicate)
                .execute();
    }

//    @Transactional
//    public void updateJsonField(EntityManager entityManager, Class<?> entityClass, Long employeeId, Map<String, String> updates) {
//        String tableName = getTableName(entityClass); // 获取动态表名
//        String jsonSetExpression = "extension_info";  // 假设数据库中的 JSON 字段名
//
//        for (Map.Entry<String, String> entry : updates.entrySet()) {
//            jsonSetExpression = String.format("JSON_SET(%s, '$.%s', '%s')", jsonSetExpression, entry.getKey(), entry.getValue());
//        }
//
//        // 构建完整的 SQL 更新语句
//        String sql = String.format("UPDATE %s SET extension_info = %s WHERE id = %d", tableName, jsonSetExpression, employeeId);
//
//        // 执行更新
//        entityManager.createNativeQuery(sql).executeUpdate();
//    }


    /**
     * 更新启用状态
     *
     * @param id
     * @param enabled
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateEnabled(long id, Boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    /**
     * 根据账号ID更新启用状态
     *
     * @param userId
     * @param enabled
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateEnabledByUserId(long userId, Boolean enabled) {
        super.updateValueByValue(QDO.enabled, enabled, QDO.userId, userId);
    }

    /**
     * 更新手机号
     *
     * @param id     员工ID
     * @param mobile 手机号
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateMobile(long id, String mobile) {
        super.updateValue(QDO.phone, mobile, id);
    }

    /**
     * 更新email
     *
     * @param id    员工ID
     * @param email 邮箱
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateEmail(long id, String email) {
        super.updateValue(QDO.email, email, id);
    }

    /**
     * 更新账号信息
     *
     * @param user
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateUserInfo(SysUserDO user) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.username, user.getUsername())
                .set(QDO.lastName, user.getLastName())
                .set(QDO.firstName, user.getFirstName())
                .set(QDO.gender, user.getGender())
                .set(QDO.nickName, user.getNickName())
                .set(QDO.birthDate, user.getBirthDate())
                .set(QDO.mobile, user.getMobile())
                .set(QDO.email, user.getEmail())
                .set(QDO.idCard, user.getIdCard())
                .set(QDO.countryCode, user.getCountryCode())
                .set(QDO.provinceCode, user.getProvinceCode())
                .set(QDO.cityCode, user.getCityCode())
                .set(QDO.countyCode, user.getCountyCode())
                .set(QDO.address, user.getAddress())
                .set(QDO.personalSign, user.getPersonalSign())
                .set(QDO.avatarUrl, user.getAvatarUrl())
                .set(QDO.avatarCode, user.getAvatarCode())
                .where(QDO.userId.eq(user.getId()))
                .execute();
    }

    /**
     * 判断编号是否存在
     *
     * @param code
     * @return
     */
    public boolean existsCode(String code) {
        return super.exists(QDO.code, code);
    }

    /**
     * 判断邮箱是否存在
     *
     * @param rootOrgId
     * @param email
     * @return
     */
    public boolean existsEmail(Long rootOrgId, String email) {
        return jpaQueryFactory.select(QDO.id)
                .from(QDO)
                .where(QDO.email.eq(email).and(QDO.rootOrgId.eq(rootOrgId)))
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 获取启用状态
     *
     * @param id
     * @return
     */
    public Boolean getEnabled(long id) {
        return super.getValue(QDO.enabled, id);
    }

    /**
     * 根据用户ID获取启用状态
     *
     * @param userId 用户ID
     * @return 启用状态
     */
    public Boolean getEnabledByUserId(long userId) {
        return super.getValueByValue(QDO.enabled, QDO.userId, userId);
    }

    /**
     * 判断是否需要同步
     *
     * @param id
     * @return
     */
    public boolean needSyncOuter(long id) {
        return jpaQueryFactory.select(QDO_ORG.id)
                .from(QDO_ORG)
                .where(QDO_ORG.id.in(
                        JPAExpressions.select(QDO_EMPLOYEE_ORG.orgId).from(QDO_EMPLOYEE_ORG).where(QDO_EMPLOYEE_ORG.employeeId.eq(id))
                ).and(QDO_ORG.syncOuter.eq(true)))
                .fetchOne() != null;
    }

    /**
     * 获取员工的用户ID
     *
     * @param id
     * @return
     */
    public Long getUserId(Long id) {
        var userId = super.getValue(QDO.userId, id);
        if (userId == null || userId == SysConstant.NO_USER_ID) {
            return null;
        }
        return userId;
    }

    public String getUsername(long id) {
        return super.getValue(QDO.username, id);
    }

    /**
     * 获取员工的用户ID
     *
     * @param id
     * @return
     */
    public Long getRealUserId(Long id) {
        return super.getValue(QDO.userId, id);
    }

    /**
     * 获取员工的用户ID
     * <p>
     * 忽略没有账户的
     *
     * @param ids
     * @return
     */
    public Map<Long, Long> getUserIdsIgnoreNoUser(Collection<Long> ids) {
        return super.jpaQueryFactory.select(QDO.id, QDO.userId)
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch()
                .stream()
                .filter(t -> {
                    var userId = t.get(QDO.userId);
                    return userId != null && userId != SysConstant.NO_USER_ID;
                })
                .collect(Collectors.toMap(t -> t.get(QDO.id), t -> t.get(QDO.userId), (t1, t2) -> t1));
    }

    /**
     * 获取员工的用户ID
     * <p>
     * 忽略没有账户的
     *
     * @param ids
     * @return
     */
    public Map<Long, Long> getUserIds(Collection<Long> ids) {
        return super.jpaQueryFactory.select(QDO.id, QDO.userId)
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.id), t -> t.get(QDO.userId), (t1, t2) -> t1));
    }

    /**
     * 获取员工的根组织ID
     *
     * @param id
     * @return
     */
    public Long getRootOrgId(Long id) {
        return super.getValue(QDO.rootOrgId, id);
    }

    /**
     * 根据用户ID获取员工ID
     *
     * @param userId 用户ID
     * @return 员工ID
     */
    public Long getIdByUserId(long userId) {
        return super.getIdByValue(QDO.userId, userId);
    }

    /**
     * 根据用户ID获取员工ID
     *
     * @param userId 用户ID
     * @return 员工ID
     */
    public Long getIdByUserId(long userId, boolean enabled) {
        return super.getValue(QDO.id, QDO.userId.eq(userId).and(QDO.enabled.eq(enabled)));
    }

    /**
     * 根据账号ID获取邮箱
     *
     * @param userId
     * @return
     */
    public String getEmailByUserId(long userId) {
        return super.getValueByValue(QDO.email, QDO.userId, userId);
    }

    /**
     * 根据用户ID获取员工编号
     *
     * @param userId 用户ID
     * @return 员工编号
     */
    public String getCodeByUserId(long userId) {
        return super.getValueByValue(QDO.code, QDO.userId, userId);
    }

    public String getCode(long id) {
        return super.getValue(QDO.code, id);
    }

    /**
     * 根据员工编号获取员工ID
     *
     * @param code 员工编号
     * @return 员工ID
     */
    public Long getIdByCode(@NotBlank String code) {
        return super.getIdByValue(QDO.code, code);
    }

    /**
     * 根据用户ID获取员工
     *
     * @param userId
     * @return
     */
    public Optional<SysEmployeeDO> getByUserId(Long userId) {
        return super.getOneOptionalByValue(QDO.userId, userId);
    }

    /**
     * 根据员工编号获取
     *
     * @param code
     * @return 员工
     */
    public Optional<SysEmployeeDO> getByCode(@NotBlank String code) {
        return super.getOneOptionalByValue(QDO.code, code);
    }

    /**
     * 根据员工编号获取
     *
     * @param codes
     * @return 员工
     */
    public List<SysEmployeeDO> getByCode(@NotEmpty Collection<String> codes) {
        return super.getListByValue(QDO.code, codes);
    }

    /**
     * 分页查询员工信息
     *
     * @param queryVO
     * @param userIds
     * @return
     */
    public PagingVO<EmployeePagedRespVO> pageMng(EmployeePageQueryVO queryVO, Set<Long> userIds) {
        var predicateAuth = DataSecurityUtil.predicateForJPA(SysEmployeeDO.class);
        var customFieldPredicate = CustomFieldJpaServiceUtil.getPredicate(queryVO.getConditions(), SysEmployeeDO.class);
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.userId, userIds)
                .andEq(QDO.username, queryVO.getUsername())
                .andLike(QDO.code, queryVO.getCode())
                .andLike(QDO.lastName, queryVO.getFullName())
                .andEq(QDO.gender, queryVO.getGender())
                .andLike(QDO.phone, queryVO.getMobile())
                .andLike(QDO.email, queryVO.getEmail())
                .andEq(QDO.type, queryVO.getType())
                .and(queryVO.getHasAccount() != null, () -> queryVO.getHasAccount() ? QDO.userId.ne(SysConstant.NO_USER_ID) : QDO.userId.eq(SysConstant.NO_USER_ID))
                .andOnNotNull(queryVO.getOrgId(), () -> QDO.id.in(
                        JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                                .from(QDO_EMPLOYEE_ORG)
                                .where(QDO_EMPLOYEE_ORG.orgId.eq(queryVO.getOrgId()))
                ))
                .and(queryVO.getOrgId() == null && queryVO.getOrgIdBelong() != null, () -> QDO.id.in(
                        JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                                .from(QDO_EMPLOYEE_ORG)
                                .leftJoin(QDO_ORG).on(QDO_ORG.id.eq(QDO_EMPLOYEE_ORG.orgId))
                                .where(orgRepoProc.predicateOfOrgChildren(queryVO.getOrgIdBelong()))
                ))
                .andEq(QDO.enabled, queryVO.getEnabled())
                .and(predicateAuth)
                .and(customFieldPredicate)
                .build();

        var jpaQuery = jpaQueryFactory.select(qBeanEmployeePagedRespVO())
                .from(QDO)
                .where(predicate);
        return queryByPage(jpaQuery, queryVO.getPageRequest());
    }

    public List<EmployeePagedRespVO> listMngOfPage(Collection<Long> ids) {
        var predicate = PredicateBuilder.builder()
                .andIn(QDO.id, ids)
                .build();
        return super.getList(qBeanEmployeePagedRespVO(), predicate);
    }

    /**
     * 分页查询
     *
     * @param queryVO
     * @param userIds
     * @return
     */
    public PagingVO<EmployeePagedRespVO> pageQuery(CommonEmployeePageQueryVO queryVO, Set<Long> userIds, Set<Long> userIdsOfDisabled) {
        Long orgId = queryVO.getOrgId(), orgIdBelong = queryVO.getOrgIdBelong();
        var currentUser = SecurityContextUtil.currentUserIfUnauthorizedThrow();

        if (Boolean.TRUE.equals(queryVO.getCurrentTenantOrg())) {
            // 当前租户组织
            orgIdBelong = ObjectUtil.defaultIfNull(currentUser.getTenantOrgId(), currentUser.getOrgId());
            if (orgIdBelong == null) {
                return PagingVO.empty();
            }
        } else if (Boolean.TRUE.equals(queryVO.getCurrentOrg())) {
            // 当前组织
            orgId = currentUser.getOrgId();
            if (orgId == null) {
                return PagingVO.empty();
            }
        }

        // 根据组织的子查询
        SubQueryExpression<Long> orgIdSubQuery = null;
        if (orgId != null) {
            orgIdSubQuery = JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                    .from(QDO_EMPLOYEE_ORG)
                    .where(QDO_EMPLOYEE_ORG.orgId.eq(orgId));
        }
        SubQueryExpression<Long> orgIdBelongSubQuery = null;
        if (orgIdBelong != null && orgId == null) {
            orgIdBelongSubQuery = JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                    .from(QDO_EMPLOYEE_ORG)
                    .leftJoin(QDO_ORG).on(QDO_ORG.id.eq(QDO_EMPLOYEE_ORG.orgId))
                    .where(orgRepoProc.predicateOfOrgChildren(orgIdBelong));
        }
        // 根据角色查询
        SubQueryExpression<Long> userIdOfRoleSubQuery = null;
        if (StringUtils.hasText(queryVO.getRoleCode())) {
            userIdOfRoleSubQuery = JPAExpressions.select(QDO_USER_ROLE.userId).from(QDO_USER_ROLE).where(QDO_USER_ROLE.roleId.in(
                    JPAExpressions.select(QDO_ROLE.id).from(QDO_ROLE).where(QDO_ROLE.code.eq(queryVO.getRoleCode()))
            ));
        }

        var predicate = PredicateBuilder.builder()
                .andIn(QDO.userId, userIds)
                .andLike(QDO.code, queryVO.getCode())
                .andLike(QDO.lastName, queryVO.getFullName())
                .andLike(new StringExpression[]{QDO.code, QDO.lastName, QDO.firstName}, queryVO.getKeyword())
                .andEq(QDO.gender, queryVO.getGender())
                .andLike(QDO.phone, queryVO.getMobile())
                .andLike(QDO.email, queryVO.getEmail())
                .andEq(true, QDO.enabled, ObjectUtil.defaultIfNull(queryVO.getEnabled(), true))
                .andEq(true, QDO.served, ObjectUtil.defaultIfNull(queryVO.getServed(), true))
                .and(queryVO.getHasAccount() != null, () -> queryVO.getHasAccount() ? QDO.userId.ne(SysConstant.NO_USER_ID) : QDO.userId.eq(SysConstant.NO_USER_ID))
                .andEq(QDO.type, queryVO.getType())
                .andBetween(true, QDO.joinTime, queryVO.getJoinTimeStart(), queryVO.getJoinTimeEnd())
                .andIn(orgIdSubQuery != null, QDO.id, orgIdSubQuery)
                .andIn(orgIdBelongSubQuery != null, QDO.id, orgIdBelongSubQuery)
                .andIn(userIdOfRoleSubQuery != null, QDO.userId, userIdOfRoleSubQuery)
                .and(CollUtil.isNotEmpty(userIdsOfDisabled), () -> {
                    if (queryVO.getEnabled() == null) {
                        return null;
                    }
                    return queryVO.getEnabled() ? QDO.userId.notIn(userIdsOfDisabled) : QDO.userId.in(userIdsOfDisabled);
                })
                .build();

        var jpaQuery = jpaQueryFactory.select(qBeanEmployeePagedRespVO())
                .from(QDO)
                .where(predicate);
        return queryByPage(jpaQuery, queryVO.getPageRequest(), QDO.createTime.asc());
    }

    /**
     * 分页查询
     *
     * @param queryDTO 查询参数
     * @param userIds  用户ID
     * @return 员工列表
     */
    public PagingVO<EmployeePageRespDTO> pageQuery(EmployeePageQueryDTO queryDTO, Set<Long> userIds, Set<Long> userIdsOfDisabled) {
        String keyword = StringUtils.hasText(queryDTO.getKeyword()) ? "%" + queryDTO.getKeyword() + "%" : null;

        // 子查询
        SubQueryExpression<Long> orgIdSubQuery = null;
        SubQueryExpression<Long> orgIdBelongSubQuery = null;
        if (queryDTO.getOrgId() != null) {
            orgIdSubQuery = JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                    .from(QDO_EMPLOYEE_ORG)
                    .where(QDO_EMPLOYEE_ORG.orgId.eq(queryDTO.getOrgId()));
        } else if (StringUtils.hasText(queryDTO.getOrgCode())) {
            orgIdSubQuery = JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                    .from(QDO_EMPLOYEE_ORG)
                    .where(QDO_EMPLOYEE_ORG.orgId.eq(
                                    JPAExpressions.select(QDO_ORG.id).from(QDO_ORG).where(QDO_ORG.code.eq(queryDTO.getOrgCode()))
                            )
                    );
        } else if (queryDTO.getOrgIdBelong() != null) {
            orgIdBelongSubQuery = JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                    .from(QDO_EMPLOYEE_ORG)
                    .leftJoin(QDO_ORG).on(QDO_ORG.id.eq(QDO_EMPLOYEE_ORG.orgId))
                    .where(orgRepoProc.predicateOfOrgChildren(queryDTO.getOrgIdBelong()));
        } else if (StrUtil.isNotBlank(queryDTO.getOrgCodeBelong())) {
            var orgIdBelong = getIdByCode(queryDTO.getOrgCodeBelong());
            if (orgIdBelong == null) {
                return PagingVO.empty();
            }
            orgIdBelongSubQuery = JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                    .from(QDO_EMPLOYEE_ORG)
                    .leftJoin(QDO_ORG).on(QDO_ORG.id.eq(QDO_EMPLOYEE_ORG.orgId))
                    .where(orgRepoProc.predicateOfOrgChildren(orgIdBelong));
        }

        var predicate = PredicateBuilder.builder()
                .andIn(QDO.id, queryDTO.getIds())
                .andIn(QDO.code, queryDTO.getCodes())
                .andIn(QDO.userId, CollUtil.defaultIfEmpty(userIds, queryDTO.getUserIds()))
                .andEq(QDO.type, queryDTO.getType())
                .andIn(QDO.type, queryDTO.getTypes())
                .andEq(QDO.enabled, queryDTO.getEnabled())
                .andLike(QDO.username, queryDTO.getUsername())
                .andLike(QDO.username, queryDTO.getUsernames())
                .andLike(QDO.code, queryDTO.getCode())
                .andLike(QDO.phone, queryDTO.getPhone())
                .andLike(QDO.mobile, queryDTO.getMobile())
                .andLike(QDO.emailWork, queryDTO.getEmailWork())
                .andLike(QDO.email, queryDTO.getEmail())
                .andIn(orgIdSubQuery != null, QDO.id, orgIdSubQuery)
                .andIn(orgIdBelongSubQuery != null, QDO.id, orgIdBelongSubQuery)
                .and(keyword != null, () -> QDO.code.like(keyword).or(QDO.lastName.like(keyword)))
                .andLike(new StringExpression[]{QDO.mobile, QDO.username}, queryDTO.getUsernameOrMobile())
                .andLike(QDO.firstName, queryDTO.getFirstName())
                .andLike(QDO.lastName, queryDTO.getLastName())
                .andEq(QDO.served, queryDTO.getServed())
                // 没有上级的
                .and(Boolean.TRUE.equals(queryDTO.getWithoutLeader()), () ->
                        JPAExpressions.select(QDO_EMPLOYEE_ORG.id).from(QDO_EMPLOYEE_ORG).where(QDO_EMPLOYEE_ORG.employeeId.eq(QDO.id).and(QDO_EMPLOYEE_ORG.leaderEmployeeId.isNotNull())).notExists())
                // 有上级的
                .and(Boolean.FALSE.equals(queryDTO.getWithoutLeader()), () ->
                        JPAExpressions.select(QDO_EMPLOYEE_ORG.id).from(QDO_EMPLOYEE_ORG).where(QDO_EMPLOYEE_ORG.employeeId.eq(QDO.id).and(QDO_EMPLOYEE_ORG.leaderEmployeeId.isNotNull())).exists())
                .and(Boolean.TRUE.equals(queryDTO.getWithoutMobile()), () -> QDO.mobile.isNull().or(QDO.mobile.eq("")))
                .and(Boolean.FALSE.equals(queryDTO.getWithoutMobile()), () -> QDO.mobile.isNotNull().and(QDO.mobile.ne("")))
                .and(CollUtil.isNotEmpty(userIdsOfDisabled), () -> {
                    if (queryDTO.getEnabled() == null) {
                        return null;
                    }
                    return queryDTO.getEnabled() ? QDO.userId.notIn(userIdsOfDisabled) : QDO.userId.in(userIdsOfDisabled);
                })
                .build();

        var jpaQuery = jpaQueryFactory.select(qBeanEmployeePageRespDTO())
                .from(QDO)
                .where(predicate);
        var pageData = queryByPage(jpaQuery, queryDTO.getPageRequest(), QDO.createTime.desc());
        if (pageData.isEmpty()) {
            return pageData;
        }

        // 下属数量
        if (Boolean.TRUE.equals(queryDTO.getWithUnderlingNum())) {
            var employeeIds = pageData.stream().map(EmployeePageRespDTO::getId).collect(Collectors.toList());
            var subPredicate = QDO_EMPLOYEE_ORG.leaderEmployeeId.in(employeeIds);
            if (queryDTO.getEnabled() != null) {
                subPredicate = subPredicate.and(
                        JPAExpressions.select(QDO.id).from(QDO).where(QDO.id.eq(QDO_EMPLOYEE_ORG.employeeId).and(QDO.enabled.eq(queryDTO.getEnabled()))).exists()
                );
            }

            var employeeSubMap = jpaQueryFactory.select(QDO_EMPLOYEE_ORG.employeeId, QDO_EMPLOYEE_ORG.leaderEmployeeId)
                    .from(QDO_EMPLOYEE_ORG)
                    .where(subPredicate)
                    .fetch()
                    .stream()
                    .collect(Collectors.groupingBy(t -> t.get(QDO_EMPLOYEE_ORG.leaderEmployeeId), Collectors.counting()));
            pageData.each(t -> t.setUnderlingNum(employeeSubMap.getOrDefault(t.getId(), 0L).intValue()));
        }

        return pageData;
    }

    /**
     * 查询员工
     *
     * @param orgId           直属组织ID
     * @param orgIdBelong     下属组织ID
     * @param rootOrgId       根组织ID
     * @param includeDisabled 是否包含禁用的
     * @return 员工列表
     */
    public List<SysEmployeeBasicDTO> queryEmployee(Long orgId, Long orgIdBelong, Long rootOrgId,
                                                   boolean includeDisabled) {
        Predicate predicate = null;
        if (orgId != null) {
            predicate = QDO_EMPLOYEE_ORG.orgId.eq(orgId);
        } else if (orgIdBelong != null) {
            predicate = QDO.id.in(
                    JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                            .from(QDO_EMPLOYEE_ORG)
                            .leftJoin(QDO_ORG).on(QDO_ORG.id.eq(QDO_EMPLOYEE_ORG.orgId))
                            .where(orgRepoProc.predicateOfOrgChildren(orgIdBelong))
            );
        } else if (rootOrgId != null) {
            predicate = QDO_EMPLOYEE_ORG.rootOrgId.eq(rootOrgId);
        }

        if (!includeDisabled) {
            predicate = super.andPredicate(predicate, QDO.enabled.eq(true));
        }

        return jpaQueryFactory.select(qBeanEmployeeBasicDTO())
                .from(QDO)
                .leftJoin(QDO_EMPLOYEE_ORG).on(QDO_EMPLOYEE_ORG.employeeId.eq(QDO.id))
                .where(predicate)
                .fetch();
    }

    /**
     * 根据用户ID查询员工基本信息
     *
     * @param sysUserIds 用户ID
     * @return 基本信息列表
     */
    public List<SysEmployeeBasicDTO> queryBasicByUserIds(Collection<Long> sysUserIds) {
        return super.jpaQueryFactory.select(this.qBeanEmployeeBasicDTO())
                .from(QDO)
                .leftJoin(QDO_EMPLOYEE_ORG).on(QDO_EMPLOYEE_ORG.employeeId.eq(QDO.id))
                .where(QDO.userId.in(sysUserIds))
                .fetch();
    }

    /**
     * 查询员工列表
     *
     * @param queryVO
     * @return
     */
    public List<EmployeeListRespVO> listQuery(EmployeeListQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andLike(QDO.code, queryVO.getCodeLike())
                .andLike(new StringExpression[]{QDO.lastName, QDO.firstName}, queryVO.getFullNameLike())
                .andLike(QDO.username, queryVO.getUsernameLike())
                .andLike(new StringExpression[]{QDO.lastName, QDO.firstName, QDO.username}, queryVO.getFullNameOrUsernameLike())
                .andLike(new StringExpression[]{QDO.code, QDO.lastName, QDO.firstName, QDO.username}, queryVO.getCodeOrFullNameOrUsernameLike())
                .build();
//        if (predicate == null) {
//            return Collections.emptyList();
//        }
        predicate = ExpressionUtils.and(predicate, QDO.enabled.eq(true).and(QDO.served.eq(true)));

        return super.jpaQueryFactory.select(qBeanEmployeeList())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 统计组织下员工的数量
     *
     * @param orgIds 组织ID
     * @return
     */
    public Map<Long, Integer> countEmployee(Collection<Long> orgIds, boolean includeDisabled) {
        Predicate predicate = QDO_EMPLOYEE_ORG.orgId.in(orgIds);
        if (!includeDisabled) {
            predicate = super.andPredicate(predicate, QDO.enabled.eq(true));
        }

        return jpaQueryFactory.select(QDO_EMPLOYEE_ORG.employeeId, QDO_EMPLOYEE_ORG.orgId)
                .from(QDO)
                .leftJoin(QDO_EMPLOYEE_ORG).on(QDO_EMPLOYEE_ORG.employeeId.eq(QDO.id))
                .where(predicate)
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> t.get(QDO_EMPLOYEE_ORG.orgId), Collectors.collectingAndThen(
                        Collectors.toList(), List::size
                )));
    }

    /**
     * 查询员工列表
     *
     * @param queryDTO
     * @return
     */
    public List<SysEmployeeDO> queryList(EmployeeQueryDTO queryDTO) {
        // 根据组织查询
        SubQueryExpression<Long> orgIdSubQuery = null, orgIdBelongSubQuery = null, postIdSubQuery = null, empGroupSubQuery = null;
        if (queryDTO.getOrgId() != null || CharSequenceUtil.isNotBlank(queryDTO.getOrgCode()) || CollUtil.isNotEmpty(queryDTO.getOrgIds()) || CollUtil.isNotEmpty(queryDTO.getOrgCodes())) {
            SubQueryExpression<Long> tempSubQuery1 = StringUtils.hasText(queryDTO.getOrgCode()) ? JPAExpressions.select(QDO_ORG.id).from(QDO_ORG).where(QDO_ORG.code.eq(queryDTO.getOrgCode())) : null;
            SubQueryExpression<Long> tempSubQuery2 = CollUtil.isNotEmpty(queryDTO.getOrgCodes()) ? JPAExpressions.select(QDO_ORG.id).from(QDO_ORG).where(QDO_ORG.code.in(queryDTO.getOrgCodes())) : null;
            orgIdSubQuery = JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                    .from(QDO_EMPLOYEE_ORG)
                    .where(PredicateBuilder.builder()
                            .andEq(QDO_EMPLOYEE_ORG.orgId, queryDTO.getOrgId())
                            .andIn(QDO_EMPLOYEE_ORG.orgId, queryDTO.getOrgIds())
                            .andIn(tempSubQuery1 != null, QDO_EMPLOYEE_ORG.orgId, tempSubQuery1)
                            .andIn(tempSubQuery2 != null, QDO_EMPLOYEE_ORG.orgId, tempSubQuery2)
                            .build());
        }
        if (orgIdSubQuery == null &&
                (queryDTO.getOrgIdBelong() != null || CharSequenceUtil.isNotBlank(queryDTO.getOrgCodeBelong()))
        ) {
            Long orgIdBelong = queryDTO.getOrgIdBelong() != null ? queryDTO.getOrgIdBelong() : jpaQueryFactory.select(QDO_ORG.id).from(QDO_ORG).where(QDO_ORG.code.eq(queryDTO.getOrgCodeBelong())).limit(1).fetchOne();
            if (orgIdBelong == null) {
                return Collections.emptyList();
            }
            orgIdBelongSubQuery = JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                    .from(QDO_EMPLOYEE_ORG)
                    .leftJoin(QDO_ORG).on(QDO_ORG.id.eq(QDO_EMPLOYEE_ORG.orgId))
                    .where(orgRepoProc.predicateOfOrgChildren(orgIdBelong));
        }
        // 岗位
        if (queryDTO.getPositionId() != null || CharSequenceUtil.isNotBlank(queryDTO.getPositionCode()) || CollUtil.isNotEmpty(queryDTO.getPositionIds()) || CollUtil.isNotEmpty(queryDTO.getPositionCodes())) {
            SubQueryExpression<Long> tempSubQuery1 = StringUtils.hasText(queryDTO.getPositionCode()) ? JPAExpressions.select(QDO_POSITION.id).from(QDO_POSITION).where(QDO_POSITION.positionCode.eq(queryDTO.getPositionCode())) : null;
            SubQueryExpression<Long> tempSubQuery2 = CollUtil.isNotEmpty(queryDTO.getPositionCodes()) ? JPAExpressions.select(QDO_POSITION.id).from(QDO_POSITION).where(QDO_POSITION.positionCode.in(queryDTO.getPositionCodes())) : null;
            postIdSubQuery = JPAExpressions.select(QDO_EMPLOYEE_ORG.employeeId)
                    .from(QDO_EMPLOYEE_ORG)
                    .where(PredicateBuilder.builder()
                            .andEq(QDO_EMPLOYEE_ORG.positionId, queryDTO.getPositionId())
                            .andIn(QDO_EMPLOYEE_ORG.positionId, queryDTO.getPositionIds())
                            .andIn(tempSubQuery1 != null, QDO_EMPLOYEE_ORG.positionId, tempSubQuery1)
                            .andIn(tempSubQuery2 != null, QDO_EMPLOYEE_ORG.positionId, tempSubQuery2)
                            .build());
        }
        // 员工组
        if (queryDTO.getEmpGroupId() != null || CharSequenceUtil.isNotBlank(queryDTO.getEmpGroupCode()) || CollUtil.isNotEmpty(queryDTO.getEmpGroupIds()) || CollUtil.isNotEmpty(queryDTO.getEmpGroupCodes())) {
            empGroupSubQuery = JPAExpressions.select(QDO_EMP_GROUP_EMP.empId)
                    .from(QDO_EMP_GROUP_EMP)
                    .leftJoin(QDO_EMP_GROUP).on(QDO_EMP_GROUP.id.eq(QDO_EMP_GROUP_EMP.groupId))
                    .where(PredicateBuilder.builder()
                            .andEq(QDO_EMP_GROUP.enabled, true)
                            .andEq(QDO_EMP_GROUP.id, queryDTO.getEmpGroupId())
                            .andIn(QDO_EMP_GROUP.id, queryDTO.getEmpGroupIds())
                            .andEq(QDO_EMP_GROUP.code, queryDTO.getEmpGroupCode())
                            .andIn(QDO_EMP_GROUP.code, queryDTO.getEmpGroupCodes())
                            .build());

        }
        var predicate = PredicateBuilder.builder()
                .andIn(CollUtil.isNotEmpty(queryDTO.getIds()), QDO.id, queryDTO.getIds())
                .andIn(CollUtil.isNotEmpty(queryDTO.getUserIds()), QDO.userId, queryDTO.getUserIds())
                .andIn(CollUtil.isNotEmpty(queryDTO.getCodes()), QDO.code, queryDTO.getCodes())
                .andIn(CollUtil.isNotEmpty(queryDTO.getPhones()), QDO.phone, queryDTO.getPhones())
                .andIn(CollUtil.isNotEmpty(queryDTO.getEmails()), QDO.email, queryDTO.getEmails())
                .andIn(orgIdSubQuery != null, QDO.id, orgIdSubQuery)
                .andIn(orgIdBelongSubQuery != null, QDO.id, orgIdBelongSubQuery)
                .andIn(postIdSubQuery != null, QDO.id, postIdSubQuery)
                .andIn(empGroupSubQuery != null, QDO.id, empGroupSubQuery)
                .andEq(StringUtils.hasText(queryDTO.getType()), QDO.type, queryDTO.getType())
                .andEq(StringUtils.hasText(queryDTO.getDuty()), QDO.duty, queryDTO.getDuty())
                .andEq(queryDTO.getServed() != null, QDO.served, queryDTO.getServed())
                .andEq(queryDTO.getEnabled() != null, QDO.enabled, queryDTO.getEnabled())
                .andLike(QDO.firstName, queryDTO.getFirstName())
                .andLike(QDO.lastName, queryDTO.getLastName())
                .andLike(new StringExpression[]{QDO.username, QDO.lastName}, queryDTO.getUsernameOrName())
                .and(StringUtils.hasText(queryDTO.getCodeOrName()), () -> QDO.code.eq(queryDTO.getCodeOrName()).or(
                        QDO.lastName.like("%" + queryDTO.getCodeOrName() + "%")
                ))
                .build();
        var jpqQuery = jpaQueryFactory.select(QDO)
                .from(QDO)
                .where(predicate);
        if (queryDTO.getLimitSize() != null) {
            jpqQuery.limit(queryDTO.getLimitSize());
        }
        return jpqQuery.fetch();
    }

    /**
     * 获取员工姓名
     *
     * @param id
     * @return
     */
    public IdCodeNameParam getEmployeeName(long id, Boolean enabled) {
        Predicate predicate = PredicateBuilder.builder()
                .andEq(QDO.id, id)
                .andEq(enabled != null, QDO.enabled, enabled)
                .build();
        Tuple t = jpaQueryFactory.select(QDO.id, QDO.code, QDO.firstName, QDO.lastName, QDO.username)
                .from(QDO)
                .where(predicate)
                .fetchOne();
        if (t == null) {
            return null;
        }
        return new IdCodeNameParam(t.get(QDO.id), t.get(QDO.code), this.buildUserName(t.get(QDO.firstName), t.get(QDO.lastName), t.get(QDO.username)));
    }

    /**
     * 获取员工姓名
     *
     * @param userId
     * @return
     */
    public IdCodeNameParam getEmployeeNameByUserId(long userId, Boolean enabled) {
        Predicate predicate = PredicateBuilder.builder()
                .andEq(QDO.userId, userId)
                .andEq(enabled != null, QDO.enabled, enabled)
                .build();
        Tuple t = jpaQueryFactory.select(QDO.id, QDO.code, QDO.firstName, QDO.lastName, QDO.username)
                .from(QDO)
                .where(predicate)
                .fetchOne();
        if (t == null) {
            return null;
        }
        return new IdCodeNameParam(t.get(QDO.id), t.get(QDO.code), this.buildUserName(t.get(QDO.firstName), t.get(QDO.lastName), t.get(QDO.username)));
    }

    /**
     * 查询员工姓名
     *
     * @param ids
     * @return
     */
    public List<IdCodeNameParam> queryEmployeeName(Collection<Long> ids) {
        return jpaQueryFactory.select(QDO.id, QDO.code, QDO.firstName, QDO.lastName, QDO.username)
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch()
                .stream()
                .map(t -> new IdCodeNameParam(t.get(QDO.id), t.get(QDO.code), this.buildUserName(t.get(QDO.firstName), t.get(QDO.lastName), t.get(QDO.username))))
                .collect(Collectors.toList());
    }

    /**
     * 查询员工姓名
     *
     * @param userIds
     * @return
     */
    public List<IdCodeNameParam> queryEmployeeNameByUserIds(Collection<Long> userIds) {
        return jpaQueryFactory.select(QDO.id, QDO.code, QDO.firstName, QDO.lastName, QDO.username)
                .from(QDO)
                .where(QDO.userId.in(userIds))
                .fetch()
                .stream()
                .map(t -> new IdCodeNameParam(t.get(QDO.id), t.get(QDO.code), this.buildUserName(t.get(QDO.firstName), t.get(QDO.lastName), t.get(QDO.username))))
                .collect(Collectors.toList());
    }

    /**
     * 查询员工姓名
     *
     * @param codes
     * @return
     */
    public List<IdCodeNameParam> queryEmployeeNameByCodes(Collection<String> codes) {
        return jpaQueryFactory.select(QDO.id, QDO.code, QDO.firstName, QDO.lastName, QDO.username)
                .from(QDO)
                .where(QDO.code.in(codes))
                .fetch()
                .stream()
                .map(t -> new IdCodeNameParam(t.get(QDO.id), t.get(QDO.code), this.buildUserName(t.get(QDO.firstName), t.get(QDO.lastName), t.get(QDO.username))))
                .collect(Collectors.toList());
    }

    /**
     * 根据角色获取员工
     *
     * @param roleId
     * @return
     */
    public List<EmployeeListRespVO> listEmployeeByRole(long roleId) {
        return jpaQueryFactory.select(qBeanEmployeeList())
                .from(QDO)
                .leftJoin(QDO_USER_ROLE).on(QDO_USER_ROLE.userId.eq(QDO.userId).and(QDO.id.isNotNull()))
                .where(QDO_USER_ROLE.roleId.eq(roleId).and(QDO.id.isNotNull()))
                .orderBy(QDO_USER_ROLE.createTime.desc())
                .fetch();
    }

    private String buildUserName(String firstName, String lastName, String username) {
        if (CharSequenceUtil.isBlank(lastName)) {
            return CharSequenceUtil.blankToDefault(firstName, username);
        }

        return lastName + CharSequenceUtil.blankToDefault(firstName, "");
    }

    private QBean<EmployeePagedRespVO> qBeanEmployeePagedRespVO() {
        return Projections.bean(EmployeePagedRespVO.class, QDO.id, QDO.userId, QDO.code, QDO.code, QDO.duty, QDO.type,
                QDO.phone, QDO.email, QDO.enabled, QDO.served, QDO.joinTime,
                QDO.username, QDO.firstName, QDO.lastName, QDO.gender, QDO.mobile, QDO.phone, QDO.email, QDO.photo,
                QDO.extensionInfo
        );
    }

    private QBean<SysEmployeeBasicDTO> qBeanEmployeeBasicDTO() {
        return Projections.bean(SysEmployeeBasicDTO.class, QDO.id, QDO.userId, QDO.rootOrgId, QDO.lastName, QDO.firstName,
                QDO.gender, QDO.code, QDO.email, QDO.phone, QDO.type, QDO.served, QDO.photo, QDO.enabled, QDO.sortNo,
                QDO_EMPLOYEE_ORG.orgId);
    }

    private QBean<EmployeePageRespDTO> qBeanEmployeePageRespDTO() {
        return Projections.bean(EmployeePageRespDTO.class, QDO.id, QDO.code, QDO.userId, QDO.username, QDO.firstName, QDO.lastName,
                QDO.gender, QDO.type, QDO.duty, QDO.served, QDO.enabled, QDO.phone, QDO.email);
    }

    private QBean<EmployeeListRespVO> qBeanEmployeeList() {
        return Projections.bean(EmployeeListRespVO.class, QDO.id, QDO.userId, QDO.code,
                QDO.duty, QDO.username, QDO.lastName, QDO.firstName, QDO.emailWork, QDO.phone, QDO.mobile, QDO.email);
    }
}
