package com.elitescloud.cloudt.system.service.repo;

import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysPlatformUdcDO;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformUdcDO;
import com.elitescloud.cloudt.system.model.bo.SysUdcBO;
import com.elitescloud.cloudt.system.model.vo.query.udc.UdcPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.udc.UdcWithValuesPageRespVO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.core.types.dsl.StringExpression;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * UDC.
 *
 * @author Kaiser（wang shao）
 * 2022/10/11
 */
@Repository
public class UdcRepoProc extends BaseRepoProc<SysPlatformUdcDO> {
    private static final QSysPlatformUdcDO QDO = QSysPlatformUdcDO.sysPlatformUdcDO;

    public UdcRepoProc() {
        super(QDO);
    }

    /**
     * 删除UDC
     *
     * @param appCode
     * @param udcCode
     */
    public void delete(@NotBlank String appCode, @NotBlank String udcCode) {
        jpaQueryFactory.delete(QDO)
                .where(QDO.udcCode.eq(udcCode).and(QDO.appCode.eq(appCode)))
                .execute();
    }

    /**
     * 分页查询
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysPlatformUdcDO> pageMng(UdcPageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.appCode, queryVO.getAppCode())
                .andEq(QDO.udcCode, queryVO.getUdcCode())
                .andLike(QDO.udcName, queryVO.getUdcName())
                .build();

        var jpaQuery = jpaQueryFactory.select(QDO)
                .from(QDO)
                .where(predicate);
        return super.queryByPage(jpaQuery, queryVO.getPageRequest(), QDO.createTime.desc());
    }

    public PagingVO<UdcWithValuesPageRespVO> pageQuery(UdcPageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.appCode, queryVO.getAppCode())
                .andEq(QDO.udcCode, queryVO.getUdcCode())
                .andLike(QDO.udcName, queryVO.getUdcName())
                .andLike(new StringExpression[]{QDO.udcCode, QDO.udcName}, queryVO.getKeyword())
                .build();

        var jpaQuery = jpaQueryFactory.select(QDO.appCode, QDO.udcCode, QDO.udcName, QDO.udcDescribe)
                .from(QDO)
                .where(predicate);
        return super.queryByPage(jpaQuery, queryVO.getPageRequest(), QDO.createTime.desc(), t -> {
            UdcWithValuesPageRespVO respVO = new UdcWithValuesPageRespVO();
            respVO.setAppCode(t.get(QDO.appCode));
            respVO.setUdcCode(t.get(QDO.udcCode));
            respVO.setUdcName(t.get(QDO.udcName));
            respVO.setUdcDescribe(t.get(QDO.udcDescribe));

            return respVO;
        });
    }

    /**
     * 根据应用编码分页查询
     *
     * @param page
     * @param pageSize
     * @param appCode
     * @return
     */
    public PagingVO<SysUdcBO> pageAll(int page, int pageSize, String appCode) {
        var jpaQuery = jpaQueryFactory.select(qBeanBO())
                .from(QDO)
                .where(QDO.appCode.eq(appCode));

        return super.queryByPage(jpaQuery, super.ofPage(page, pageSize));
    }

    /**
     * 根据应用编码和UDC编码查询
     *
     * @param appCode
     * @param udcCodes
     * @return
     */
    public List<SysPlatformUdcDO> queryByUdcCode(String appCode, Set<String> udcCodes) {
        return jpaQueryFactory.select(QDO)
                .from(QDO)
                .where(QDO.udcCode.in(udcCodes).and(QDO.appCode.eq(appCode)))
                .fetch();
    }

    /**
     * 根据应用编码和UDC编码查询
     *
     * @param appCode
     * @param udcCodes
     * @return
     */
    public List<SysUdcBO> queryBoByUdcCode(String appCode, Set<String> udcCodes) {
        BooleanExpression predicate = QDO.udcCode.in(udcCodes);
        if (StringUtils.hasText(appCode)) {
            predicate = predicate.and(QDO.appCode.eq(appCode));
        }
        return jpaQueryFactory.select(qBeanBO())
                .from(QDO)
                .where(predicate)
                .fetch();
    }

    /**
     * 根据应用编码和UDC编码查询
     *
     * @param appCode
     * @param udcCode
     * @return
     */
    public SysPlatformUdcDO getByAppCodeAndUdcCode(String appCode, String udcCode) {
        var predicate = QDO.udcCode.eq(udcCode).and(QDO.appCode.eq(appCode));
        return jpaQueryFactory.select(QDO)
                .from(QDO)
                .where(predicate)
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据应用编码和UDC编码查询
     *
     * @param appCode
     * @param udcCode
     * @return
     */
    public SysUdcBO getBoByAppCodeAndUdcCode(String appCode, String udcCode) {
        var predicate = QDO.udcCode.eq(udcCode).and(QDO.appCode.eq(appCode));
        return jpaQueryFactory.select(qBeanBO())
                .from(QDO)
                .where(predicate)
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据UDC编码获取udc名称
     *
     * @param appCode
     * @param udcCodes
     * @return
     */
    public Map<String, String> getUdcNameByUdcCode(String appCode, Collection<String> udcCodes) {
        BooleanExpression predicate = QDO.udcCode.in(udcCodes);
        if (StringUtils.hasText(appCode)) {
            predicate = predicate.and(QDO.appCode.eq(appCode));
        }
        return jpaQueryFactory.select(QDO.udcCode, QDO.udcName)
                .from(QDO)
                .where(predicate)
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.udcCode), t -> t.get(QDO.udcName), (t1, t2) -> t1));
    }

    private QBean<SysUdcBO> qBeanBO() {
        return Projections.bean(SysUdcBO.class, QDO.id, QDO.appCode, QDO.udcCode, QDO.udcName, QDO.allowUpdate,
                QDO.allowAddValue, QDO.udcDescribe, QDO.parentUdcCode);
    }
}
