package com.elitescloud.cloudt.system.config.support;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.client.common.AuthorizationException;
import com.elitescloud.boot.auth.client.token.AbstractCustomAuthenticationToken;
import com.elitescloud.boot.auth.provider.common.AuthorizationConstant;
import com.elitescloud.boot.auth.provider.common.LoginParameterNames;
import com.elitescloud.boot.auth.provider.security.AuthenticationCheckService;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.common.SettingNameConstant;
import com.elitescloud.cloudt.system.service.repo.SysSettingRepoProc;
import com.elitescloud.cloudt.system.util.HttpServletUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.Getter;
import lombok.Setter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.AuthenticationException;
import org.springframework.stereotype.Component;

import java.io.Serial;
import java.io.Serializable;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2025/7/10 周四
 */
@Component
public class UserTypeClientAuthenticationCheckService<T extends AbstractCustomAuthenticationToken<T>> implements AuthenticationCheckService<T> {
    private static final Logger logger = LoggerFactory.getLogger(UserTypeClientAuthenticationCheckService.class);

    @Autowired
    private SysSettingRepoProc settingRepoProc;
    @Autowired
    private TenantDataIsolateProvider tenantDataIsolateProvider;

    @Override
    public void additionalAuthenticationChecks(GeneralUserDetails userDetails, T authentication) throws AuthenticationException {
        if (userDetails.isSystemAdmin() || userDetails.isTenantAdmin()) {
            // 系统管理员和租户管理员不受限制
            return;
        }

        var settingValue = tenantDataIsolateProvider.byTenantDirectly(() -> settingRepoProc.getSettingValBySettingNo(SettingNameConstant.USER_TYPE_LOGIN_CLIENT), userDetails.getTenant());
        if (CharSequenceUtil.isBlank(settingValue)) {
            return;
        }

        String clientId = obtainClientId();
        logger.debug("登录终端：{}", clientId);
        if (CharSequenceUtil.isBlank(clientId)) {
            return;
        }

        List<UserTypeClientCfg> cfgList = JSONUtil.json2Obj(settingValue, new TypeReference<>() {});
        if (CollUtil.isEmpty(cfgList)) {
            throw new AuthorizationException("账号类型与登录客户端的配置有误, 请联系管理员");
        }

        for (UserTypeClientCfg cfg : cfgList) {
            if (clientId.equals(cfg.getClientId())) {
                if (CollUtil.isEmpty(cfg.getUserTypes())) {
                    throw new AuthorizationException("登录失败，不支持登录该客户端");
                }
                if (cfg.getUserTypes().contains("*")) {
                    return;
                }
                if (CollUtil.isEmpty(userDetails.getUser().getUserTypeList())) {
                    throw new AuthorizationException("登录失败，不支持登录该客户端");
                }
                for (var userType : userDetails.getUser().getUserTypeList()) {
                    if (cfg.getUserTypes().contains(userType.getUserType())) {
                        return;
                    }
                }
                throw new AuthorizationException("登录失败，不支持登录该客户端");
            }
        }
        throw new AuthorizationException("登录失败，不支持登录该客户端");
    }

    private String obtainClientId() {
        var request = HttpServletUtil.currentRequest();
        if (request == null) {
            return null;
        }

        String clientId = request.getParameter(LoginParameterNames.CLIENT_ID);
        if (CharSequenceUtil.isNotBlank(clientId)) {
            return clientId;
        }

        return (String) request.getAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_CLIENT_ID);
    }

    @Getter
    @Setter
    static class UserTypeClientCfg implements Serializable {
        @Serial
        private static final long serialVersionUID = 8476309616466393193L;

        private String clientId;

        private List<String> userTypes;
    }
}
