package com.elitescloud.cloudt.system.controller.devops;

import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.model.vo.query.devops.CacheMethodDataQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.devops.CacheMethodRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.devops.CacheServiceRespVO;
import com.elitescloud.cloudt.system.service.devops.CacheMngService;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import java.util.List;
import java.util.Map;

/**
 * 基础数据缓存管理.
 *
 * @author Kaiser（wang shao）
 * @date 2024/9/12
 */
@Api(tags = "基础数据缓存管理")
@RestController
@RequestMapping(value = "/devops/cache", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class CacheMngController {

    private CacheMngService mngService;

    @ApiOperation(value = "获取服务列表")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "serviceId", value = "服务实例ID", required = true),
    })
    @GetMapping(value = "/cacheServices")
    public ApiResult<List<CacheServiceRespVO>> queryCacheServices(@RequestParam(name = "serviceId") @NotBlank(message = "服务实例ID为空") String serviceId) {
        return mngService.queryCacheServices(serviceId);
    }

    @ApiOperation(value = "获取缓存服务的方法")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "serviceId", value = "服务实例ID", required = true),
            @ApiImplicitParam(name = "cacheServiceKey", value = "缓存服务标识Key", required = true),
    })
    @GetMapping(value = "/cacheMethods")
    public ApiResult<List<CacheMethodRespVO>> queryCacheMethods(@RequestParam(name = "serviceId") @NotBlank(message = "服务实例ID为空") String serviceId,
                                                                @RequestParam(name = "cacheServiceKey") @NotBlank(message = "缓存服务标识为空") String cacheServiceKey) {
        return mngService.queryCacheMethods(serviceId, cacheServiceKey);
    }

    @ApiOperation(value = "根据缓存方法查询数据")
    @ApiOperationSupport(order = 3)
    @PostMapping(value = "/queryCacheDataByCacheMethod")
    public ApiResult<Object> queryCacheDataByCacheMethod(@RequestBody @Valid CacheMethodDataQueryVO queryVO) {
        return mngService.queryCacheDataByCacheMethod(queryVO.getServiceId(), queryVO.getCacheServiceKey(), queryVO.getMethod(), queryVO.getParams(), queryVO.getTenantCode());
    }

    @ApiOperation(value = "获取缓存的数据项")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "serviceId", value = "服务实例ID", required = true),
            @ApiImplicitParam(name = "cacheServiceKey", value = "缓存服务标识Key", required = true),
            @ApiImplicitParam(name = "tenantCode", value = "租户编码"),
    })
    @GetMapping(value = "/items")
    public ApiResult<List<String>> queryCacheItems(@RequestParam(name = "serviceId") @NotBlank(message = "服务实例ID为空") String serviceId,
                                                   @RequestParam(name = "cacheServiceKey") @NotBlank(message = "缓存服务标识为空") String cacheServiceKey,
                                                   @RequestParam(name = "tenantCode", required = false) String tenantCode) {
        return mngService.queryCacheItems(serviceId, cacheServiceKey, tenantCode);
    }

    @ApiOperation(value = "获取缓存项的数据")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "serviceId", value = "服务实例ID", required = true),
            @ApiImplicitParam(name = "cacheServiceKey", value = "缓存服务标识Key", required = true),
            @ApiImplicitParam(name = "cacheItem", value = "缓存项", required = true),
            @ApiImplicitParam(name = "tenantCode", value = "租户编码"),
    })
    @GetMapping(value = "/item/data")
    public ApiResult<Object> queryCacheItemData(@RequestParam(name = "serviceId") @NotBlank(message = "服务实例ID为空") String serviceId,
                                                @RequestParam(name = "cacheServiceKey") @NotBlank(message = "缓存服务标识为空") String cacheServiceKey,
                                                @RequestParam(name = "cacheItem") @NotBlank(message = "缓存项为空") String cacheItem,
                                                @RequestParam(name = "tenantCode", required = false) String tenantCode) {
        return mngService.queryCacheItemData(serviceId, cacheServiceKey, cacheItem, tenantCode);
    }

    @ApiOperation(value = "获取缓存服务下的所有数据")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "serviceId", value = "服务实例ID", required = true),
            @ApiImplicitParam(name = "cacheServiceKey", value = "缓存服务标识Key", required = true),
            @ApiImplicitParam(name = "tenantCode", value = "租户编码"),
    })
    @GetMapping(value = "/item/data/all")
    public ApiResult<Map<String, Object>> queryCacheDataByCacheKey(@RequestParam(name = "serviceId") @NotBlank(message = "服务实例ID为空") String serviceId,
                                                                   @RequestParam(name = "cacheServiceKey") @NotBlank(message = "缓存服务标识为空") String cacheServiceKey,
                                                                   @RequestParam(name = "tenantCode", required = false) String tenantCode) {
        return mngService.queryAllCacheItemData(serviceId, cacheServiceKey, tenantCode);
    }

    @ApiOperation(value = "重新加载缓存")
    @ApiOperationSupport(order = 21)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "serviceId", value = "服务实例ID", required = true),
            @ApiImplicitParam(name = "cacheServiceKey", value = "缓存服务标识Key", required = true),
    })
    @PutMapping(value = "/reload")
    public ApiResult<Boolean> reload(@RequestParam(name = "serviceId") @NotBlank(message = "服务实例ID为空") String serviceId,
                                     @RequestParam(name = "cacheServiceKey") @NotBlank(message = "缓存服务标识为空") String cacheServiceKey) {
        return mngService.reload(serviceId, cacheServiceKey);
    }

    @ApiOperation(value = "重新加载缓存")
    @ApiOperationSupport(order = 21)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "serviceId", value = "服务实例ID", required = true),
            @ApiImplicitParam(name = "cacheServiceKey", value = "缓存服务标识Key"),
    })
    @DeleteMapping(value = "/clear")
    public ApiResult<Boolean> clear(@RequestParam(name = "serviceId") @NotBlank(message = "服务实例ID为空") String serviceId,
                                    @RequestParam(name = "cacheServiceKey", required = false) String cacheServiceKey) {
        return mngService.clear(serviceId, cacheServiceKey);
    }

    @Autowired
    public void setMngService(CacheMngService mngService) {
        this.mngService = mngService;
    }
}
