package com.elitescloud.cloudt.system.modules.orgtree.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.constant.OrgTreeNodeType;
import com.elitescloud.cloudt.context.util.CollectionUtil;
import com.elitescloud.cloudt.context.util.TreeDataUtil;
import com.elitescloud.cloudt.system.dto.SysOrgBasicDTO;
import com.elitescloud.cloudt.system.dto.resp.OrgTreeNodeRespVO;
import com.elitescloud.cloudt.system.modules.orgtree.service.MultiOrgTreeQueryService;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.MultiOrgTreeRepoProc;
import com.elitescloud.cloudt.system.param.SysOrgQueryDTO;
import com.elitescloud.cloudt.system.service.repo.EmployeeRepoProc;
import com.elitescloud.cloudt.system.service.repo.OrgRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/5/7
 */
@Slf4j
@Service
public class MultiOrgTreeQueryServiceImpl implements MultiOrgTreeQueryService {

    @Autowired
    private MultiOrgTreeRepoProc repoProc;
    @Autowired
    private OrgRepoProc orgRepoProc;
    @Autowired
    private EmployeeRepoProc employeeRepoProc;
    @Autowired
    private MultiOrgTreeRepoProc multiOrgTreeRepoProc;

    @Override
    public ApiResult<List<IdCodeNameParam>> listOrgTree() {
        var treeList = repoProc.queryListTree();
        return ApiResult.ok(treeList);
    }

    @Override
    public ApiResult<List<OrgTreeNodeRespVO>> getTree(Long treeId, Boolean tree, Boolean employee, Boolean all) {
        Assert.notNull(treeId, "树ID为空");

        List<OrgTreeNodeRespVO> respVoList = new ArrayList<>(128);
        // 查询组织
        respVoList.addAll(this.queryOrgOfTree(treeId));
        // 查询员工
        if (Boolean.TRUE.equals(employee)) {
            var orgIdCodeMap = respVoList.stream().collect(Collectors.toMap(OrgTreeNodeRespVO::getId, OrgTreeNodeRespVO::getCode, (t1, t2) -> t1));
            respVoList.addAll(this.queryEmployeeOfTree(treeId, orgIdCodeMap));
        }

        // 转为树形结构
        if (respVoList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }
        TreeDataUtil<OrgTreeNodeRespVO> treeDataUtil = new TreeDataUtil<>(respVoList, OrgTreeNodeRespVO::getId, OrgTreeNodeRespVO::getParentId,
                OrgTreeNodeRespVO::setChildren, Comparator.comparingInt(OrgTreeNodeRespVO::getSortNo));
        List<OrgTreeNodeRespVO> result = tree == null || tree ? (List<OrgTreeNodeRespVO>) treeDataUtil.getRoots() :
                CollectionUtil.expandTree((List<OrgTreeNodeRespVO>) treeDataUtil.getRoots(), OrgTreeNodeRespVO::getChildren);
        return ApiResult.ok(result);
    }

    private List<OrgTreeNodeRespVO> queryEmployeeOfTree(long treeId, Map<Long, String> orgIdCodeMap) {
        return multiOrgTreeRepoProc.queryBoundEmployee(treeId).stream()
                .map(t -> {
                    OrgTreeNodeRespVO respVO = new OrgTreeNodeRespVO();
                    respVO.setId(t.getId());
                    respVO.setCode(t.getCode());
                    respVO.setName(t.getFullName());
                    respVO.setNodeType(OrgTreeNodeType.EMPLOYEE.getValue());
                    respVO.setNodeTypeName(OrgTreeNodeType.EMPLOYEE.getDescription());
                    respVO.setDataType(t.getType());
                    respVO.setSortNo(t.getSortNo());
                    respVO.setParentId(t.getOrgId());
                    respVO.setParentCode(orgIdCodeMap.get(t.getOrgId()));
                    respVO.setHasChildren(false);
                    respVO.setChildren(Collections.emptyList());
                    respVO.setUserId(t.getUserId());

                    return respVO;
                }).collect(Collectors.toList());
    }

    private List<OrgTreeNodeRespVO> queryOrgOfTree(long treeId) {
        // 查询树形结构
        var treeLeafList = repoProc.queryTreeLeaf(treeId);
        if (treeLeafList.isEmpty()) {
            return Collections.emptyList();
        }

        // 查询组织信息
        SysOrgQueryDTO orgQueryDTO = new SysOrgQueryDTO();
        orgQueryDTO.setEnabled(true);
        var orgMap = orgRepoProc.queryList(orgQueryDTO).stream().collect(Collectors.toMap(SysOrgBasicDTO::getId, Function.identity(), (t1, t2) -> t1));
        if (orgMap.isEmpty()) {
            return Collections.emptyList();
        }

        var treeLeafMap = treeLeafList.stream().collect(Collectors.toMap(OrgTreeNodeRespVO::getId, Function.identity(), (t1, t2) -> t1));
        return treeLeafList.stream()
                .filter(t -> orgMap.containsKey(t.getTempId()))
                .map(t -> {
                    var org = orgMap.get(t.getTempId());

                    OrgTreeNodeRespVO respVO = new OrgTreeNodeRespVO();
                    respVO.setNodeType(OrgTreeNodeType.ORG.getValue());
                    respVO.setNodeTypeName(OrgTreeNodeType.ORG.getDescription());
                    respVO.setId(org.getId());
                    respVO.setCode(org.getCode());
                    respVO.setName(org.getPrettyName());
                    respVO.setSortNo(ObjectUtil.defaultIfNull(t.getSortNo(), 0));

                    // 上级信息
                    if (t.getParentId() != null) {
                        var parent = treeLeafMap.get(t.getParentId());
                        if (parent == null) {
                            log.info("上级不存在：{}，{}-{}", treeId, t.getId(), t.getParentId());
                            return respVO;
                        }
                        var orgParent = orgMap.get(parent.getTempId());
                        if (orgParent == null) {
                            log.info("上级组织不存在：{}，{}", treeId, parent.getTempId());
                            return respVO;
                        }
                        respVO.setParentId(orgParent.getId());
                        respVO.setParentCode(orgParent.getCode());
                    }

                    return respVO;
                }).collect(Collectors.toList());
    }
}
