package com.elitescloud.cloudt.system.modules.wecom.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.system.config.SystemProperties;
import com.elitescloud.cloudt.system.modules.wecom.common.WecomUserConverter;
import com.elitescloud.cloudt.system.modules.wecom.model.login.Code2UserInfoResult;
import com.elitescloud.cloudt.system.modules.wecom.model.login.WecomLoginPropsVO;
import com.elitescloud.cloudt.system.modules.wecom.service.WecomAuthService;
import com.elitescloud.cloudt.system.modules.wecom.util.WeComTool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.AuthenticationException;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.util.UriComponentsBuilder;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/12/1
 */
@Component
public class WecomAuthServiceImpl implements WecomAuthService {
    private static final Logger logger = LoggerFactory.getLogger(WecomAuthServiceImpl.class);

    @Autowired
    private SystemProperties systemProperties;
    @Autowired
    private WecomUserConverter wecomUserConverter;
    @Autowired
    private InternalAuthenticationGranter internalAuthenticationGranter;

    @Override
    public ApiResult<WecomLoginPropsVO> getLoginProps() {
        var authProps = systemProperties.getWecomAuth();
        if (CharSequenceUtil.isBlank(authProps.getAppId()) || CharSequenceUtil.isBlank(authProps.getAgentId())) {
            return ApiResult.fail("企业微信未配置");
        }

        WecomLoginPropsVO props = new WecomLoginPropsVO();
        props.setAppId(authProps.getAppId());
        props.setAgentId(authProps.getAgentId());
        props.setLoginType(authProps.getLoginType());
        return ApiResult.ok(props);
    }

    @Override
    public ApiResult<String> getAuthorizeUrl(String redirectUri, String state) {
        Assert.notBlank(redirectUri, "重定向地址为空");

        var authProps = systemProperties.getWecomAuth();
        if (CharSequenceUtil.isBlank(authProps.getAppId()) || CharSequenceUtil.isBlank(authProps.getAgentId())) {
            return ApiResult.fail("企业微信未配置");
        }

        String uri = UriComponentsBuilder.fromHttpUrl(authProps.getAuthUrl())
                .queryParam("login_type", authProps.getLoginType())
                .queryParam("appid", authProps.getAppId())
                .queryParam("agentid", authProps.getAgentId())
                .queryParam("redirect_uri", redirectUri)
                .queryParamIfPresent("state", Optional.ofNullable(CharSequenceUtil.blankToDefault(state, null)))
                .toUriString();
        logger.info("认证地址：{}", uri);
        return ApiResult.ok(uri);
    }

    @Override
    public ApiResult<OAuthToken> code2Token(HttpServletRequest request, HttpServletResponse response, String code) {
        Assert.notBlank(code, "授权码为空");

        logger.info("授权码换取token：{}", code);
        var authProps = systemProperties.getWecomAuth();

        // 授权码换取企微用户信息
        String accessToken = getAccessTokenOfWecom(authProps.getAppId(), authProps.getAgentSecret());
        Code2UserInfoResult userInfoResult = WeComTool.code2UserInfo(accessToken, code);
        if (userInfoResult.isSuccess()) {
            logger.info("授权码换取用户信息：{}, {}", userInfoResult.getUserid(), userInfoResult.getExternal_userid());
        } else {
            logger.error("授权码换取用户信息失败：{}", JSONUtil.toJsonString(userInfoResult));
            return ApiResult.fail("企微认证失败：" + userInfoResult.getErrcode() + ", " + userInfoResult.getErrmsg());
        }

        // 转换用户信息
        InternalAuthenticationGranter.InternalAuthenticationToken authentication = null;
        try {
            authentication = wecomUserConverter.convert(userInfoResult.getUserid(), userInfoResult.getUser_ticket(), accessToken);
        } catch (Exception e) {
            String msg = e instanceof BusinessException ? ", " + e.getMessage() : "";
            throw new BusinessException("认证失败" + msg, e);
        }

        OAuthToken token = null;
        try {
            token = internalAuthenticationGranter.authenticate(request, response, authentication);
        } catch (AuthenticationException e) {
            return ApiResult.fail("认证异常，" + e.getMessage());
        }
        return ApiResult.ok(token);
    }

    /**
     * 获取token
     *
     * @return
     */
    private static String getAccessTokenOfWecom(String corpId, String corpSecret) {
        var redisUtils = SpringContextHolder.getBean(RedisUtils.class);
        String tokenKey = "wecom:accessToken:" + corpId + ":" + corpSecret;

        // 先从缓存获取
        String token = (String) redisUtils.get(tokenKey);
        if (StringUtils.hasText(token)) {
            return token;
        }

        // 生成token
        var accessToken = WeComTool.getToken(corpId, corpSecret);
        if (accessToken.isSuccess()) {
            token = accessToken.getAccess_token();
            redisUtils.set(tokenKey, token, accessToken.getExpires_in() - 60, TimeUnit.SECONDS);
            return token;
        }
        throw new BusinessException("企业微信授权失败," + ObjUtil.defaultIfNull(accessToken.getErrcode(), -1) +
                ObjUtil.defaultIfNull(accessToken.getErrmsg(), ""));
    }
}
