package com.elitescloud.cloudt.system.service.devops.init;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import lombok.extern.slf4j.Slf4j;

import java.util.*;

/**
 * API接口.
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/8
 */
@Slf4j
public class PlatformApiParameterInitProvider extends AbstractBasicDataInitProvider {

    @Override
    public String typeName() {
        return "API接口参数";
    }

    @Override
    public String tableName() {
        return "sys_platform_api_parameter";
    }

    @Override
    public List<String> fields() {
        return List.of("api_id", "api_code", "field_name", "field_type", "field_is_null", "field_explain", "in_out_type", "std_version",
               "remark", "create_time", "modify_time");
    }

    @Override
    public Map<String, String> fieldTitles() {
        Map<String, String> titles = new LinkedHashMap<>(16);
        titles.put("api_id", "API ID");
        titles.put("api_code", "API编码");
        titles.put("field_name", "字段名称");
        titles.put("field_type", "字段类型");
        titles.put("field_is_null", "是否可以为空");
        titles.put("field_explain", "字段描述");
        titles.put("in_out_type", "参数类型");
        titles.put("create_time", "创建时间");
        titles.put("modify_time", "最后修改时间");
        return titles;
    }

    @Override
    public List<OrderItem> orderItems() {
        return List.of(OrderItem.desc("create_time"));
    }

    @Override
    public String fieldAppCode() {
        return "";
    }

    @Override
    public UpdateType convertForImport(Map<String, Object> data, List<Map<String, Object>> currentData, List<Map<String, Object>> lastStepData) {
        if (CollUtil.isEmpty(lastStepData)) {
            return UpdateType.IGNORE;
        }

        // 查询API编码
        var apiCode = this.getStringValue(data, "api_code", null);
        if (CharSequenceUtil.isBlank(apiCode)) {
            var originalApiId = getLongValue(data, "api_id", null);
            Assert.notNull(originalApiId, "缺少api_id");
            var originalApiMap = lastStepData.parallelStream()
                    .filter(t -> Objects.equals(originalApiId, t.get("original_id")))
                    .findFirst()
                    .orElse(Collections.emptyMap());
            apiCode = getStringValue(originalApiMap, "api_code", null);
            if (CharSequenceUtil.isBlank(apiCode)) {
                log.info("未查询到API：{}", originalApiId);
                return UpdateType.IGNORE;
            }
        }

        var fieldName = this.getStringValue(data, "field_name", null);
        Assert.notBlank(fieldName, "参数名称为空");
        var type = this.getStringValue(data, "in_out_type", null);
        Assert.notBlank(type, "参数类型为空");

        normalizeBooleanValue(data, "field_is_null");

        if (currentData.isEmpty()) {
            return UpdateType.ADD;
        }

        String finalApiCode = apiCode;
        var id = currentData.stream()
                .filter(t -> finalApiCode.equals(t.get("api_code")) && fieldName.equals(t.get("field_name")) && type.equals(t.get("in_out_type")))
                .findFirst()
                .map(t -> t.get("id"))
                .orElse(null);
        data.put("id", id);

        // 获取api的id
        var currentApiId = lastStepData.parallelStream()
                .filter(t -> finalApiCode.equals(getStringValue(t, "api_code", ""))).findFirst()
                .map(t -> getLongValue(t, "id", null))
                .orElse(null);
        if (currentApiId == null) {
            return UpdateType.IGNORE;
        }
        data.put("api_id", currentApiId);

        return id == null ? UpdateType.ADD : UpdateType.UPDATE;
    }
}
