package com.elitescloud.cloudt.system.service.impl;

import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.auth.config.AuthorizationSdkProperties;
import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.auth.provider.common.AuthorizationConstant;
import com.elitescloud.boot.auth.provider.security.grant.InternalAuthenticationGranter;
import com.elitescloud.boot.auth.provider.sso2.common.TicketProvider;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.JwtUtil;
import com.elitescloud.boot.util.RsaUtil;
import com.elitescloud.boot.util.encrypt.BaseEncrypt;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.system.common.IdEncodedTypeEnum;
import com.elitescloud.cloudt.system.service.AuthUserService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.bootstrap.encrypt.KeyProperties;
import org.springframework.security.crypto.encrypt.TextEncryptor;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/10/16
 */
@Slf4j
@Service
public class AuthUserServiceImpl extends BaseServiceImpl implements AuthUserService {

    @Autowired
    private InternalAuthenticationGranter authenticationGranter;
    @Autowired
    private AuthorizationSdkProperties authorizationSdkProperties;
    @Autowired
    private TextEncryptor encryptor;
    @Autowired
    private KeyProperties keyProperties;

    @Override
    public ApiResult<OAuthToken> authenticate(HttpServletRequest request, HttpServletResponse response,
                                              String idType, String id) {
        Assert.hasText(id, "账户标识为空");

        // 创建认证令牌
        InternalAuthenticationGranter.IdType idTypeValue = InternalAuthenticationGranter.IdType.USERNAME;
        if (StringUtils.hasText(idType)) {
            try {
                idTypeValue = InternalAuthenticationGranter.IdType.valueOf(idType);
            } catch (IllegalArgumentException e) {
                return ApiResult.fail("不支持的账号标识类型：" + idType);
            }
        }
        InternalAuthenticationGranter.InternalAuthenticationToken authenticationToken = new InternalAuthenticationGranter.InternalAuthenticationToken(idTypeValue, id);

        // 开始认证
        if (request == null) {
            request = HttpServletUtil.currentRequest();
        }
        if (request != null) {
            var clientId = authorizationSdkProperties.getCasClient().getOauth2Client().getClientId();
            request.setAttribute(AuthorizationConstant.REQUEST_ATTRIBUTE_CLIENT_ID, clientId);
        }

        try {
            var token = authenticationGranter.authenticate(request, response, authenticationToken);
            return ApiResult.ok(token);
        } catch (Exception e) {
            log.info("认证异常：", e);
            return ApiResult.fail("认证失败，" + e.getMessage());
        }
    }

    @Override
    public ApiResult<OAuthToken> authenticateForEncoded(HttpServletRequest request, HttpServletResponse response,
                                                        String idType, String idEncoded, String cipherType) {
        Assert.hasText(idEncoded, "账户标识为空");

        String id = null;
        try {
            id = this.decode(idEncoded, cipherType);
        } catch (Exception e) {
            log.info("解密认证异常：{}，{}", idEncoded, cipherType, e);
            return ApiResult.fail("解密失败，" + e.getMessage());
        }
        return this.authenticate(request, response, idType, id);
    }

    @Override
    public ApiResult<String> ticket2Token(String ticket) {
        if (!StringUtils.hasText(ticket)) {
            return ApiResult.fail("ticket为空");
        }

        String token = (String) SpringContextHolder.getBean(TicketProvider.class).exchangeTicket(ticket);
        return ApiResult.ok(token);
    }

    private String decode(String idEncoded, String cipherType) throws Exception {
        IdEncodedTypeEnum encodedType = IdEncodedTypeEnum.RSA;
        if (StringUtils.hasText(cipherType)) {
            try {
                encodedType = IdEncodedTypeEnum.valueOf(cipherType);
            } catch (IllegalArgumentException e) {
                throw new IllegalArgumentException("不支持的解密算法" + cipherType);
            }
        }

        String idDecoded = null;
        switch (encodedType) {
            case NOOP:
                idDecoded = idEncoded;
                break;
            case CONFIG:
                idDecoded = encryptor.decrypt(idEncoded);
                break;
            case RSA:
                var privateKey = loadPrivateKey();
                idDecoded = RsaUtil.decrypt(privateKey, null, idEncoded);
                break;
            case BASE64:
                idDecoded = new String(BaseEncrypt.decodeBase64(idEncoded));
                break;
            default:
                throw new BusinessException("暂不支持的加密方式" + cipherType);
        }

        var ids = idDecoded.split("&");
        if (ids.length < 2) {
            throw new IllegalArgumentException("账号标识的格式不正确");
        }
        long timestamp = Long.parseLong(ids[1]);
        if (Math.abs(System.currentTimeMillis() - timestamp) > 60 * 1000 * 5) {
            throw new IllegalArgumentException("认证已超时");
        }
        return ids[0];
    }

    private PrivateKey loadPrivateKey() throws UnrecoverableKeyException, KeyStoreException, NoSuchAlgorithmException {
        if (privateKey != null) {
            return privateKey;
        }

        var props = keyProperties.getKeyStore();
        var keystore = JwtUtil.loadKeystore(props.getLocation(), props.getType(), props.getPassword(), props.getAlias(), props.getSecret());
        privateKey = (PrivateKey) keystore.getKey(props.getAlias(), props.getSecret().toCharArray());

        return privateKey;
    }

    private PrivateKey privateKey;
}
