package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.system.constant.SysEmpGroupType;
import com.elitescloud.cloudt.system.convert.EmpGroupConvert;
import com.elitescloud.cloudt.system.model.vo.query.org.EmpGroupPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpGroupListRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpGroupPageRespVO;
import com.elitescloud.cloudt.system.provider.dto.SysEmpGroupDTO;
import com.elitescloud.cloudt.system.service.EmpGroupQueryService;
import com.elitescloud.cloudt.system.service.repo.EmpGroupEmpRepoProc;
import com.elitescloud.cloudt.system.service.repo.EmpGroupRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2/14/2023
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class EmpGroupQueryServiceImpl extends BaseServiceImpl implements EmpGroupQueryService {

    @Autowired
    private EmpGroupRepoProc repoProc;
    @Autowired
    private EmpGroupEmpRepoProc empGroupEmpRepoProc;

    @Override
    public ApiResult<PagingVO<EmpGroupPageRespVO>> pageQuery(EmpGroupPageQueryVO queryVO) {
        var pageData = repoProc.pageMng(queryVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData.map(EmpGroupConvert.INSTANCE::do2PageRespVO));
        }

        // 员工组类型
        var typeNames = super.udcMap(new SysEmpGroupType());
        var result = pageData.map(t -> {
            var vo = EmpGroupConvert.INSTANCE.do2PageRespVO(t);
            vo.setTypeName(typeNames.get(t.getType()));
            return vo;
        });
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<List<EmpGroupListRespVO>> listQuery(String type, Boolean enabled) {
        var dataList = repoProc.listQuery(type, enabled);
        if (dataList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 员工组类型
        var typeNames = super.udcMap(new SysEmpGroupType());
        var result = dataList.stream().map(t -> {
            var vo = EmpGroupConvert.INSTANCE.do2ListRespVO(t);
            vo.setTypeName(typeNames.get(t.getType()));
            return vo;
        }).collect(Collectors.toList());
        return ApiResult.ok(result);
    }

    @Override
    public ApiResult<SysEmpGroupDTO> getByCode(String groupCode) {
        if (CharSequenceUtil.isBlank(groupCode)) {
            return ApiResult.fail("员工组编码为空");
        }

        var groupDTO = repoProc.getDtoByCode(groupCode);
        return ApiResult.ok(groupDTO);
    }

    @Override
    public ApiResult<SysEmpGroupDTO> get(Long groupId) {
        if (groupId == null) {
            return ApiResult.fail("员工组ID为空");
        }

        var groupDTO = repoProc.getDto(groupId);
        return ApiResult.ok(groupDTO);
    }

    @Override
    public ApiResult<List<SysEmpGroupDTO>> listAll() {
        var groupList = repoProc.getDtoList(true);
        return ApiResult.ok(groupList);
    }

    @Override
    public ApiResult<List<Long>> getUserIdsByGroupCode(String groupCode) {
        Assert.hasText(groupCode, "员工组编号为空");

        var userIds = empGroupEmpRepoProc.queryUserIdOfGroup(groupCode);
        return ApiResult.ok(userIds);
    }

    @Override
    public ApiResult<Map<String, List<Long>>> queryUserIdsByGroupCodes(Set<String> groupCodes) {
        if (groupCodes != null) {
            groupCodes = groupCodes.stream().filter(StringUtils::hasText).collect(Collectors.toSet());
        }

        var groupCodeUserIdMap = empGroupEmpRepoProc.queryUserIdOfGroup(groupCodes);
        return ApiResult.ok(groupCodeUserIdMap);
    }

    @Override
    public ApiResult<Map<String, List<IdCodeNameParam>>> queryUsersByGroupCodes(Set<String> groupCodes) {
        if (groupCodes != null) {
            groupCodes = groupCodes.stream().filter(StringUtils::hasText).collect(Collectors.toSet());
        }

        var groupCodeUserIdMap = empGroupEmpRepoProc.queryUserOfGroup(groupCodes);
        return ApiResult.ok(groupCodeUserIdMap);
    }
}
