package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.common.OwnerType;
import com.elitescloud.cloudt.system.model.entity.SysFrontTableCfgDO;
import com.elitescloud.cloudt.system.model.vo.save.extend.FrontTableCfgSaveVO;
import com.elitescloud.cloudt.system.service.FrontTableCfgService;
import com.elitescloud.cloudt.system.service.repo.FrontTableCfgRepo;
import com.elitescloud.cloudt.system.service.repo.FrontTableCfgRepoProc;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/4/20
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@Log4j2
public class FrontTableCfgServiceImpl extends BaseServiceImpl implements FrontTableCfgService {
    @Autowired
    private FrontTableCfgRepo repo;
    @Autowired
    private FrontTableCfgRepoProc repoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Boolean> saveCfg(FrontTableCfgSaveVO saveVO, boolean global) {
        // 解析拥有者
        var currentUser = super.currentUser(true);
        OwnerType ownerType = null;
        Long ownerId = null;
        if (global) {
            ownerType = OwnerType.TENANT;
            ownerId = ObjectUtil.defaultIfNull(currentUser.getTenantId(), TenantConstant.DEFAULT_TENANT_ID);
            this.updateGlobalCache(saveVO.getTableCode(), saveVO.getConfig(), ownerId);
        } else {
            ownerType = OwnerType.USER;
            ownerId = currentUser.getUserId();
            this.updateUserCache(saveVO.getTableCode(), saveVO.getConfig(), ownerId);
        }

        Long id = repoProc.getId(ownerId, ownerType, saveVO.getTableCode());
        if (StringUtils.hasText(saveVO.getConfig())) {
            // 保存配置
            if (id == null) {
                // 新增配置
                SysFrontTableCfgDO cfgDO = new SysFrontTableCfgDO();
                cfgDO.setTableCode(saveVO.getTableCode());
                cfgDO.setOwnerId(ownerId);
                cfgDO.setOwnerType(ownerType);
                cfgDO.setConfig(saveVO.getConfig());
                repo.save(cfgDO);
                return ApiResult.ok(true);
            }
            // 修改配置
            repoProc.updateConfig(id, saveVO.getConfig());
            return ApiResult.ok(true);
        }
        // 删除配置
        repoProc.delete(id);

        return ApiResult.ok(true);
    }

    @Override
    public ApiResult<String> getCfg(String tableCode, boolean global) {
        // 解析拥有者
        OwnerType ownerType = null;
        Long ownerId = null;
        if (global) {
            ownerType = OwnerType.TENANT;
            ownerId = super.currentTenantId();
        } else {
            var currentUser = super.currentUser(true);
            ownerType = OwnerType.USER;
            ownerId = currentUser.getUserId();
        }

        // 获取配置
        var cfgDO = repoProc.get(ownerId, ownerType, tableCode);
        return ApiResult.ok(cfgDO == null ? null : cfgDO.getConfig());
    }

    @Override
    public ApiResult<String> getUserCfg(String tableCode) {
        String config = null;
        Set<Long> ownerIds = new HashSet<>();
        var currentUser = super.currentUser(false);
        Long tenantId = null;
        if (currentUser == null) {
            // 优先查缓存
            tenantId = super.currentTenantId();
            config = this.getGlobalCache(tableCode, tenantId);
            if (config != null) {
                return ApiResult.ok(config);
            }
            ownerIds.add(tenantId);
        } else {
            // 优先查缓存
            tenantId = ObjectUtil.defaultIfNull(currentUser.getTenantId(), TenantConstant.DEFAULT_TENANT_ID);
            config = ObjectUtil.defaultIfBlank(this.getUserCache(tableCode, currentUser.getUserId()), this.getGlobalCache(tableCode, tenantId));
            if (config != null) {
                return ApiResult.ok(config);
            }
            ownerIds.add(currentUser.getUserId());
            ownerIds.add(tenantId);
        }

        var cfgMap = repoProc.queryUserConfig(ownerIds, tableCode);
        if (cfgMap.isEmpty()) {
            if (currentUser == null) {
                this.updateGlobalCache(tableCode, null, tenantId);
            } else {
                this.updateUserCache(tableCode, null, currentUser.getUserId());
            }
            return ApiResult.ok(null);
        }
        config = cfgMap.get(OwnerType.USER);
        if (CharSequenceUtil.isBlank(config)) {
            config = cfgMap.get(OwnerType.TENANT);
            this.updateGlobalCache(tableCode, null, tenantId);
        } else if (currentUser != null){
            this.updateUserCache(tableCode, null, currentUser.getUserId());
        }

        return ApiResult.ok(config);
    }

    private String cacheKeyTenant(String tableCode, long tenantId) {
        return "cloudt:frontConfigTable:tenant:" + tenantId + ":" + tableCode;
    }

    private String cacheKeyUser(String tableCode, long userId) {
        return "cloudt:frontConfigTable:user:" + userId + ":" + tableCode;
    }

    private String getGlobalCache(String tableCode, long tenantId) {
        var key = this.cacheKeyTenant(tableCode, tenantId);
        return (String) redisUtils.get(key);
    }

    private String getUserCache(String tableCode, long userId) {
        var key = this.cacheKeyUser(tableCode, userId);
        return (String) redisUtils.get(key);
    }

    private void updateGlobalCache(String tableCode, String config, long tenantId) {
        var key = this.cacheKeyTenant(tableCode, tenantId);
        redisUtils.set(key, ObjectUtil.defaultIfNull(config, ""), 7, TimeUnit.DAYS);
    }

    private void updateUserCache(String tableCode, String config, long userId) {
        var key = this.cacheKeyUser(tableCode, userId);
        redisUtils.set(key, ObjectUtil.defaultIfNull(config, ""), 7, TimeUnit.DAYS);
    }
}
