package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.openfeign.common.DynamicClientHelper;
import com.elitescloud.boot.swagger.openapi.common.OpenApiRpcClient;
import com.elitescloud.boot.swagger.openapi.model.BusinessObjectOperationInfo;
import com.elitescloud.boot.swagger.openapi.service.impl.OpenApiService;
import com.elitescloud.boot.swagger.openapi.service.impl.OpenApiServiceUtil;
import com.elitescloud.boot.swagger.openapi.swagger3.core.util.Json;
import com.elitescloud.boot.swagger.openapi.swagger3.models.OpenAPI;
import com.elitescloud.boot.swagger.openapi.swagger3.models.Operation;
import com.elitescloud.boot.swagger.openapi.swagger3.models.tags.Tag;
import com.elitescloud.boot.util.*;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.OpenApiConvert;
import com.elitescloud.cloudt.system.datasource.SysPlatformDatabaseSourceQueryParam;
import com.elitescloud.cloudt.system.datasource.service.SysPlatformDatabaseSourceRpcService;
import com.elitescloud.cloudt.system.model.bo.OpenApiBO;
import com.elitescloud.cloudt.system.model.vo.query.sys.OpenApiOperationPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.sys.OpenApiPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.OpenApiOperationPageRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.OpenApiPageRespVO;
import com.elitescloud.cloudt.system.service.OpenApiMngService;
import com.elitescloud.cloudt.system.service.common.constant.OpenApiSourceEnum;
import com.elitescloud.cloudt.system.service.common.constant.OpenApiStatusEnum;
import com.elitescloud.cloudt.system.service.model.entity.*;
import com.elitescloud.cloudt.system.service.repo.*;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/4
 */
@Slf4j
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
public class OpenApiMngServiceImpl implements OpenApiMngService {

    @Autowired
    private OpenApiInfoRepoProc infoRepoProc;
    @Autowired
    private OpenApiOperationRepoProc operationRepoProc;
    @Autowired
    private OpenApiTagRepoProc tagRepoProc;
    @Autowired
    private OpenApiSchemaRepoProc schemaRepoProc;
    @Autowired
    private OpenApiResourceRepoProc resourceRepoProc;
    @Autowired
    private BusinessObjectRepoProc businessObjectRepoProc;
    @Autowired
    private AppRepoProc appRepoProc;

    @Autowired
    private OpenApiService openApiService;
    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    private PlatformTransactionManager transactionManager;

    @Override
    public ApiResult<OpenAPI> testAnalyzeController(Class<?>... beanClasses) {
        // 模仿从RPC接口获取到的OpenApi信息
        List<Object> controllers = new ArrayList<>();
        if (beanClasses != null) {
            for (var beanClass : beanClasses) {
                controllers.add(SpringContextHolder.getBean(beanClass));
            }
        }
        var openApiInfo = controllers.isEmpty() ? openApiService.generateOpenApi() :
                openApiService.generateOpenApi(controllers.toArray());
        var openApiInfoRpc = JSONUtil.json2Obj(JSONUtil.toJsonString(openApiInfo), OpenAPI.class, true);
        System.out.println("##########1");
        System.out.println(Json.pretty(openApiInfoRpc));

        var openApiBo = this.initOpenApiInfo(openApiInfo);
        this.convertOpenApi(openApiBo, openApiInfo);

        var openApiBO2 = this.initOpenApiInfo(openApiInfoRpc);
        this.convertOpenApi(openApiBO2, openApiInfoRpc);

        return ApiResult.ok(openApiInfo);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateOpenApi(String appCode) {
        if (CharSequenceUtil.isBlank(appCode)) {
            return ApiResult.fail("应用编码为空");
        }

        // 先根据应用查询是否已有
        var openApiId = infoRepoProc.getId(null, null, appCode);
        if (openApiId != null) {
            return this.updateOpenApi(openApiId);
        }

        // 保存OpenApi信息
        openApiId = this.updateOpenApiByRpcService(null, appCode);
        return ApiResult.ok(openApiId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateOpenApi(Long openApiId) {
        if (openApiId == null) {
            return ApiResult.fail("请选择要更新的记录");
        }

        // 获取服务名
        var appCode = infoRepoProc.getAppCode(openApiId);
        if (CharSequenceUtil.isBlank(appCode)) {
            return ApiResult.fail("更新的记录不存在或记录信息异常");
        }

        // 保存OpenApi信息
        this.updateOpenApiByRpcService(openApiId, appCode);
        return ApiResult.ok(openApiId);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateOpenApi(OpenAPI openApi, OpenApiSourceEnum source) {
        var openApiId = this.convertAndSave(null, openApi, source);
        return ApiResult.ok(openApiId);
    }

    @Override
    public ApiResult<PagingVO<OpenApiPageRespVO>> pageMng(OpenApiPageQueryVO queryVO) {
        PagingVO<OpenApiPageRespVO> pageData = infoRepoProc.pageMng(queryVO)
                .map(t -> {
                    var respVO = OpenApiConvert.INSTANCE.do2PageRespVO(t);
                    if (CharSequenceUtil.isBlank(respVO.getFailReason())) {
                        respVO.setFailReason(t.getBusinessObjectFailReason());
                    }
                    return respVO;
                });
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData);
        }

        var ids = pageData.stream().map(OpenApiPageRespVO::getId).collect(Collectors.toList());
        var codes = pageData.stream().map(OpenApiPageRespVO::getOpenApiCode).collect(Collectors.toList());
        // 获取操作数量
        var operationCountMap = operationRepoProc.countByOpenApiId(ids);
        // RPC获取操作数量
        var feignOperationCountMap = operationRepoProc.countByFeignOpenApiId(ids);
        // 获取业务对象数量
        var businessObjectCountMap = businessObjectRepoProc.countByOpenApi(codes);

        pageData.each(t -> {
            if (CharSequenceUtil.isNotBlank(t.getState())) {
                ObjUtil.ifNotNull(OpenApiStatusEnum.parse(t.getState()), s -> t.setStateName(s.getDescription()));
            }
            if (CharSequenceUtil.isNotBlank(t.getBusinessObjectState())) {
                ObjUtil.ifNotNull(OpenApiStatusEnum.parse(t.getBusinessObjectState()), s -> t.setBusinessObjectStateName(s.getDescription()));
            }
            t.setOperationNum(operationCountMap.getOrDefault(t.getId(), 0L));
            t.setBusinessObjectNum(businessObjectCountMap.getOrDefault(t.getOpenApiCode(), 0L));
            t.setFeignOperationNum(feignOperationCountMap.getOrDefault(t.getId(), 0L));
        });

        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<PagingVO<OpenApiOperationPageRespVO>> pageMng(OpenApiOperationPageQueryVO queryVO) {
        if (queryVO.getOpenApiId() == null) {
            return ApiResult.fail("OpenApiId为空");
        }
        var pageData = operationRepoProc.pageMng(queryVO)
                .map(t -> {
                    var respVO = OpenApiConvert.INSTANCE.do2PageRespVO(t);
                    if (StringUtils.hasText(t.getTagsJson())) {
                        respVO.setTags(JSONUtil.json2Obj(t.getTagsJson(), new TypeReference<String[]>() {
                        }));
                    }
                    return respVO;
                });
        return ApiResult.ok(pageData);
    }

    @Override
    public ApiResult<List<String>> selectDistinctOperationGroupByApiType() {
        return ApiResult.ok(operationRepoProc.selectDistinctOperationGroupByApiType());
    }

    private Long updateOpenApiByRpcService(Long openApiId, @NotBlank String appCode) {
        // 构建客户端实例
        var openApiClient = DynamicClientHelper.getClient(appCode, OpenApiRpcClient.class, OpenApiRpcClient.URI);
        // rpc调用获取openApi信息
        ApiResult<OpenAPI> openApiResult = null;
        try {
            openApiResult = openApiClient.getOpenApi();
        } catch (Exception e) {
            log.error("调用服务{}获取OpenAPI异常：", appCode, e);
            throw new BusinessException("调用服务异常，请确认服务" + appCode + "在线且支持OpenApi");
        }

        if (openApiResult.isFailed() || openApiResult.getData() == null) {
            throw new BusinessException(openApiResult.getMsg());
        }

        // 保存OpenApi信息
        return this.convertAndSave(openApiId, openApiResult.getData(), OpenApiSourceEnum.GATHER_PULL);
    }

    private Long convertAndSave(Long existsOpenApiId, @NotNull OpenAPI openApi, @NotNull OpenApiSourceEnum source) {
        // 初始化基本信息记录
        OpenApiBO openApiBo = this.initOpenApiInfo(openApi);
        openApiBo.setId(existsOpenApiId);

        // 判断应用是否存在
        var appCodes = appRepoProc.allCodes(null);
        Assert.isTrue(appCodes.contains(openApiBo.getAppCode()), "应用编码不正确或应用不存在");

        // 加锁，根据状态判断是否有正在进行中的
        String lockKey = "CloudtSystem:OpenApi:Create:" + this.generateOpenApiCode(openApiBo);
        existsOpenApiId = LockUtil.executeByLock(lockKey, () -> {
            var tempOpenApiId = openApiBo.getId() == null ? this.getExistsOpenApiId(openApiBo) : openApiBo.getId();
            if (tempOpenApiId == null) {
                // 之前没有，则新增
                openApiBo.setOriginalDataJson(Json.pretty(openApi).getBytes(StandardCharsets.UTF_8));
                var openApiInfoDO = this.saveOpenApiBaseInfo(openApiBo, false);
                return openApiInfoDO.getId();
            }

            // 之前已有记录，判断是否有正在更新中的
            if (OpenApiStatusEnum.UPDATING == infoRepoProc.getState(tempOpenApiId)) {
                throw new BusinessException("应用[" + openApiBo.getAppName() + "]正在更新OpenAPI中，请稍后再试");
            }
            if (OpenApiStatusEnum.UPDATING == infoRepoProc.getBusinessObjectState(tempOpenApiId)) {
                throw new BusinessException("应用[" + openApiBo.getAppName() + "]正在更新业务对象中，请稍后再试");
            }
            infoRepoProc.updateStartUpdate(tempOpenApiId);
            return tempOpenApiId;
        }, Duration.ofMinutes(2));
        openApiBo.setId(existsOpenApiId);


        // 开始转换和保存操作记录、参数
        CompletableFuture.supplyAsync(() -> {
            // 转换对象
            var openApiBoNew = this.convertOpenApi(openApiBo, openApi);

            // 保存OpenApiInfo
            DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            transactionDefinition.setName("OpenApi2BusinessObject");
            var transaction = transactionManager.getTransaction(transactionDefinition);
            Long openApiId = null;
            try {
                openApiId = this.saveOrUpdate(openApiBoNew, source);
                transactionManager.commit(transaction);
            } catch (Exception e) {
                if (!transaction.isCompleted()) {
                    transactionManager.rollback(transaction);
                }
                throw e;
            }
            return openApiId;
        }, taskExecutor).whenComplete((openApiId, e) -> {
            if (e == null) {
                // 更新成功
                infoRepoProc.updateEndUpdate(openApiId, OpenApiStatusEnum.FINISHED, null);
                infoRepoProc.updateBusinessObjectStatus(openApiId, OpenApiStatusEnum.TO_UPDATE, null);
                return;
            }
            log.error("保存OpenAPI信息异常：", e);
            infoRepoProc.updateEndUpdate(openApiBo.getId(), OpenApiStatusEnum.FAILED, e.getMessage());
        });
        return existsOpenApiId;
    }

    private SysOpenApiInfoDO saveOpenApiBaseInfo(OpenApiBO openApiBo, boolean saveResource) {
        var existsOpenApi = openApiBo.getId() == null ? new SysOpenApiInfoDO() : infoRepoProc.get(openApiBo.getId());
        var openApiInfoDO = OpenApiConvert.INSTANCE.bo2Do(openApiBo, existsOpenApi);
        openApiInfoDO.setOpenApiCode(this.generateOpenApiCode(openApiBo));
        openApiInfoDO.setState(OpenApiStatusEnum.UPDATING.name());
        if (openApiInfoDO.getId() == null) {
            openApiInfoDO.setStartUpdateTime(LocalDateTime.now());
        }
        openApiInfoDO.setBusinessObjectState(OpenApiStatusEnum.PREPARING.name());
        openApiInfoDO.setResourceId(null);
        openApiInfoDO.setBusinessResourceId(null);
        infoRepoProc.save(openApiInfoDO);

        // 保存资源
        if (saveResource) {
            Long resourceId = this.saveResource(openApiInfoDO.getId(), openApiBo.getOriginalDataJson(), "OpenApiInfo");
            if (resourceId != null) {
                infoRepoProc.updateResourceId(openApiInfoDO.getId(), resourceId);
            }

            // 保存业务对象信息
            if (CollUtil.isNotEmpty(openApiBo.getBusinessObjectMap())) {
                resourceId = this.saveResource(openApiInfoDO.getId(), JSONUtil.toJsonBytes(openApiBo.getBusinessObjectMap()), "OpenApiBusinessObject");
                if (resourceId != null) {
                    infoRepoProc.updateBusinessResourceId(openApiInfoDO.getId(), resourceId);
                }
            }
        }

        return openApiInfoDO;
    }

    private void saveOpenApiOperation(SysOpenApiInfoDO openApiInfoDO, OpenApiBO openApi, OpenApiSourceEnum source) {
        if (CollUtil.isEmpty(openApi.getOperationList())) {
            return;
        }
        var openApiId = openApiInfoDO.getId();

        List<TagBO> tagBoList = new ArrayList<>();
        List<SysOpenApiOperationDO> operationDoList = openApi.getOperationList().stream().map(t -> {
            // 保存资源
            var resourceId = this.saveResource(openApiId, t.getOriginalDataJson(), "OpenApiOperation");

            var openApiOperationDO = OpenApiConvert.INSTANCE.bo2Do(t);
            openApiOperationDO.setOpenApiInfoId(openApiId);
            openApiOperationDO.setOperationCode(this.generateOperationCode(openApi, t));
            openApiOperationDO.setDataSource(source.name());
            openApiOperationDO.setTagsJson(JSONUtil.toJsonString(t.getTags()));
            openApiOperationDO.setResourceId(resourceId);
            openApiOperationDO.setBusinessOperation(t.getBusinessOperation());

            // tag信息
            if (ArrayUtil.isNotEmpty(t.getTags())) {
                int i = 0;
                for (String tag : t.getTags()) {
                    tagBoList.add(new TagBO(tag, i++, openApiOperationDO));
                }
            }

            return openApiOperationDO;
        }).collect(Collectors.collectingAndThen(
                Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(SysOpenApiOperationDO::getOperationCode))),
                ArrayList::new
        ));
        operationRepoProc.save(operationDoList);

        // 保存标签
        var tagDoList = tagBoList.stream().map(t -> {
            SysOpenApiTagDO tagDO = new SysOpenApiTagDO();
            tagDO.setOpenApiInfoId(openApiId);
            tagDO.setOperationId(t.operationDO.getId());
            tagDO.setTag(t.tagName);
            tagDO.setSortNo(t.sortNo);

            return tagDO;
        }).filter(t -> t.getOperationId() != null).collect(Collectors.toList());
        if (!tagDoList.isEmpty()) {
            tagRepoProc.save(tagDoList);
        }
    }

    private void saveOpenApiCommentSchema(SysOpenApiInfoDO openApiInfoDO, OpenApiBO openApi) {
        if (CollUtil.isEmpty(openApi.getComponentSchemaList())) {
            return;
        }
        var openApiId = openApiInfoDO.getId();

        List<SysOpenApiComponentSchemaDO> schemaDoList = openApi.getComponentSchemaList().stream().map(t -> {
            // 保存资源
            var resourceId = this.saveResource(openApiId, t.getOriginalDataJson(), "OpenApiComponentSchema");

            var schemaDO = OpenApiConvert.INSTANCE.bo2Do(t);
            schemaDO.setOpenApiInfoId(openApiId);
            schemaDO.setResourceId(resourceId);
            return schemaDO;
        }).collect(Collectors.collectingAndThen(
                Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(SysOpenApiComponentSchemaDO::getSchemaPath))),
                ArrayList::new
        ));
        schemaRepoProc.save(schemaDoList);
    }

    private Long saveOrUpdate(@NotNull OpenApiBO openApi, @NotNull OpenApiSourceEnum source) {
        log.info("保存OpenApi信息：{}，{}，{}", openApi.getProject(), openApi.getEnv(), openApi.getAppCode());

        // 先删除现有数据
        operationRepoProc.deleteByOpenApiId(openApi.getId());
        schemaRepoProc.deleteByOpenApiId(openApi.getId());
        tagRepoProc.deleteByOpenApiId(openApi.getId());
        resourceRepoProc.deleteByOpenApiId(openApi.getId());

        // 保存基本信息
        var openApiInfoDO = this.saveOpenApiBaseInfo(openApi, true);
        var openApiId = openApiInfoDO.getId();

        // 保存接口信息
        this.saveOpenApiOperation(openApiInfoDO, openApi, source);

        // 保存接口参数
        this.saveOpenApiCommentSchema(openApiInfoDO, openApi);

        return openApiId;
    }

    private Long saveResource(long openApiId, byte[] resourceContent, String... bs) {
        if (resourceContent == null) {
            return null;
        }
        SysOpenApiResourceDO resourceDO = new SysOpenApiResourceDO();
        resourceDO.setOpenApiInfoId(openApiId);
        resourceDO.setResourceByte(resourceContent);
        if (ArrayUtil.isNotEmpty(bs)) {
            resourceDO.setExt1(bs[0]);
            if (bs.length == 2) {
                resourceDO.setExt2(bs[1]);
            }
        }

        resourceRepoProc.save(resourceDO);
        return resourceDO.getId();
    }

    private Long getExistsOpenApiId(OpenApiBO openApiBO) {
        return infoRepoProc.getId(openApiBO.getProject(), openApiBO.getEnv(), openApiBO.getAppCode());
    }

    private String generateOpenApiCode(OpenApiBO openApiBO) {
        return openApiBO.getAppCode();
    }

    private String generateOperationCode(OpenApiBO openApiBO, OpenApiBO.Operation operation) {
        if (StringUtils.hasText(operation.getOperationCode())) {
            return operation.getOperationCode();
        }
        return openApiBO.getAppCode() + ":" + operation.getMethod() + ":" + operation.getUrl();
    }

    private void deleteByService(String appCode, String env, String project) {
        var openId = infoRepoProc.getId(project, env, appCode);
        if (openId == null) {
            return;
        }

        infoRepoProc.delete(openId);
        operationRepoProc.deleteByOpenApiId(openId);
        schemaRepoProc.deleteByOpenApiId(openId);
    }

    @SuppressWarnings("unchecked")
    private OpenApiBO convertOpenApi(OpenApiBO bo, OpenAPI openApi) {
        if (bo == null) {
            bo = this.initOpenApiInfo(openApi);
        }
        if (ArrayUtil.isEmpty(bo.getOriginalDataJson())) {
            bo.setOriginalDataJson(Json.pretty(openApi).getBytes(StandardCharsets.UTF_8));
        }

        // 扩展信息
        var extensions = openApi.getExtensions();
        if (extensions != null) {
            bo.setBusinessObjectMap((Map<String, Object>) extensions.get(OpenApiServiceUtil.X_BUSINESS_OBJECTS));
        }

        // 接口信息
        bo.setOperationList(this.obtainOperationBO(openApi, bo));

        // 参数信息
        bo.setComponentSchemaList(this.obtainSchemaBO(openApi));

        return bo;
    }

    private OpenApiBO initOpenApiInfo(OpenAPI openAPI) {
        Assert.notNull(openAPI, "OpenAPI为空");
        var apiInfo = openAPI.getInfo();
        Assert.notNull(apiInfo, "OpenAPI信息异常");

        OpenApiBO bo = new OpenApiBO();

        bo.setTitle(apiInfo.getTitle());
        bo.setDescription(apiInfo.getDescription());
        bo.setVersion(apiInfo.getVersion());
        // 扩展信息
        var ext = apiInfo.getExtensions();
        if (MapUtil.isNotEmpty(ext)) {
            var buildTimeStr = ext.get(OpenApiServiceUtil.X_NEW_DATE);
            if (buildTimeStr instanceof String) {
                bo.setBuildTime(DatetimeUtil.parseLocalDateTime((String) buildTimeStr));
            }
            bo.setProject((String) ext.get(OpenApiServiceUtil.X_PROJECT));
            bo.setProjectName((String) ext.get(OpenApiServiceUtil.X_PROJECT_NAME));
            bo.setEnv((String) ext.get(OpenApiServiceUtil.X_ENV));
            bo.setEnvName((String) ext.get(OpenApiServiceUtil.X_ENV_NAME));
            bo.setAppCode((String) ext.get(OpenApiServiceUtil.X_SERVICE));
            bo.setAppName((String) ext.get(OpenApiServiceUtil.X_SERVICE_NAME));
            bo.setProjectVersion((String) ext.get(OpenApiServiceUtil.X_PROJECT_VERSION));
            bo.setCloudtBootVersion((String) ext.get(OpenApiServiceUtil.X_CLOUDT_BOOT_VERSION));
        }

        // 标签
        var tags = openAPI.getTags() == null ? null : openAPI.getTags().stream().map(Tag::getName).filter(StringUtils::hasText).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(tags)) {
            bo.setTagsJson(JSONUtil.toJsonString(tags));
        }

        return bo;
    }

    private List<OpenApiBO.Operation> obtainOperationBO(OpenAPI openApi, OpenApiBO bo) {
        if (CollUtil.isEmpty(openApi.getPaths())) {
            return Collections.emptyList();
        }
        return openApi.getPaths().entrySet().parallelStream().flatMap(pathsEntry ->
                Stream.of(
                        this.convertOperation(pathsEntry.getValue().getGet(), pathsEntry.getKey(), RequestMethod.GET),
                        this.convertOperation(pathsEntry.getValue().getPost(), pathsEntry.getKey(), RequestMethod.POST),
                        this.convertOperation(pathsEntry.getValue().getPut(), pathsEntry.getKey(), RequestMethod.PUT),
                        this.convertOperation(pathsEntry.getValue().getPatch(), pathsEntry.getKey(), RequestMethod.PATCH),
                        this.convertOperation(pathsEntry.getValue().getDelete(), pathsEntry.getKey(), RequestMethod.DELETE),

                        this.convertOperation(pathsEntry.getValue().getTrace(), pathsEntry.getKey(), RequestMethod.TRACE),
                        this.convertOperation(pathsEntry.getValue().getOptions(), pathsEntry.getKey(), RequestMethod.OPTIONS),
                        this.convertOperation(pathsEntry.getValue().getHead(), pathsEntry.getKey(), RequestMethod.HEAD)
                )
        ).filter(Objects::nonNull).collect(Collectors.toList());
    }

    @SuppressWarnings("unchecked")
    private OpenApiBO.Operation convertOperation(Operation operation, String path, RequestMethod method) {
        if (operation == null) {
            return null;
        }
        OpenApiBO.Operation bo = new OpenApiBO.Operation();
        bo.setOperationId(operation.getOperationId());
        bo.setSummary(operation.getSummary());
        bo.setDescription(operation.getDescription());
        bo.setTags(operation.getTags() == null ? new String[0] : operation.getTags().stream().filter(StringUtils::hasText).distinct().toArray(String[]::new));
        bo.setOriginalDataJson(JSONUtil.toJsonBytes(operation));
        bo.setEnabled(true);
        bo.setUrl(path);
        bo.setMethod(method.name());
        bo.setBusinessOperation(this.supportBusinessObjectOperation(operation));
        if (operation.getExtensions() != null) {
            var operationGroup = operation.getExtensions().get(OpenApiServiceUtil.OPERATION_GROUP);
            if (operationGroup != null) {
                bo.setOperationGroup((String) operationGroup);
            }
            var apiType = operation.getExtensions().get(OpenApiServiceUtil.API_TYPE);
            if (apiType != null) {
                bo.setApiType((Integer) apiType);
            }
            var parameterTypesJson = operation.getExtensions().get(OpenApiServiceUtil.PARAMETER_TYPES_JSON);
            if (parameterTypesJson != null) {
                bo.setParameterTypesJson((String) parameterTypesJson);
            }
            var returnTypeJson = operation.getExtensions().get(OpenApiServiceUtil.RETURN_TYPE_JSON);
            if (returnTypeJson != null) {
                bo.setReturnTypeJson((String) returnTypeJson);
            }

        }

        if (bo.getBusinessOperation()) {
            // 支持业务对象操作的，从业务操作中获取操作编码
            var businessObjectOperationInfo = operation.getExtensions().get(OpenApiServiceUtil.X_BUSINESS_OBJECT_OPERATION);
            if (businessObjectOperationInfo instanceof BusinessObjectOperationInfo) {
                bo.setOperationCode(((BusinessObjectOperationInfo) businessObjectOperationInfo).getOperationCode());
            } else if (businessObjectOperationInfo instanceof Map) {
                var operationCode = ((Map<String, Object>) businessObjectOperationInfo).get("operationCode");
                bo.setOperationCode(operationCode == null ? null : operationCode.toString());
            } else {
                throw new BusinessException("解析业务操作异常");
            }
        }

        return bo;
    }

    private boolean supportBusinessObjectOperation(Operation operation) {
        if (operation == null) {
            return false;
        }

        var ext = operation.getExtensions();
        return ext != null && ext.containsKey(OpenApiServiceUtil.X_BUSINESS_OBJECT_OPERATION);
    }

    private List<OpenApiBO.ComponentSchema> obtainSchemaBO(OpenAPI openApi) {
        var components = openApi.getComponents();
        if (components == null) {
            return Collections.emptyList();
        }
        var schemas = components.getSchemas();
        if (CollUtil.isEmpty(schemas)) {
            return Collections.emptyList();
        }

        return schemas.entrySet().parallelStream()
                .map(entry -> {
                    OpenApiBO.ComponentSchema schema = new OpenApiBO.ComponentSchema();
                    schema.setSchemaPath("#/components/schemas/" + entry.getKey());
                    schema.setSchemaType(entry.getValue().getType());
                    schema.setOriginalDataJson(JSONUtil.toJsonBytes(entry.getValue()));

                    return schema;
                }).collect(Collectors.toList());
    }

    static class TagBO {
        private final String tagName;
        private final int sortNo;
        private final SysOpenApiOperationDO operationDO;

        public TagBO(String tagName, int sortNo, SysOpenApiOperationDO operationDO) {
            this.tagName = tagName;
            this.sortNo = sortNo;
            this.operationDO = operationDO;
        }
    }
}
