package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.convert.RoleGroupConvert;
import com.elitescloud.cloudt.system.model.vo.query.role.RoleGroupPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RoleGroupDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.role.RoleGroupPageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.role.RoleGroupSaveVO;
import com.elitescloud.cloudt.system.service.RoleGroupMngService;
import com.elitescloud.cloudt.system.service.common.constant.BelongType;
import com.elitescloud.cloudt.system.service.common.constant.SysInnerConstant;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleDO;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleGroupDO;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleRelatedDO;
import com.elitescloud.cloudt.system.service.repo.OrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.RoleGroupRepoProc;
import com.elitescloud.cloudt.system.service.repo.RoleRelatedRepoProc;
import com.elitescloud.cloudt.system.service.repo.RoleRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/11/24
 */
@Service
public class RoleGroupMngServiceImpl implements RoleGroupMngService {

    @Autowired
    private RoleGroupRepoProc roleGroupRepoProc;
    @Autowired
    private RoleRepoProc roleRepoProc;
    @Autowired
    private RoleRelatedRepoProc roleRelatedRepoProc;
    @Autowired
    private OrgRepoProc orgRepoProc;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> upsert(RoleGroupSaveVO saveVO, BelongType.Belonger belong) {
        if (belong == null) {
            belong = BelongType.getBelonger(false);
        }

        SysRoleGroupDO groupDO = null;
        if (saveVO.getId() == null) {
            if (CharSequenceUtil.isNotBlank(saveVO.getGroupCode()) && roleGroupRepoProc.existsCode(saveVO.getGroupCode(), belong)) {
                return ApiResult.fail("分组编码已存在");
            }
            groupDO = RoleGroupConvert.INSTANCE.saveVo2DO(saveVO);
        } else {
            groupDO = roleGroupRepoProc.get(saveVO.getId());
            if (groupDO == null) {
                return ApiResult.fail("数据不存在");
            }
            if (CharSequenceUtil.isNotBlank(saveVO.getGroupCode()) && !saveVO.getGroupCode().equals(groupDO.getGroupCode())) {
                if (roleGroupRepoProc.existsCode(saveVO.getGroupCode(), belong)) {
                    return ApiResult.fail("分组编码已存在");
                }
            }
            RoleGroupConvert.INSTANCE.copySaveVO(saveVO, groupDO);
        }

        groupDO.setEnabled(ObjectUtil.defaultIfNull(groupDO.getEnabled(), true));
        groupDO.setType(belong.getBelongType().getValue());
        groupDO.setTypeId(belong.getBelongId());

        // 保存数据
        roleGroupRepoProc.save(groupDO);
        // 保存角色关联关系
        saveRelated(groupDO.getId().toString(), saveVO.getRelatedOrgCodes());
        return ApiResult.ok(groupDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        if (id == null) {
            return ApiResult.fail("id不能为空");
        }
        if (id == SysRoleDO.DEFAULT_GROUP_ID) {
            return ApiResult.fail("默认分组不可删除");
        }
        if (!roleGroupRepoProc.exists(id)) {
            return ApiResult.ok(id);
        }

        // 判断下面是否还有角色
        if (roleRepoProc.existsGroupId(id)) {
            return ApiResult.fail("请先删除组内的角色");
        }

        String groupCode = roleGroupRepoProc.getCode(id);

        roleGroupRepoProc.delete(id);
        // 删除角色的关联关系
        roleRepoProc.clearGroupId(id);
        roleRelatedRepoProc.deletedByRoleCode(id.toString(), true);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id, Boolean enabled) {
        if (id == null) {
            return ApiResult.fail("id不能为空");
        }
        if (!roleGroupRepoProc.exists(id)) {
            return ApiResult.ok(id);
        }

        roleGroupRepoProc.updateEnabled(id, enabled);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateName(Long id, String name) {
        Assert.notNull(id, "id不能为空");
        Assert.hasText(name, "名称不能为空");

        roleGroupRepoProc.updateName(id, name);
        return ApiResult.ok(id);
    }

    @Override
    public ApiResult<RoleGroupDetailRespVO> getDetail(Long id) {
        if (id == null) {
            return ApiResult.fail("id不能为空");
        }

        var roleGroup = roleGroupRepoProc.get(id);
        if (roleGroup == null) {
            return ApiResult.noData();
        }

        var respVO = RoleGroupConvert.INSTANCE.do2DetailRespVO(roleGroup);

        // 查询关联信息
        var relatedList = roleRelatedRepoProc.queryByRoleCode(roleGroup.getId().toString(), true);
        if (!relatedList.isEmpty()) {
            List<String> orgCodes = relatedList.stream()
                    .filter(t -> SysInnerConstant.ROLE_RELATED_TYPE_ORG.equals(t.getRelatedType()))
                    .map(SysRoleRelatedDO::getRelatedId).collect(Collectors.toList());
            if (!orgCodes.isEmpty()) {
                var nameMap = orgRepoProc.queryIdCodeNameByCodes(orgCodes).stream().collect(Collectors.toMap(IdCodeNameParam::getCode, IdCodeNameParam::getName, (t1,t2) -> t1));
                respVO.setRelatedOrgCodes(orgCodes);
                respVO.setRelatedOrgNames(orgCodes.stream().map(nameMap::get).filter(Objects::nonNull).collect(Collectors.toList()));
            }
        }
        return ApiResult.ok(respVO);
    }

    @Override
    public ApiResult<PagingVO<RoleGroupPageRespVO>> pageMng(RoleGroupPageQueryVO queryVO, BelongType.Belonger belong) {

        var pageData = roleGroupRepoProc.pageMng(queryVO, belong).map(RoleGroupConvert.INSTANCE::do2PageRespVO);
        if (pageData.isEmpty()) {
            return ApiResult.ok(pageData);
        }

        // 统计角色数量
        var ids = pageData.stream().map(RoleGroupPageRespVO::getId).collect(Collectors.toList());
        var countMap = roleRepoProc.countByGroupId(ids);
        if (!countMap.isEmpty()) {
            pageData.each(t -> {
                t.setRoleNum(countMap.getOrDefault(t.getId(), 0L));
            });
        }

        return ApiResult.ok(pageData);
    }

    private void saveRelated(String groupCode, List<String> orgCodes) {
        roleRelatedRepoProc.deletedByRoleCode(groupCode, true);

        if (CollUtil.isNotEmpty(orgCodes)) {
            var relatedList = orgCodes.stream().filter(CharSequenceUtil::isNotBlank).distinct().map(orgCode -> {
                SysRoleRelatedDO relatedDO = new SysRoleRelatedDO();
                relatedDO.setRelatedType(SysInnerConstant.ROLE_RELATED_TYPE_ORG);
                relatedDO.setRelatedId(orgCode);
                relatedDO.setRoleCode(groupCode);
                relatedDO.setRoleGroup(true);
                return relatedDO;
            }).toList();
            if (!relatedList.isEmpty()) {
                roleRelatedRepoProc.save(relatedList);
            }
        }
    }
}
