package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.constant.TaxRateType;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.TaxRateConvert;
import com.elitescloud.cloudt.system.dto.req.SysTaxRateQueryDTO;
import com.elitescloud.cloudt.system.dto.resp.SysTaxRateRespDTO;
import com.elitescloud.cloudt.system.model.vo.resp.common.CommonTaxRateRespVO;
import com.elitescloud.cloudt.system.service.TaxRateQueryService;
import com.elitescloud.cloudt.system.service.repo.TaxRateRepoProc;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 3/7/2023
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
@TenantOrgTransaction(useTenantOrg = false)
public class TaxRateQueryServiceImpl extends BaseServiceImpl implements TaxRateQueryService {
    private static final TaxRateConvert CONVERT = TaxRateConvert.INSTANCE;

    @Autowired
    private TaxRateRepoProc repoProc;

    @Override
    public ApiResult<List<CommonTaxRateRespVO>> queryList(String taxRateNo, String taxRateType) {
        var respVOList = repoProc.queryList(taxRateNo, taxRateType);
        if (respVOList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 税率类型
        var typeMap = super.udcMap(TaxRateType.OUTPUT);
        for (CommonTaxRateRespVO respVO : respVOList) {
            respVO.setTaxRateTypeName(typeMap.get(respVO.getTaxRateType()));
        }
        return ApiResult.ok(respVOList);
    }

    @Override
    public ApiResult<SysTaxRateRespDTO> getByTaxRateNo(String taxRateNo) {
        return this.getByTaxRateNo(taxRateNo, null);
    }

    @Override
    public ApiResult<SysTaxRateRespDTO> getByTaxRateNo(String taxRateNo, String taxRateType) {
        Assert.hasText(taxRateNo, "税率编号为空");
        var taxRateNoDO = repoProc.getByTaxRateNo(taxRateNo, taxRateType);
        if (taxRateNoDO == null) {
            return ApiResult.noData();
        }

        var respDTO = CONVERT.do2DTO(taxRateNoDO);
        if (CharSequenceUtil.isNotBlank(respDTO.getTaxRateType())) {
            respDTO.setTaxRateTypeName(super.udcValue(new TaxRateType(respDTO.getTaxRateType())));
        }
        return ApiResult.ok(respDTO);
    }

    @Override
    public ApiResult<List<SysTaxRateRespDTO>> queryList(SysTaxRateQueryDTO queryDTO) {
        var doList = repoProc.queryList(queryDTO);
        if (doList.isEmpty()) {
            return ApiResult.ok(Collections.emptyList());
        }

        // 税率类型
        var typeMap = super.udcMap(TaxRateType.OUTPUT);
        var dtoList = doList.stream().map(t -> {
            var dto = CONVERT.do2DTO(t);
            dto.setTaxRateTypeName(typeMap.get(dto.getTaxRateType()));
            return dto;
        }).collect(Collectors.toList());
        return ApiResult.ok(dtoList);
    }
}
