package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.ObjectUtil;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.common.param.FileByteVO;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.SysApiCacheConstant;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.cacheable.SysCacheTmplApiRpcService;
import com.elitescloud.cloudt.system.convert.TmplConvert;
import com.elitescloud.cloudt.system.dto.SysTmplDTO;
import com.elitescloud.cloudt.system.model.vo.query.extend.TmplQueryParam;
import com.elitescloud.cloudt.system.model.vo.resp.extend.TmplPageQueryRespVO;
import com.elitescloud.cloudt.system.service.ResourceByteQueryService;
import com.elitescloud.cloudt.system.service.TmplMngService;
import com.elitescloud.cloudt.system.service.TmplQueryService;
import com.elitescloud.cloudt.system.service.manager.ResourceByteMngManager;
import com.elitescloud.cloudt.system.service.repo.SysTmplRepoProc;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import java.io.ByteArrayOutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2/20/2023
 */
@Slf4j
abstract class TmplBaseServiceImpl extends BaseServiceImpl {

    @Autowired
    protected SysTmplRepoProc tmplRepoProc;
    @Autowired
    protected ResourceByteMngManager resourceByteMngManager;
    @Autowired
    private FileService<String> fileService;
    @Autowired
    private ResourceByteQueryService resourceByteQueryService;

    @Autowired
    private SysCacheTmplApiRpcService cacheTmplApiRpcService;

    /**
     * 加入缓存
     *
     * @param dto
     */
    protected void cache(SysTmplDTO dto) {
        if (dto == null) {
            return;
        }
        redisUtils.set(SysApiCacheConstant.KEY_TMPL.replace("{TMPL_CODE}", dto.getCode()), dto);
    }

    /**
     * 删掉缓存
     *
     * @param code
     */
    protected void clearCache(String code) {
        if (!StringUtils.hasText(code)) {
            return;
        }
        redisUtils.del(SysApiCacheConstant.KEY_TMPL.replace("{TMPL_CODE}", code));

        cacheTmplApiRpcService.clearCache();
    }

    public HttpEntity<StreamingResponseBody> streamByFileCode(String fileCode) {
        if (CharSequenceUtil.isBlank(fileCode)) {
            return ResponseEntity.badRequest().build();
        }
        // 先从资源文件搜索
        var res = resourceByteQueryService.download(fileCode);
        if (res.hasBody()) {
            return res;
        }

        // 调用文件服务下载
        var downloadResult = fileService.download(fileCode, null);
        if (downloadResult.hasBody()) {
            return downloadResult;
        }

        // 不存在
        var notFound = "模板文件不存在或未配置".getBytes(StandardCharsets.UTF_8);
        return ResponseEntity.ok()
                .contentType(MediaType.APPLICATION_OCTET_STREAM)
                .header(HttpHeaders.CONTENT_DISPOSITION, ContentDisposition.builder("attachment")
                        .filename("模板文件不存在或未配置.xlsx", StandardCharsets.UTF_8)
                        .build().toString())
                .contentLength(notFound.length)
                .body(outputStream -> {
                    try {
                        StreamUtils.copy(notFound, outputStream);
                    } catch (Exception e) {
                        log.error("下载资源文件异常：", e);
                    }
                });
    }

    /**
     * 获取文件
     *
     * @param fileCode
     * @return
     */
    protected byte[] getFile(String fileCode) {
        var contents = resourceByteMngManager.getResourceContent(fileCode);
        if (ArrayUtil.isNotEmpty(contents)) {
            return contents;
        }

        var resp = fileService.download(fileCode, null);
        if (resp == null || !resp.hasBody() || resp.getBody() == null) {
            return new byte[0];
        }

        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try {
            resp.getBody().writeTo(outputStream);
        } catch (Exception e) {
            log.warn("转成模板文件{}失败", fileCode, e);
            return new byte[0];
        }
        contents = outputStream.toByteArray();

        // 转存至资源表
        var fileInfo = fileService.get(fileCode).getData();
        if (fileInfo != null) {
            resourceByteMngManager.saveResourceForCompatibility(contents, fileInfo, ResourceByteMngManager.BUSINESS_TYPE_TMPL, fileCode, true);
        }

        return contents;
    }

    /**
     * 获取文件信息
     *
     * @param fileCode
     * @return
     */
    protected FileByteVO getFileVO(String fileCode) {
        return resourceByteMngManager.getFileForCompatibility(fileCode);
    }
}
