package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.excel.EasyExcelFactory;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.common.param.FileInfoVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.constant.SheetLimitStrategy;
import com.elitescloud.cloudt.core.annotation.TenantOrgTransaction;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.TmplConvert;
import com.elitescloud.cloudt.system.model.vo.query.extend.TmplQueryParam;
import com.elitescloud.cloudt.system.model.vo.resp.extend.TmplEditRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.TmplPageMngRespVO;
import com.elitescloud.cloudt.system.model.vo.save.extend.TmplSaveVO;
import com.elitescloud.cloudt.system.service.TmplMngService;
import com.elitescloud.cloudt.system.service.manager.ResourceByteMngManager;
import com.elitescloud.cloudt.system.service.model.entity.SysTmplDO;
import com.elitescloud.cloudt.system.service.repo.SysTmplRepo;
import com.elitescloud.cloudt.system.service.repo.SysTmplRepoProc;
import io.swagger.models.auth.In;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.method.annotation.StreamingResponseBody;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2021/5/31
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
@TenantOrgTransaction(useTenantOrg = false)
@Slf4j
public class TmplMngServiceImpl extends TmplBaseServiceImpl implements TmplMngService {
    private static final int UN_LIMIT = -1;
    private static final Integer SHEET_ROW_MAX = 1000000;

    @Autowired
    private SysTmplRepo tmplRepo;
    @Autowired
    private SysTmplRepoProc tmplRepoProc;
    @Autowired
    private FileService<String> fileService;
    @Autowired
    private ResourceByteMngManager resourceManager;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<FileInfoVO> uploadTmpl(MultipartFile file) {
        if (file == null || file.isEmpty()) {
            return ApiResult.fail("模板文件不存在或为空");
        }

        // 为了兼容性，先上传至原文件系统
        byte[] content = null;
        try {
            content = file.getBytes();
        } catch (IOException e) {
            log.warn("读取模板文件异常：", e);
            return ApiResult.fail("上传模板文件异常");
        }
        var uploadResult = fileService.upload(content, file.getOriginalFilename());
        if (!uploadResult.isSuccess() || uploadResult.getData() == null) {
            return ApiResult.fail(uploadResult.getMsg());
        }

        // 保存至数据库
        var resourceInfo = resourceManager.saveResourceForCompatibility(content, uploadResult.getData(), ResourceByteMngManager.BUSINESS_TYPE_TMPL, null, true);
        return ApiResult.ok(resourceInfo);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> save(TmplSaveVO saveVO) {
        // 保存前的参数检查
        checkForSave(saveVO, null);
        // 获取模板文件
        var tmplFile = getFile(saveVO.getFileCode());

        // 开始保存
        var tmplDO = TmplConvert.INSTANCE.vo2Do(saveVO);
        fillAttributeList(tmplDO, tmplFile);
        tmplRepo.save(tmplDO);

        // 取消临时标记
        resourceManager.updateTemp(tmplDO.getFileCode(), false);

        // 删除缓存
        clearCache(saveVO.getCode());

        return ApiResult.ok(tmplDO.getId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> update(Long id, TmplSaveVO saveVO) {
        // 修改前的参数检查
        checkForSave(saveVO, id);
        // 获取模板文件
        var tmplFile = getFile(saveVO.getFileCode());

        String oldFileCode = tmplRepoProc.getFileCode(id);

        // 开始保存
        var tmplDO = TmplConvert.INSTANCE.vo2Do(saveVO);
        tmplDO.setId(id);
        fillAttributeList(tmplDO, tmplFile);
        tmplRepo.save(tmplDO);

        // 取消临时标记
        resourceManager.updateTemp(tmplDO.getFileCode(), false);

        // 删除缓存
        clearCache(saveVO.getCode());

        // 删除老的
        if (StringUtils.hasText(oldFileCode) || !oldFileCode.equals(tmplDO.getFileCode())) {
            resourceManager.deleteByBusinessKey(oldFileCode);
        }

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> updateEnabled(Long id) {
        Boolean enabled = tmplRepoProc.getEnabled(id);
        tmplRepoProc.updateEnabled(id, !ObjectUtil.defaultIfNull(enabled, false));

        // 删除缓存
        String code = tmplRepoProc.getCode(id);
        clearCache(code);

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<Long> delete(Long id) {
        var tmplInfo = tmplRepoProc.get(id);
        if (tmplInfo == null) {
            return ApiResult.ok(id);
        }

        // 删除缓存
        clearCache(tmplInfo.getCode());

        // 删除模板
        tmplRepoProc.delete(id);
        resourceManager.deleteByBusinessKey(tmplInfo.getFileCode());

        return ApiResult.ok(id);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> deleteTmplFile(String fileCode) {
        resourceManager.deleteByBusinessKey(fileCode);
        CompletableFuture.runAsync(() -> {
            fileService.delete(fileCode);
        }).whenComplete((res, e) -> {
            if (e != null) {
                log.warn("删除模板文件异常", e);
            }
        });

        return ApiResult.ok(fileCode);
    }

    @Override
    public ApiResult<TmplEditRespVO> get(Long id) {
        var tmplDO = tmplRepo.findById(id).orElse(null);
        if (tmplDO == null) {
            return ApiResult.fail("数据不存在");
        }

        TmplEditRespVO respVO = TmplConvert.INSTANCE.do2Vo(tmplDO);

        if (StrUtil.isNotBlank(respVO.getFileCode())) {
            respVO.setFileInfo(resourceManager.getForCompatibility(respVO.getFileCode()));
        }

        return ApiResult.ok(respVO);
    }

    @Override
    public HttpEntity<StreamingResponseBody> download(Long id) {
        String fileCode = tmplRepoProc.getFileCode(id);
        return this.downloadByFileCode(fileCode);
    }

    @Override
    public HttpEntity<StreamingResponseBody> downloadByFileCode(String fileCode) {
        return super.streamByFileCode(fileCode);
    }

    @Override
    public HttpEntity<StreamingResponseBody> downloadByCode(String code) {
        if (CharSequenceUtil.isBlank(code)) {
            return ResponseEntity.badRequest().build();
        }
        String fileCode = tmplRepoProc.getFileCodeByCode(code);
        return super.streamByFileCode(fileCode);
    }

    @Override
    public ApiResult<PagingVO<TmplPageMngRespVO>> search(TmplQueryParam queryParam) {
        PagingVO<SysTmplDO> tmplDoPage = tmplRepoProc.pageMng(queryParam);
        if (tmplDoPage.isEmpty()) {
            return ApiResult.ok(PagingVO.empty());
        }

        var pageData = tmplDoPage.map(TmplConvert.INSTANCE::do2PageVo);
        return ApiResult.ok(pageData);
    }

    private void fillAttributeList(SysTmplDO tmplDO, byte[] file) {
        var attributes = readAttribute(file, tmplDO.getHeadRow());
        Assert.notEmpty(attributes, "头部所占行数设置有误，未读取到有效模板数据");
        Assert.isTrue(attributes.size() >= tmplDO.getFieldTypeRow(), "数据字段所在行设置有误，未读取到有效数据字段");

        tmplDO.setAttributeList(super.obj2Json(attributes));
    }

    private List<List<String>> readAttribute(byte[] file, Integer headRow) {
        List<Object> excelData = null;
        try {
            ByteArrayInputStream inputStream = new ByteArrayInputStream(file);
            excelData = EasyExcelFactory.read(inputStream)
                    .sheet(0)
                    .headRowNumber(0)
                    .doReadSync();
        } catch (Exception e) {
            throw new BusinessException("读取模板文件失败，请确认上传的模板文件格式", e);
        }
        boolean hasFieldRow = excelData != null && excelData.size() >= headRow;
        if (!hasFieldRow) {
            return Collections.emptyList();
        }

        return excelData.stream().map(data -> ((Map<Integer, String>) data)
                .entrySet().stream()
                .sorted(Comparator.comparingInt(Map.Entry::getKey))
                .map(Map.Entry::getValue)
                .collect(Collectors.toList())
        ).collect(Collectors.toList());
    }

    private void checkForSave(TmplSaveVO saveVO, Long id) {
        boolean isAdd = id == null;

        // 判断分类是否存在
        boolean exists = false;

        // 修改时，判断数据是否存在
        if (!isAdd) {
            exists = tmplRepo.existsById(id);
            Assert.isTrue(exists, "修改的数据不存在");

            // 判断编号是否修改
            String codeOld = tmplRepoProc.getCode(id);
            Assert.isTrue(CharSequenceUtil.equals(codeOld, saveVO.getCode()), "模板编号不能修改");
        }

        // 判断编号是否存在
        exists = isAdd ? tmplRepoProc.existsByCode(saveVO.getCode()) : tmplRepoProc.existsByCode(saveVO.getCode(), id);
        Assert.isFalse(exists, "模板编号已存在");

        saveVO.setExport(ObjectUtil.defaultIfNull(saveVO.getExport(), false));
        saveVO.setEnabled(ObjectUtil.defaultIfNull(saveVO.getEnabled(), true));

        if (saveVO.getFieldTypeRow() > saveVO.getHeadRow()) {
            throw new BusinessException("数据字段所在行数不能大于头部所占行数");
        }

        saveVO.setDataLimitPer(ObjectUtil.defaultIfNull(saveVO.getDataLimitPer(), UN_LIMIT));
        saveVO.setAsyncThreshold(ObjectUtil.defaultIfNull(saveVO.getAsyncThreshold(), UN_LIMIT));
        saveVO.setConcurrentLimit(ObjectUtil.defaultIfNull(saveVO.getConcurrentLimit(), UN_LIMIT));

        if (saveVO.getExport()) {
            saveVO.setExportSheetLimit(ObjectUtil.defaultIfNull(saveVO.getExportSheetLimit(), SHEET_ROW_MAX));
            saveVO.setExportSheetStrategy(ObjectUtil.defaultIfNull(saveVO.getExportSheetStrategy(), SheetLimitStrategy.NEW_SHEET));
            saveVO.setSheetNos(null);
        } else {
            saveVO.setExportSheetLimit(null);
            saveVO.setExportSheetStrategy(null);
            saveVO.setSheetNos(this.normalizeSheetNos(saveVO.getSheetNos()));
        }
    }

    private String normalizeSheetNos(String sheetNos) {
        if (CharSequenceUtil.isBlank(sheetNos)) {
            return null;
        }
        var sheetNoArray = sheetNos.replace("，", ",").split(",");
        List<String> sheetNoList = new ArrayList<>(sheetNoArray.length);
        Integer sheetNo = null;
        for (String s : sheetNoArray) {
            try {
                sheetNo = Integer.parseInt(s.trim());
            } catch (NumberFormatException e) {
                throw new IllegalArgumentException("sheet页格式设置有误，必须为大于0的整数");
            }
            if (sheetNo < 1) {
                throw new IllegalArgumentException("sheet页格式设置有误，必须为大于0的整数");
            }
            if (sheetNoList.contains(sheetNo.toString())) {
                continue;
            }
            sheetNoList.add(sheetNo.toString());
        }

        return String.join(",", sheetNoList);
    }
}
