package com.elitescloud.cloudt.system.service.impl;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.util.ClassUtil;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.service.UserOuterappQueryService;
import com.elitescloud.cloudt.system.service.common.service.UserOuterAppProvider;
import com.elitescloud.cloudt.system.service.dto.SysUserOuterAppDTO;
import com.elitescloud.cloudt.system.service.model.entity.SysUserOuterappDO;
import com.elitescloud.cloudt.system.service.repo.SysUserOuterappRepoProc;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.servlet.http.HttpServletRequest;
import java.time.LocalDateTime;
import java.util.concurrent.CompletableFuture;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2025/2/12
 */
@Service
@TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
public class UserOuterappQueryServiceImpl implements UserOuterappQueryService {
    private static final Logger logger = LoggerFactory.getLogger(UserOuterappQueryServiceImpl.class);

    @Autowired
    private SysUserOuterappRepoProc repoProc;
    @Autowired
    private TaskExecutor taskExecutor;
    @Autowired
    private ObjectProvider<UserOuterAppProvider> userOuterAppProviders;

    @Override
    public ApiResult<String> getOpenId(HttpServletRequest request, String outerAppId) {
        Assert.hasText(outerAppId, "外部应用标识为空");

        var userId = SecurityContextUtil.currentUserIfUnauthorizedThrow().getUserId();

        // 查询是否已绑定
        var openId = repoProc.getOpenIdByUserIdAndOuterAppId(userId, outerAppId);
        if (CharSequenceUtil.isNotBlank(openId)) {
            return ApiResult.ok(openId);
        }

        // 获取openId
        SysUserOuterAppDTO outerAppDTO = queryOuterAppInfo(request, outerAppId);
        if (outerAppDTO == null) {
            return ApiResult.fail("获取openId失败");
        }

        // 保存openId
        CompletableFuture.runAsync(() -> {
                    SysUserOuterappDO userOuterappDO = new SysUserOuterappDO();
                    userOuterappDO.setUserId(userId);
                    userOuterappDO.setOuterAppId(outerAppId);
                    userOuterappDO.setOpenId(outerAppDTO.getOpenId());
                    userOuterappDO.setUnionId(outerAppDTO.getUnionId());
                    userOuterappDO.setBindTime(LocalDateTime.now());
                    userOuterappDO.setTenantCode(outerAppDTO.getTenantCode());

                    repoProc.save(userOuterappDO);
                })
                .whenComplete((r, e) -> {
                    if (e != null) {
                        logger.error("保存openId失败：{}，{}，{}", userId, outerAppDTO.getOpenId(), outerAppDTO.getUnionId(), e);
                    }
                });

        return ApiResult.ok(outerAppDTO.getOpenId());
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<String> unbind(String outerAppId) {
        Assert.hasText(outerAppId, "外部应用标识为空");

        var userId = SecurityContextUtil.currentUserIfUnauthorizedThrow().getUserId();
        var openId = repoProc.getOpenIdByUserIdAndOuterAppId(userId, outerAppId);
        // 未绑定，则直接返回
        if (CharSequenceUtil.isBlank(openId)) {
            return ApiResult.ok(openId);
        }

        repoProc.deleteByAppIdAndUserId(userId, outerAppId);
        return ApiResult.ok(openId);
    }

    private SysUserOuterAppDTO queryOuterAppInfo(HttpServletRequest request, String outerAppId) {
        for (UserOuterAppProvider userOuterAppProvider : userOuterAppProviders) {
            SysUserOuterAppDTO outerAppDTO = userOuterAppProvider.get(outerAppId, request);
            if (outerAppDTO != null) {
                logger.info("匹配到应用：{}，{}", outerAppId, ClassUtil.getTargetClass(userOuterAppProvider).getName());
                if (CharSequenceUtil.isAllBlank(outerAppDTO.getOpenId(), outerAppDTO.getUnionId())) {
                    logger.warn("获取openId失败：{}", outerAppId);
                    return null;
                }
                return outerAppDTO;
            }
        }
        return null;
    }
}
