package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.RandomUtil;
import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitescloud.boot.common.param.FileByteVO;
import com.elitescloud.boot.common.param.FileInfoVO;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.util.DatetimeUtil;
import com.elitescloud.boot.util.FileUtil;
import com.elitescloud.boot.util.StrUtil;
import com.elitescloud.cloudt.system.model.entity.SysResourceByteDO;
import com.elitescloud.cloudt.system.service.repo.ResourceByteRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.io.ByteArrayOutputStream;
import java.io.IOException;

/**
 * 资源文件.
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/25
 */
@Slf4j
@Component
public class ResourceByteMngManager {
    /**
     * 导入导出模板
     */
    public static final String BUSINESS_TYPE_TMPL = "tmpl";

    @Autowired
    private ResourceByteRepoProc repoProc;
    @Autowired
    private FileService<String> fileService;

    /**
     * 保存资源文件
     *
     * @param multipartFile
     * @param businessType
     * @param businessKey
     * @param temp
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public FileInfoVO saveResource(@NotNull MultipartFile multipartFile, @NotBlank String businessType, String businessKey, boolean temp) {
        if (multipartFile == null || multipartFile.isEmpty()) {
            throw new BusinessException("资源文件为空");
        }
        Assert.notBlank(businessType, "业务类型为空");

        if (CharSequenceUtil.isBlank(businessKey)) {
            businessKey = this.generateBusinessKey();
        }

        SysResourceByteDO byteDO = new SysResourceByteDO();
        byteDO.setResourceType(businessType);
        byteDO.setResourceKey(businessKey);
        try {
            byteDO.setResource(multipartFile.getBytes());
        } catch (IOException e) {
            throw new IllegalStateException("保存资源失败", e);
        }
        byteDO.setMimeType(multipartFile.getContentType());
        byteDO.setSuffix(FileUtil.getSuffix(multipartFile.getOriginalFilename()));
        byteDO.setResourceName(multipartFile.getOriginalFilename());
        byteDO.setShowName(multipartFile.getOriginalFilename());
        byteDO.setSize(multipartFile.getSize());
        byteDO.setTemp(temp);

        repoProc.save(byteDO);

        return this.convertVO(byteDO);
    }

    /**
     * 兼容性保存资源文件
     *
     * @param fileCode
     * @param businessType
     * @param businessKey
     * @param temp
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public FileInfoVO saveResourceForCompatibility(@NotBlank String fileCode,
                                                   @NotBlank String businessType, String businessKey, boolean temp) {
        Assert.notBlank(fileCode, "文件编码为空");
        var fileInfo = fileService.get(fileCode).getData();
        Assert.notNull(fileInfo, "文件不存在");

        var resp = fileService.download(fileCode, null);
        if (resp == null || !resp.hasBody() || resp.getBody() == null) {
            throw new BusinessException("文件信息为空");
        }

        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try {
            resp.getBody().writeTo(outputStream);
        } catch (Exception e) {
            log.warn("转成模板文件{}失败", fileCode, e);
            throw new BusinessException("存储文件失败", e);
        }
        var contents = outputStream.toByteArray();

        return this.saveResourceForCompatibility(contents, fileInfo, businessType, businessKey, temp);
    }

    /**
     * 兼容性保存资源文件
     *
     * @param resource
     * @param fileObjRespVO
     * @param businessType
     * @param businessKey
     * @param temp
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public FileInfoVO saveResourceForCompatibility(@NotBlank byte[] resource, @NotNull FileObjRespVO<?> fileObjRespVO,
                                                   @NotBlank String businessType, String businessKey, boolean temp) {
        if (ArrayUtil.isEmpty(resource)) {
            throw new BusinessException("资源文件为空");
        }
        Assert.notNull(fileObjRespVO, "资源文件信息为空");
        Assert.notBlank(businessType, "业务类型为空");

        if (CharSequenceUtil.isBlank(businessKey)) {
            businessKey = fileObjRespVO.getFileCode();
        }

        String suffix = StrUtil.blankToDefault(fileObjRespVO.getSuffix(), () -> FileUtil.getSuffix(fileObjRespVO.getOriginalName()));

        SysResourceByteDO byteDO = new SysResourceByteDO();
        byteDO.setResourceType(businessType);
        byteDO.setResourceKey(businessKey);
        byteDO.setResource(resource);
        byteDO.setMimeType(StrUtil.blankToDefault(fileObjRespVO.getMimeType(), () -> FileUtil.convertMimeTypeByFilename(fileObjRespVO.getOriginalName())));
        byteDO.setSuffix(suffix);
        byteDO.setResourceName(fileObjRespVO.getOriginalName());
        byteDO.setShowName(fileObjRespVO.getOriginalName());
        byteDO.setSize(fileObjRespVO.getFileSize());
        byteDO.setTemp(temp);

        repoProc.save(byteDO);
        return this.convertVO(byteDO);
    }

    /**
     * 更新临时文件标识
     *
     * @param businessKey
     * @param temp
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateTemp(@NotBlank String businessKey, boolean temp) {
        Assert.notBlank(businessKey, "资源标识为空");
        repoProc.updateTemp(businessKey, temp);
    }

    /**
     * 根据业务标识删除
     *
     * @param id
     */
    @Transactional(rollbackFor = Exception.class)
    public void delete(@NotNull Long id) {
        Assert.notNull(id, "ID为空");

        repoProc.delete(id);
    }

    /**
     * 根据业务标识删除
     *
     * @param businessType
     */
    @Transactional(rollbackFor = Exception.class)
    public void deleteByBusinessKey(@NotBlank String businessType) {
        Assert.notBlank(businessType, "业务标识为空");

        repoProc.deleteByResourceKey(businessType);
    }

    /**
     * 获取资源文件内容
     *
     * @param businessKey 文件标识
     * @return
     */
    public byte[] getResourceContent(@NotBlank String businessKey) {
        Assert.notBlank(businessKey, "资源标识为空");
        return repoProc.getResourceByResourceKey(businessKey);
    }

    /**
     * 获取文件信息
     *
     * @param businessKey
     * @return
     */
    public FileInfoVO get(@NotBlank String businessKey) {
        Assert.notBlank(businessKey, "业务标识为空");

        var resourceInfo = repoProc.getByResourceKey(businessKey);
        if (resourceInfo == null) {
            return null;
        }
        return this.convertVO(resourceInfo);
    }

    /**
     * 兼容性获取文件信息
     *
     * @param fileCode
     * @return
     */
    public FileInfoVO getForCompatibility(@NotBlank String fileCode) {
        Assert.notBlank(fileCode, "文件编码为空");
        var fileInfo = this.get(fileCode);
        if (fileInfo != null) {
            return fileInfo;
        }

        var objInfo = fileService.get(fileCode).getData();
        if (objInfo == null) {
            return null;
        }
        fileInfo = new FileInfoVO();
        fileInfo.setFileCode(fileCode);
        fileInfo.setOriginalName(objInfo.getOriginalName());
        fileInfo.setFileSize(objInfo.getFileSize());
        fileInfo.setSuffix(objInfo.getSuffix());
        fileInfo.setMimeType(objInfo.getMimeType());
        fileInfo.setUploadTime(objInfo.getUploadTime());
        fileInfo.setUrlPreview(objInfo.getUrlPreview());
        fileInfo.setUrlDownload(objInfo.getUrlDownload());
        fileInfo.setUrlDelete(objInfo.getUrlDelete());
        fileInfo.setUrl(objInfo.getUrl());

        return fileInfo;
    }

    /**
     * 获取文件信息
     *
     * @param fileCode
     * @return
     */
    public FileByteVO getFileForCompatibility(@NotBlank String fileCode) {
        Assert.notBlank(fileCode, "文件编码为空");
        var resourceInfo = repoProc.getByResourceKey(fileCode);
        if (resourceInfo != null) {
            FileByteVO byteVO = new FileByteVO();
            byteVO.setBytes(resourceInfo.getResource());
            byteVO.setOriginalName(resourceInfo.getResourceName());
            byteVO.setFileSize(resourceInfo.getSize());
            byteVO.setSuffix(resourceInfo.getSuffix());
            byteVO.setMimeType(resourceInfo.getMimeType());
            return byteVO;
        }

        // 从文件服务获取
        var objInfo = fileService.get(fileCode).getData();
        if (objInfo == null) {
            return null;
        }
        var resp = fileService.download(fileCode, null);
        if (resp == null || !resp.hasBody() || resp.getBody() == null) {
            return null;
        }
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try {
            resp.getBody().writeTo(outputStream);
        } catch (Exception e) {
            log.warn("转成模板文件{}失败", fileCode, e);
            throw new BusinessException("存储文件失败", e);
        }
        var contents = outputStream.toByteArray();
        FileByteVO byteVO = new FileByteVO();
        byteVO.setBytes(contents);
        byteVO.setOriginalName(objInfo.getOriginalName());
        byteVO.setFileSize(objInfo.getFileSize());
        byteVO.setSuffix(objInfo.getSuffix());
        byteVO.setMimeType(objInfo.getMimeType());
        return byteVO;
    }

    private FileInfoVO convertVO(SysResourceByteDO byteDO) {
        FileInfoVO vo = new FileInfoVO();
        vo.setFileCode(byteDO.getResourceKey());
        vo.setOriginalName(byteDO.getResourceName());
        vo.setFileSize(byteDO.getSize());
        vo.setSuffix(byteDO.getSuffix());
        vo.setMimeType(byteDO.getMimeType());
        vo.setUploadTime(byteDO.getCreateTime());
        vo.setUrlPreview(null);
        vo.setUrlDownload(null);
        vo.setUrlDelete(null);
        vo.setUrl(null);

        return vo;
    }

    private String generateBusinessKey() {
        return DatetimeUtil.currentTimeLong() + RandomUtil.randomNumbers(6);
    }
}
