package com.elitescloud.cloudt.system.service.manager;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.auth.client.config.AuthorizationProperties;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.security.common.InnerRole;
import com.elitescloud.cloudt.constant.SysRoleBusiness;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.system.convert.RoleConvert;
import com.elitescloud.cloudt.system.model.vo.resp.role.GroupRoleRespVO;
import com.elitescloud.cloudt.system.service.callback.RoleChangedCallback;
import com.elitescloud.cloudt.system.service.common.constant.BelongType;
import com.elitescloud.cloudt.system.service.model.bo.SysRoleSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleDO;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleGroupDO;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleRelatedDO;
import com.elitescloud.cloudt.system.service.repo.*;
import com.google.common.base.Functions;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 角色管理.
 *
 * @author Kaiser（wang shao）
 * 2022/10/13
 */
@Component
@TenantTransaction(isolateType = TenantIsolateType.TENANT)
public class RoleMngManager {
    private static final RoleConvert CONVERT = RoleConvert.INSTANCE;

    @Autowired
    private RoleRepoProc roleRepoProc;
    @Autowired
    private RolePermissionRepoProc rolePermissionRepoProc;
    @Autowired
    private RoleFieldPermissionRepoProc roleFieldPermissionRepoProc;
    @Autowired
    private RoleDataPermissionRepoProc roleDataPermissionRepoProc;
    @Autowired
    private RoleGroupRepoProc roleGroupRepoProc;
    @Autowired
    private RoleRelatedRepoProc roleRelatedRepoProc;
    @Autowired
    private UserRoleRepoProc userRoleRepoProc;
    @Autowired
    private SysDprRoleApiRowRuleRepoProc apiRowRuleRepoProc;
    @Autowired
    private SysDpcrApiFieldsRepoProc apiFieldsRepoProc;

    @Autowired
    private ObjectProvider<RoleChangedCallback> roleChangedCallbacks;
    @Autowired
    private AuthorizationProperties authorizationProperties;

    /**
     * 保存角色信息
     *
     * @param saveBO 角色信息
     * @return 角色信息
     */
    public SysRoleDO upsert(@NotNull SysRoleSaveBO saveBO) {
        // 检查保存参数
        boolean isAdd = saveBO.getId() == null;
        SysRoleDO roleDO = isAdd ? checkForInsert(saveBO) : checkForUpdate(saveBO);

        SysRoleDO roleOld = isAdd ? null : roleRepoProc.get(saveBO.getId());

        // 保存角色信息
        roleRepoProc.save(roleDO);

        // 保存后的回调
        roleChangedCallbacks.forEach(t -> t.onUpsert(isAdd, saveBO, roleDO));
        if (!isAdd && !Objects.equals(roleDO.getEnabled(), roleOld.getEnabled())) {
            roleChangedCallbacks.forEach(t -> t.onEnabled(saveBO.getId(), roleDO.getEnabled()));
        }

        return roleDO;
    }

    /**
     * 更新角色的启用状态
     *
     * @param roleId  角色ID
     * @param enabled 启用状态
     */
    public void updateEnabled(@NotNull Long roleId, Boolean enabled) {
        if (enabled == null) {
            enabled = true;
        }
        roleRepoProc.updateEnabled(roleId, enabled);

        Boolean finalEnabled = enabled;
        roleChangedCallbacks.forEach(t -> t.onEnabled(roleId, finalEnabled));
    }

    /**
     * 修改角色名称
     *
     * @param roleId 角色ID
     * @param name   角色名称
     */
    public void updateName(long roleId, @NotBlank String name) {
        Assert.hasText(name, "角色名称为空");

        roleRepoProc.updateName(roleId, name);
    }

    /**
     * 删除角色数据
     *
     * @param roleId 角色ID
     */
    public void delete(@NotNull Long roleId) {
        var role = roleRepoProc.get(roleId);
        if (role == null) {
            return;
        }

        // 判断角色下是否还有账号
        if (userRoleRepoProc.existsRole(roleId)) {
            throw new BusinessException("请先移除角色下的成员");
        }

        // 删除角色
        roleRepoProc.delete(roleId);
        // 删除角色与权限的关联
        rolePermissionRepoProc.deleteByRole(roleId);
        // 删除数据权限
        apiRowRuleRepoProc.deleteByRole(roleId);
        apiFieldsRepoProc.deleteByRole(roleId);
        roleFieldPermissionRepoProc.deleteByRole(role.getCode());
        roleDataPermissionRepoProc.deleteByRole(role.getCode());

        // 删除后的回调
        roleChangedCallbacks.forEach(t -> t.onDelete(role));
    }

    /**
     * 获取角色组和角色
     *
     * @param showDisabled 是否显示禁用的
     * @return 角色组和角色
     */
    public List<GroupRoleRespVO> listGroupRole(boolean showDisabled, BelongType.Belonger belonger) {
        Boolean enabled = !showDisabled ? true : null;
        // 先查询角色组
        var groupList = roleGroupRepoProc.listIdCodeName(belonger, enabled).stream()
                .map(t -> {
                    GroupRoleRespVO respVO = new GroupRoleRespVO();
                    respVO.setGroupId(t.getId());
                    respVO.setGroupCode(t.getCode());
                    respVO.setGroupName(t.getName());
                    respVO.setRoles(new ArrayList<>(32));

                    return respVO;
                }).collect(Collectors.toList());
        var groupMap = groupList.stream().collect(Collectors.toMap(GroupRoleRespVO::getGroupId, Functions.identity(), (t1, t2) -> t1));
        // 默认角色组
        var groupDefault = groupMap.computeIfAbsent(SysRoleDO.DEFAULT_GROUP_ID, t -> {
            GroupRoleRespVO respVO = new GroupRoleRespVO();
            respVO.setGroupId(t);
            respVO.setGroupName(SysRoleDO.DEFAULT_GROUP_NAME);
            respVO.setRoles(new ArrayList<>(32));

            return respVO;
        });

        // 查询角色
        var roleList = roleRepoProc.queryIdCodeNames(enabled, null, belonger, null);
        if (roleList.isEmpty()) {
            return groupList;
        }

        // 向角色组放置角色
        GroupRoleRespVO.Role roleRespVO = null;
        for (var role : roleList) {
            roleRespVO = new GroupRoleRespVO.Role();
            roleRespVO.setRoleId(role.getId());
            roleRespVO.setRoleCode(role.getCode());
            roleRespVO.setRoleName(role.getName());

            if (role.getGroupId() == null || !groupMap.containsKey(role.getGroupId())) {
                groupDefault.getRoles().add(roleRespVO);
                continue;
            }
            groupMap.get(role.getGroupId()).getRoles().add(roleRespVO);
        }
        if (CharSequenceUtil.isBlank(groupDefault.getGroupCode())) {
            groupList.add(0, groupDefault);
        }

        return groupList;
    }

    /**
     * 根据关联对象获取角色组和角色
     *
     * @param relatedType
     * @param relationId
     * @param belonger
     * @return
     */
    public List<GroupRoleRespVO> listGroupRoleByRelated(@NotBlank String relatedType, @NotBlank String relationId, BelongType.Belonger belonger) {
        // 查询关联对象
        var relatedList = roleRelatedRepoProc.queryByRelated(relationId, relatedType);
        if (relatedList.isEmpty()) {
            return Collections.emptyList();
        }

        Set<Long> roleGroupCodes = new HashSet<>(relatedList.size());
        for (SysRoleRelatedDO related : relatedList) {
            if (related.getRoleGroup()) {
                roleGroupCodes.add(Long.parseLong(related.getRoleCode()));
            }
        }
        if (roleGroupCodes.isEmpty()) {
            return Collections.emptyList();
        }
        Boolean enabled = true;

        var groupList = roleGroupRepoProc.listIdCodeName(roleGroupCodes, belonger, enabled).stream()
                .map(t -> {
                    GroupRoleRespVO respVO = new GroupRoleRespVO();
                    respVO.setGroupId(t.getId());
                    respVO.setGroupCode(t.getCode());
                    respVO.setGroupName(t.getName());
                    respVO.setRoles(new ArrayList<>(32));

                    return respVO;
                }).toList();
        if (groupList.isEmpty()) {
            return Collections.emptyList();
        }
        var groupMap = groupList.stream().collect(Collectors.toMap(GroupRoleRespVO::getGroupId, Functions.identity(), (t1, t2) -> t1));
        var roleList = roleRepoProc.queryIdCodeNames(enabled, groupMap.keySet(), belonger, null);
        // 向角色组放置角色
        GroupRoleRespVO.Role roleRespVO = null;
        for (var role : roleList) {
            roleRespVO = new GroupRoleRespVO.Role();
            roleRespVO.setRoleId(role.getId());
            roleRespVO.setRoleCode(role.getCode());
            roleRespVO.setRoleName(role.getName());

            if (role.getGroupId() == null || !groupMap.containsKey(role.getGroupId())) {
                continue;
            }
            groupMap.get(role.getGroupId()).getRoles().add(roleRespVO);
        }

        return groupList;
    }

    private SysRoleDO checkForInsert(SysRoleSaveBO saveBO) {
        // 角色编码
        Assert.hasText(saveBO.getCode(), "角色编码为空");
        String codePrefix = rolePrefix();
        Assert.isTrue(!saveBO.getCode().startsWith(codePrefix), "角色编码的前缀不能是" + codePrefix);
        boolean exists = roleRepoProc.existsCode(saveBO.getCode());
        Assert.isTrue(!exists, "角色编码已存在");
        if (InnerRole.valueOf(saveBO.getCode()) != null) {
            throw new IllegalArgumentException(saveBO.getCode() + "是系统内置角色，不可使用");
        }

        if (saveBO.getGroupId() == null) {
            saveBO.setGroupId(SysRoleDO.DEFAULT_GROUP_ID);
        }

        if (CharSequenceUtil.isNotBlank(saveBO.getParentCode())) {
            exists = roleRepoProc.existsCode(saveBO.getParentCode());
            Assert.isTrue(exists, "上级角色不存在");
        } else {
            saveBO.setParentCode("");
        }

        // 是否启用
        if (saveBO.getEnabled() == null) {
            saveBO.setEnabled(true);
        }

        // 角色类型
        Assert.hasText(saveBO.getType(), "未知角色类型");

        // 默认系统角色
        saveBO.setBusinessKey(CharSequenceUtil.blankToDefault(saveBO.getBusinessKey(), SysRoleBusiness.SYS.getValue()));

        return CONVERT.saveBo2DO(saveBO);
    }

    private SysRoleDO checkForUpdate(SysRoleSaveBO saveBO) {
        SysRoleDO roleDO = roleRepoProc.get(saveBO.getId());
        Assert.notNull(roleDO, "角色不存在");

        // 角色编码
        Assert.isTrue(roleDO.getCode().equals(saveBO.getCode()), "角色编码不可修改");
        if (InnerRole.valueOf(saveBO.getCode()) != null) {
            throw new IllegalArgumentException(saveBO.getCode() + "是系统内置角色，不可使用");
        }

        if (saveBO.getGroupId() == null) {
            saveBO.setGroupId(SysRoleDO.DEFAULT_GROUP_ID);
        }

        if (CharSequenceUtil.isNotBlank(saveBO.getParentCode())) {
            if (!saveBO.getParentCode().equals(roleDO.getParentCode())) {
                var exists = roleRepoProc.existsCode(saveBO.getParentCode());
                Assert.isTrue(exists, "上级角色不存在");
            }
        } else {
            saveBO.setParentCode("");
        }

        // 是否启用
        if (saveBO.getEnabled() == null) {
            saveBO.setEnabled(roleDO.getEnabled());
        }

        // 角色类型
        if (!StringUtils.hasText(saveBO.getType())) {
            saveBO.setType(roleDO.getType());
        }
        if (!StringUtils.hasText(saveBO.getTypeId())) {
            saveBO.setTypeId(roleDO.getTypeId());
        }

        // 默认系统角色
        saveBO.setBusinessKey(CharSequenceUtil.blankToDefault(saveBO.getBusinessKey(), SysRoleBusiness.SYS.getValue()));

        CONVERT.copySaveBO(saveBO, roleDO);
        return roleDO;
    }

    private String rolePrefix() {
        return CharSequenceUtil.blankToDefault(authorizationProperties.getRolePrefix(), "ROLE_");
    }
}
