package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.query.sys.OpenApiPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.sys.OpenApiPageRespVO;
import com.elitescloud.cloudt.system.service.common.constant.OpenApiStatusEnum;
import com.elitescloud.cloudt.system.service.model.entity.QSysOpenApiInfoDO;
import com.elitescloud.cloudt.system.service.model.entity.QSysOpenApiOperationDO;
import com.elitescloud.cloudt.system.service.model.entity.SysOpenApiInfoDO;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/3/4
 */
@Slf4j
@Repository
public class OpenApiInfoRepoProc extends BaseRepoProc<SysOpenApiInfoDO> {
    private static final QSysOpenApiInfoDO QDO = QSysOpenApiInfoDO.sysOpenApiInfoDO;
    private static final QSysOpenApiOperationDO QDO_OPERATION = QSysOpenApiOperationDO.sysOpenApiOperationDO;

    public OpenApiInfoRepoProc() {
        super(QDO);
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(long id, OpenApiStatusEnum status, String failReason) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.state, status.name())
                .set(QDO.failReason, failReason)
                .where(QDO.id.eq(id))
                .execute();
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateStartUpdate(long id) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.state, OpenApiStatusEnum.UPDATING.name())
                .setNull(QDO.failReason)
                .set(QDO.startUpdateTime, LocalDateTime.now())
                .where(QDO.id.eq(id))
                .execute();
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateEndUpdate(long id, OpenApiStatusEnum status, String failReason) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.state, status.name())
                .set(QDO.failReason, failReason)
                .set(QDO.endUpdateTime, LocalDateTime.now())
                .where(QDO.id.eq(id))
                .execute();
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateBusinessObjectStatus(long id, OpenApiStatusEnum status, String failReason) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.businessObjectState, status.name())
                .set(QDO.businessObjectFailReason, failReason)
                .where(QDO.id.eq(id))
                .execute();
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateBusinessObjectStart(long id) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.businessObjectState, OpenApiStatusEnum.UPDATING.name())
                .setNull(QDO.businessObjectFailReason)
                .set(QDO.businessObjectStartUpdateTime, LocalDateTime.now())
                .where(QDO.id.eq(id))
                .execute();
    }

    @Transactional(rollbackFor = Exception.class)
    public void updateBusinessObjectEnd(long id, OpenApiStatusEnum status, String failReason) {
        super.jpaQueryFactory.update(QDO)
                .set(QDO.businessObjectState, status.name())
                .set(QDO.businessObjectFailReason, failReason)
                .set(QDO.businessObjectEndUpdateTime, LocalDateTime.now())
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新资源ID
     *
     * @param id
     * @param resourceId
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateResourceId(long id, long resourceId) {
        super.updateValue(QDO.resourceId, resourceId, id);
    }

    /**
     * 更新资源ID
     *
     * @param id
     * @param resourceId
     */
    @Transactional(rollbackFor = Exception.class)
    public void updateBusinessResourceId(long id, long resourceId) {
        super.updateValue(QDO.businessResourceId, resourceId, id);
    }

    /**
     * 获取记录ID
     *
     * @param project
     * @param env
     * @param appCode
     * @return
     */
    public Long getId(String project, String env, String appCode) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.appCode, appCode)
//                .andEq(QDO.env, env)
//                .andEq(QDO.project, project)
                .build();
        if (predicate == null) {
            return null;
        }

        return super.getIdByPredicate(predicate);
    }

    /**
     * 获取服务编码
     *
     * @param id
     * @return
     */
    public String getAppCode(long id) {
        return super.getValue(QDO.appCode, id);
    }

    /**
     * 分页查询管理
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysOpenApiInfoDO> pageMng(OpenApiPageQueryVO queryVO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.appCode, queryVO.getAppCode())
                .andEq(QDO.project, queryVO.getProject())
                .andEq(QDO.env, queryVO.getEnv())
                .build();

        return super.queryByPage(predicate, queryVO.getPageRequest(), QDO.modifyTime.desc());
    }

    public OpenApiStatusEnum getState(long id) {
        var state = super.getValue(QDO.state, id);
        if (CharSequenceUtil.isBlank(state)) {
            return null;
        }
        return OpenApiStatusEnum.valueOf(state);
    }

    public OpenApiStatusEnum getBusinessObjectState(long id) {
        var state = super.getValue(QDO.businessObjectState, id);
        if (CharSequenceUtil.isBlank(state)) {
            return null;
        }
        return OpenApiStatusEnum.valueOf(state);
    }

    public Long getResourceId(long id) {
        return super.getValue(QDO.resourceId, id);
    }

    private QBean<OpenApiPageRespVO> qBeanPageRespVO() {
        return Projections.bean(OpenApiPageRespVO.class,
                QDO.id,
                QDO.openApiCode,
                QDO.project,
                QDO.projectName,
                QDO.appCode,
                QDO.appName,
                QDO.env,
                QDO.envName,
                QDO.title,
                QDO.description,
                QDO.version,
                QDO.buildTime,
                QDO.projectVersion,
                QDO.cloudtBootVersion,
                QDO.state,
                QDO.failReason,
                QDO.modifyTime,
                QDO.createTime
        );
    }
}
