package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.common.param.IdCodeNameParam;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.service.common.constant.BelongType;
import com.elitescloud.cloudt.system.model.vo.query.role.RoleGroupPageQueryVO;
import com.elitescloud.cloudt.system.service.model.entity.QSysRoleGroupDO;
import com.elitescloud.cloudt.system.service.model.entity.SysRoleGroupDO;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.core.types.dsl.StringExpression;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.util.Collection;
import java.util.List;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/11/24
 */
@Repository
public class RoleGroupRepoProc extends BaseRepoProc<SysRoleGroupDO> {
    private static final QSysRoleGroupDO QDO = QSysRoleGroupDO.sysRoleGroupDO;

    public RoleGroupRepoProc() {
        super(QDO);
    }

    /**
     * 更新启用标识
     *
     * @param id
     * @param enabled
     */
    public void updateEnabled(long id, boolean enabled) {
        super.updateValue(QDO.enabled, enabled, id);
    }

    public void updateName(long id, String name) {
        super.updateValue(QDO.groupName, name, id);
    }

    /**
     * 是否编码已存在
     *
     * @param groupCode
     * @return
     */
    public boolean existsCode(@NotBlank String groupCode, @NotNull BelongType.Belonger belong) {
        var predicate = QDO.groupCode.eq(groupCode).and(QDO.type.eq(belong.getBelongType().getValue())).and(QDO.typeId.eq(belong.getBelongId()));
        return super.exists(predicate);
    }

    /**
     * 获取分组名称
     *
     * @param id
     * @return
     */
    public String getName(long id) {
        return super.getValue(QDO.groupName, id);
    }

    /**
     * 获取分组编码
     *
     * @param id
     * @return
     */
    public String getCode(long id) {
        return super.getValue(QDO.groupCode, id);
    }

    /**
     * 分页查询管理
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysRoleGroupDO> pageMng(RoleGroupPageQueryVO queryVO, BelongType.Belonger belong) {
        var predicate = PredicateBuilder.builder()
                .andLike(QDO.groupCode, queryVO.getGroupCode())
                .andLike(QDO.groupName, queryVO.getGroupName())
                .andEq(QDO.enabled, queryVO.getEnabled())
                .andEq(QDO.type, belong.getBelongType().getValue())
                .andEq(QDO.typeId, belong.getBelongId())
                .andLike(new StringExpression[]{QDO.groupCode, QDO.groupName}, queryVO.getKeyword())
                .build();

        return super.queryByPage(predicate, queryVO.getPageRequest(), QDO.createTime.desc());
    }

    /**
     * 查询角色组列表
     *
     * @return
     */
    public List<IdCodeNameParam> listIdCodeName(BelongType.Belonger belonger, Boolean enabled) {
        Predicate predicate = enabled == null ? null : QDO.enabled.eq(enabled);
        if (belonger != null) {
            predicate = super.andPredicate(predicate, QDO.type.eq(belonger.getBelongType().getValue()).and(QDO.typeId.eq(belonger.getBelongId())));
        }
        return super.jpaQueryFactory.select(qBeanIdCodeName())
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.createTime.desc())
                .fetch();
    }

    /**
     * 查询角色组列表
     *
     * @return
     */
    public List<IdCodeNameParam> listIdCodeName(Collection<Long> ids, BelongType.Belonger belonger, Boolean enabled) {
        Predicate predicate = enabled == null ? null : QDO.enabled.eq(enabled);
        if (belonger != null) {
            predicate = super.andPredicate(predicate, QDO.type.eq(belonger.getBelongType().getValue()).and(QDO.typeId.eq(belonger.getBelongId())));
        }
        if (CollUtil.isNotEmpty(ids)) {
            predicate = super.andPredicate(predicate, QDO.id.in(ids));
        }
        return super.jpaQueryFactory.select(qBeanIdCodeName())
                .from(QDO)
                .where(predicate)
                .orderBy(QDO.createTime.desc())
                .fetch();
    }

    /**
     * 根据ID查询
     *
     * @param ids
     * @return
     */
    public List<IdCodeNameParam> queryByList(Collection<Long> ids) {
        return super.jpaQueryFactory.select(qBeanIdCodeName())
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch();
    }

    private QBean<IdCodeNameParam> qBeanIdCodeName() {
        return Projections.bean(IdCodeNameParam.class, QDO.id, QDO.groupCode.as("code"), QDO.groupName.as("name"));
    }
}
