package com.elitescloud.cloudt.system.service.repo;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.jpa.common.BaseRepoProc;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.dto.req.ThirdApiLogQueryDTO;
import com.elitescloud.cloudt.system.model.entity.QSysThirdApiLogDO;
import com.elitescloud.cloudt.system.model.entity.SysThirdApiLogDO;
import com.elitescloud.cloudt.system.model.vo.query.extend.ThirdApiLogQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.extend.ThirdApiRetryLogQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.ThirdApiLogAggPageRespVO;
import com.querydsl.core.types.Projections;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Repository;

import javax.validation.constraints.NotNull;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/4
 */
@Repository
public class ThirdApiLogRepoProc extends BaseRepoProc<SysThirdApiLogDO> {
    private static final QSysThirdApiLogDO QDO = QSysThirdApiLogDO.sysThirdApiLogDO;

    public ThirdApiLogRepoProc() {
        super(QDO);
    }

    /**
     * 更新重试记录ID
     *
     * @param id
     * @param lastRetryLogId
     */
    public void updateLastRetryLogId(long id, long lastRetryLogId) {
        super.updateValue(QDO.lastRetryId, lastRetryLogId, id);
    }

    /**
     * 更新发送结果
     *
     * @param id
     * @param success
     * @param failReason
     */
    public void updateReqResult(long id, boolean success, String failReason) {
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalId.eq(id)).fetchOne();

        jpaQueryFactory.update(QDO)
                .set(QDO.reqSuccess, success)
                .set(QDO.retryFailReason, failReason)
                .set(QDO.respTime, LocalDateTime.now())
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.id.eq(id))
                .execute();

    }

    /**
     * 更新重试次数
     *
     * @param id
     */
    public void updateRetryTimes(long id) {
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalId.eq(id)).fetchOne();

        jpaQueryFactory.update(QDO)
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新响应结果
     *
     * @param id
     * @param success
     * @param failReason
     * @param respBody
     */
    public void updateRespResult(long id, boolean success, String failReason, String respBody) {
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalId.eq(id)).fetchOne();
        jpaQueryFactory.update(QDO)
                .set(QDO.reqSuccess, true)
                .set(QDO.respSuccess, success)
                .set(QDO.respTime, LocalDateTime.now())
                .set(QDO.retryFailReason, failReason)
                .set(QDO.respBody, respBody)
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 更新请求时间
     *
     * @param id
     * @param reqTime
     */
    public void updateReqTime(long id, LocalDateTime reqTime) {
        super.updateValue(QDO.reqTime, reqTime, id);
    }

    /**
     * 更新重试失败的结果
     *
     * @param id
     * @param reason
     */
    public void updateRetryFailResult(long id, String reason) {
        var retryTimes = jpaQueryFactory.select(QDO.id.count())
                .from(QDO)
                .where(QDO.originalId.eq(id)).fetchOne();

        jpaQueryFactory.update(QDO)
                .set(QDO.retryFailReason, reason)
                .set(QDO.retryTimes, ObjectUtil.defaultIfNull(retryTimes, 0L).intValue())
                .where(QDO.id.eq(id))
                .execute();
    }

    /**
     * 清理过期日志
     *
     * @param expiredTime
     */
    public void clearExpiredLogs(@NotNull LocalDateTime expiredTime) {
        super.delete(QDO.reqTime.lt(expiredTime));
    }

    /**
     * 获取原始记录ID
     *
     * @param id
     * @return
     */
    public Long getOriginalLogId(long id) {
        var originalId = super.getValue(QDO.originalId, id);
        if (originalId == null) {
            var retried = super.getValue(QDO.retried, id);
            if (Boolean.FALSE.equals(retried)) {
                return id;
            }
        }
        return originalId;
    }

    /**
     * 获取最后一次重试请求的时间
     *
     * @param originalId
     * @return
     */
    public LocalDateTime getLastReqTime(long originalId) {
        return super.getValue(QDO.reqTime, QDO.originalId.eq(originalId).and(QDO.reqTime.isNotNull()), QDO.reqTime.desc());
    }

    /**
     * 查询最后一次重试记录ID
     *
     * @param ids
     * @return
     */
    public Map<Long, Long> queryLastRetry(Collection<Long> ids) {
        return super.jpaQueryFactory.select(QDO.id, QDO.lastRetryId)
                .from(QDO)
                .where(QDO.id.in(ids))
                .fetch()
                .stream()
                .collect(Collectors.toMap(t -> t.get(QDO.id), t -> t.get(QDO.lastRetryId), (t1, t2) -> t1));
    }

    /**
     * 分页查询
     *
     * @param requestDTO
     * @return
     */
    public PagingVO<SysThirdApiLogDO> queryByPage(Long sysTenantId, ThirdApiLogQueryDTO requestDTO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.sysTenantId, sysTenantId)
                .andEq(QDO.appCode, requestDTO.getAppCode())
                .andEq(QDO.restful, requestDTO.getRestful())
                .andEq(QDO.server, requestDTO.getServer())
                .andEq(QDO.clientId, requestDTO.getClientId())
                .andEq(QDO.userId, requestDTO.getUserId())
                .andEq(QDO.username, requestDTO.getUsername())
                .andEq(QDO.uri, requestDTO.getUri())
                .andEq(QDO.reqSuccess, requestDTO.getReqSuccess())
                .andEq(QDO.respSuccess, requestDTO.getRespSuccess())
                .andBetween(QDO.reqTime, requestDTO.getReqTimeStart(), requestDTO.getReqTimeEnd())
                .andEq(QDO.retried, false)
                .build();

        return super.queryByPage(predicate, requestDTO.getPageRequest());
    }

    /**
     * 分页查询
     *
     * @param queryVO
     * @return
     */
    public PagingVO<SysThirdApiLogDO> queryByPage(ThirdApiLogQueryVO queryVO, Long sysTenantId, Boolean server, String thirdApp) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.appCode, queryVO.getAppCode())
                .andEq(QDO.server, server)
                .andEq(QDO.thirdApp, thirdApp)
                .andEq(QDO.businessType, queryVO.getBusinessType())
                .andEq(QDO.businessKey, queryVO.getBusinessKey())
                .andRightLike(QDO.uri, queryVO.getUri())
                .andEq(QDO.respSuccess, queryVO.getRespSuccess())
                .andBetween(QDO.reqTime, queryVO.getReqTimeStart(), queryVO.getReqTimeEnd())
                .andEq(QDO.retried, false)
                .andEq(QDO.sysTenantId, sysTenantId)
                .build();

        return super.queryByPage(predicate, queryVO.getPageRequest(), QDO.reqTime.desc());
    }

    /**
     * 分页查询聚合
     *
     * @param queryVO
     * @param sysTenantId
     * @param server
     * @param thirdApp
     * @return
     */
    public PagingVO<ThirdApiLogAggPageRespVO> queryAggByPage(ThirdApiLogQueryVO queryVO, Long sysTenantId, Boolean server, String thirdApp) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.appCode, queryVO.getAppCode())
                .andEq(QDO.server, server)
                .andEq(QDO.thirdApp, thirdApp)
                .andEq(QDO.businessType, queryVO.getBusinessType())
                .andEq(QDO.businessKey, queryVO.getBusinessKey())
                .andRightLike(QDO.uri, queryVO.getUri())
                .andEq(QDO.respSuccess, queryVO.getRespSuccess())
                .andBetween(QDO.reqTime, queryVO.getReqTimeStart(), queryVO.getReqTimeEnd())
                .andEq(QDO.sysTenantId, sysTenantId)
                .build();;

        var qbean = Projections.bean(ThirdApiLogAggPageRespVO.class, QDO.thirdApp, QDO.businessType, QDO.businessKey,
                QDO.respSuccess, QDO.reqTime.max().as("reqTime"), QDO.count().as("reqTimes"),
                QDO.id.max().as("id"));
        var jpaQuery = super.jpaQueryFactory.select(qbean)
                .from(QDO)
                .where(predicate)
                .groupBy(QDO.thirdApp, QDO.businessType, QDO.businessKey, QDO.server, QDO.respSuccess);
        // 不使用前端的排序
        var pageRequest = queryVO.getPageRequest();
        var pageData = super.queryByPage(jpaQuery, PageRequest.of(pageRequest.getPageNumber(), pageRequest.getPageSize()), QDO.reqTime.max().desc());
        if (pageData.isEmpty()) {
            return pageData;
        }

        var ids = pageData.stream().map(ThirdApiLogAggPageRespVO::getId).collect(Collectors.toSet());
        var detailMap = super.get(ids).stream()
                .collect(Collectors.toMap(SysThirdApiLogDO::getId, t -> t, (t1, t2) -> t1));
        pageData.each(t -> {
            var detail = detailMap.get(t.getId());
            if (detail == null) {
                return;
            }
            t.setAppCode(detail.getAppCode());
            t.setServer(detail.getServer());
            t.setServerAddr(detail.getServerAddr());
            t.setUri(detail.getUri());
            t.setReqMethod(detail.getReqMethod());
        });
        return pageData;
    }

    /**
     * 查询聚合的明细列表
     *
     * @param queryVO
     * @param recordDO
     * @return
     */
    public List<SysThirdApiLogDO> listDetailForAgg(ThirdApiLogQueryVO queryVO, SysThirdApiLogDO recordDO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.appCode, recordDO.getAppCode())
                .andEq(QDO.server, recordDO.getServer())
                .andEq(QDO.thirdApp, recordDO.getThirdApp())
                .andEq(QDO.businessType, recordDO.getBusinessType())
                .andEq(QDO.businessKey, recordDO.getBusinessKey())
                .andEq(QDO.respSuccess, recordDO.getRespSuccess())
                .andBetween(QDO.reqTime, queryVO.getReqTimeStart(), queryVO.getReqTimeEnd())
                .andEq(QDO.sysTenantId, recordDO.getSysTenantId())
                .build();
        return super.getList(predicate).stream()
                .sorted(Comparator.comparing(SysThirdApiLogDO::getReqTime, Comparator.nullsFirst(LocalDateTime::compareTo)))
                .collect(Collectors.toList());
    }

    /**
     * 查询聚合的明细列表
     *
     * @param queryVO
     * @param recordDO
     * @return
     */
    public PagingVO<SysThirdApiLogDO> pageDetailForAgg(ThirdApiLogQueryVO queryVO, SysThirdApiLogDO recordDO) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.appCode, recordDO.getAppCode())
                .andEq(QDO.server, recordDO.getServer())
                .andEq(QDO.thirdApp, recordDO.getThirdApp())
                .andEq(QDO.businessType, recordDO.getBusinessType())
                .andEq(QDO.businessKey, recordDO.getBusinessKey())
                .andEq(QDO.respSuccess, recordDO.getRespSuccess())
                .andBetween(QDO.reqTime, queryVO.getReqTimeStart(), queryVO.getReqTimeEnd())
                .andEq(QDO.sysTenantId, recordDO.getSysTenantId())
                .build();
        return super.queryByPage(predicate, queryVO.getPageRequest(), QDO.reqTime.asc());
    }

    /**
     * 获取重试列表
     *
     * @param originalId
     * @return
     */
    public List<SysThirdApiLogDO> listRetryRecord(long originalId) {
        return super.getListByValue(QDO.originalId, originalId);
    }

    /**
     * 分页查询重试记录
     *
     * @param originalId
     * @param requestTimeIsNull
     * @param pageRequest
     * @return
     */
    public PagingVO<SysThirdApiLogDO> pageRetryRecord(long originalId, Boolean requestTimeIsNull, PageRequest pageRequest) {
        var predicate = PredicateBuilder.builder()
                .andEq(QDO.originalId, originalId)
                .and(requestTimeIsNull != null, () -> Boolean.TRUE.equals(requestTimeIsNull) ? QDO.reqTime.isNull() : QDO.reqTime.isNotNull())
                .build();
        return super.queryByPage(predicate, pageRequest, QDO.reqTime.desc());
    }
}
