package com.elitescloud.cloudt.system.util;

import cn.hutool.core.util.StrUtil;

import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import static com.elitescloud.cloudt.system.util.SqlUtil.SqlLikeOption.*;

/**
 * @Auther: Mark
 * @Date: 2024/12/13 14:20
 * @Description:
 */
public abstract class SqlUtil {
    /**
     * SQL中的空集合
     */
    public static final String SQL_EMPTY_SET = "(NULL)";

    /**
     * 模糊查询选项
     */
    public enum SqlLikeOption {
        BOTH_SIDES, LEFT_SIDE, RIGHT_SIDE
    }

    /**
     * @param val 文本值
     * @return SQL文本值
     */
    public static String toSqlString(CharSequence val) {
        if (StrUtil.isEmpty(val)) {
            throw new IllegalArgumentException("[EDP-SQL] Parameter(val) is null or empty");
        }
        return escapeString(new StringBuilder(), val).toString();
    }

    private static StringBuilder escapeString(StringBuilder sb, CharSequence input) {
        sb.append('\'');
        for (int i = 0, n = input.length(); i < n; i++) {
            char ch = input.charAt(i);
            sb = ch == '\'' ? sb.append("''") : sb.append(ch);
        }
        sb.append('\'');
        return sb;
    }

    /**
     * @param val    匹配值
     * @param option 匹配选项
     * @return SQL匹配值
     */
    public static String toSqlLikeString(CharSequence val, SqlLikeOption option) {
        if (StrUtil.isEmpty(val)) {
            throw new IllegalArgumentException("[EDP-SQL] Parameter(val) is null or empty");
        }
        return escapeLikeString(new StringBuilder(), val, option).toString();
    }

    private static StringBuilder escapeLikeString(StringBuilder sb, CharSequence input, SqlLikeOption option) {
        Map<Character, String> escapeTable = escapeTable();
        sb.append(option != RIGHT_SIDE ? "%" : "");
        for (int i = 0, n = input.length(); i < n; i++) {
            char ch = input.charAt(i);
            String escaped = escapeTable.get(ch);
            sb = escaped != null ? sb.append(escaped) : sb.append(ch);
        }
        sb.append(option != LEFT_SIDE ? "%" : "");
        // sb.append(escapeClause());
        return sb;
    }

    /**
     * @param val 匹配值
     * @return SQL匹配值（左右全匹配场合）
     */
    public static String toSqlLikeString(String val) {
        return toSqlLikeString(val, BOTH_SIDES);
    }

    /**
     * @param vals 文本值集
     * @return SQL文本值集（用于`IN`操作）
     */
    public static String toSqlStringSet(Collection<? extends CharSequence> vals) {
        List<? extends CharSequence> nonNullVals = vals.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        return nonNullVals.isEmpty() ? SQL_EMPTY_SET : nonNullVals.stream()
                .map(SqlUtil::toSqlString)
                .collect(Collectors.joining(",", "(", ")"));
    }

    /**
     * @param vals 数值集
     * @return SQL数值集（用于`IN`操作）
     */
    public static String toSqlNumberSet(Collection<? extends Number> vals) {
        List<? extends Number> nonNullVals = vals.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        return nonNullVals.isEmpty() ? SQL_EMPTY_SET : nonNullVals.stream()
                .map(String::valueOf)
                .collect(Collectors.joining(",", "(", ")"));
    }

    private static final Pattern RE_WORD = Pattern.compile("^\\w+$");

    /**
     * 对SQL词汇进行输入检查
     *
     * @param input 输入的词汇
     * @return 检查后的SQL词汇
     * @throws IllegalArgumentException 检查通不过时抛出
     */
    public static String toSqlWord(String input) {
        if (!RE_WORD.matcher(input).matches()) {
            throw new IllegalArgumentException("[CORE-JDBC] Illegal inputted sql word: " + input);
        }
        return input;
    }

    /**
     * 转义字符集
     */
    private static final Map<Character, String> SQL_ESC = new HashMap<>();

    static {
        SQL_ESC.put('\'', "''");
        SQL_ESC.put('%', "\\%");
        SQL_ESC.put('_', "\\_");
        SQL_ESC.put('&', "\\&");
    }

    public static Map<Character, String> escapeTable() {
        return SQL_ESC;
    }

    public static String escapeClause() {
        return " escape '\\'";
    }
}
