package com.elitescloud.cloudt.system.util;

import cn.hutool.extra.spring.SpringUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;

/**
 * @Auther: Mark
 * @Date: 2024/12/25 19:57
 * @Description: 事务相关工具类<br>
 * <p>
 * 1，#runAfterCommit：在事务提交后同步执行后续业务<br>
 * 2，#runAsyncAfterCommit：在当前事务提交后开启异步线程执行后续业务<br>
 * 3，#runNewTx：开启新事务执行后续业务<br>
 * 4，#runTx：开启事务执行业务
 */
@Slf4j
public class TransactionUtil {
    private static TransactionService newTransactionService;

    /**
     * 当前数据库事务提交后执行
     */
    public static void runAfterCommit(Runnable runnable) {
        if (TransactionSynchronizationManager.isActualTransactionActive()) {
            TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronization() {
                @Override
                public void afterCompletion(int status) {
                    runNewTx(runnable);
                }
            });
        } else {
            runnable.run();
        }
    }

    /**
     * 新业务开启新事务
     *
     * @param runnable
     */
    public static void runNewTx(Runnable runnable) {
        if (newTransactionService == null) {
            newTransactionService = SpringUtil.getBean(TransactionService.class);
        }

        newTransactionService.executeWithNew(runnable);
    }

    /**
     * 事务包裹
     *
     * @param runnable
     */
    public static void runTx(Runnable runnable) {
        if (newTransactionService == null) {
            newTransactionService = SpringUtil.getBean(TransactionService.class);
        }

        newTransactionService.execute(runnable);
    }
}
