package com.elitescloud.cloudt.comm.rest;

import com.elitescloud.cloudt.comm.vo.param.ComPaymentTermQueryParamVO;
import com.elitescloud.cloudt.comm.vo.resp.ComPaymentTermRespVO;
import com.elitescloud.cloudt.comm.vo.save.ComPaymentTermSaveVO;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.service.IComPaymentTermService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;


/**
 * 支付条款
 *
 * @author shihao.ma
 * 2020/7/2
 */
@RestController
@RequestMapping("/com/comPaymentTerm")
@RequiredArgsConstructor
@Api(value = "支付条款", tags = {"【过渡】支付条款"})
public class ComPaymentTermController {
    private final IComPaymentTermService comPaymentTermService;

    /**
     * 列出所有的付款条款
     *
     * @return 付款条款
     */
    @GetMapping("/list")
    @ApiOperation("列出所有的付款条款")
    public ApiResult<?> list() {
        return ApiResult.ok(comPaymentTermService.list());
    }

    /**
     * 保存支付条款
     *
     * @param param 支付条款
     * @return 保存结果
     */
    @PostMapping("/createOne")
    @ApiOperation("保存支付条款")
    public ApiResult<?> createOne(@RequestBody ComPaymentTermSaveVO param) {
        Long id = comPaymentTermService.createOne(param);
        return ApiResult.ok(id);
    }

    /**
     * 批量保存支付条款
     *
     * @param param 支付条款
     * @return 保存结果
     */
    @PostMapping("/createBatch")
    @ApiOperation("批量保存支付条款")
    public ApiResult<?> createBatch(@RequestBody List<ComPaymentTermSaveVO> param) {
        List<Long> idList = comPaymentTermService.createBatch(param);
        return ApiResult.ok(idList);
    }

    /**
     * 更新支付条款
     *
     * @param paymentTermId 记录ID
     * @param param         支付条款
     * @return 更新结果
     */
    @PutMapping("/update/{paymentTermId}")
    @ApiOperation("更新支付条款")
    public ApiResult<?> update(@PathVariable Long paymentTermId, @RequestBody ComPaymentTermSaveVO param) {
        param.setId(paymentTermId);
        comPaymentTermService.update(param);
        return ApiResult.ok();
    }

    /**
     * 根据ID查询支付条款
     *
     * @param paymentTermIdList 支付条款ID
     * @return 支付条款
     */
    @PostMapping("/findIdBatch")
    @ApiOperation("根据ID查询支付条款")
    public ApiResult<?> findIdBatch(@RequestBody List<String> paymentTermIdList) {
        // 将入参的String转为Long类型
        List<Long> list = paymentTermIdList.stream().map(Long::valueOf).collect(Collectors.toList());
        List<ComPaymentTermRespVO> result = comPaymentTermService.findIdBatch(list);
        return ApiResult.ok(result);
    }

    /**
     * 根据ID查询支付条款
     *
     * @param paymentTermId 支付条款ID
     * @return 支付条款
     */
    @PostMapping("/findIdOne/{paymentTermId}")
    @ApiOperation("根据ID查询支付条款")
    public ApiResult<?> findIdOne(@PathVariable String paymentTermId) {
        Optional<ComPaymentTermRespVO> result = comPaymentTermService.findIdOne(Long.valueOf(paymentTermId));
        return ApiResult.ok(result);
    }

    /**
     * 根据Code查询支付条款
     *
     * @param paymentTermCode 支付条款code
     * @return 支付条款
     */
    @PostMapping("/findCodeOne/{paymentTermCode}")
    @ApiOperation("根据Code查询支付条款")
    public ApiResult<?> findCodeOne(@PathVariable String paymentTermCode) {
        Optional<ComPaymentTermRespVO> result = comPaymentTermService.findCodeOne(paymentTermCode);
        return ApiResult.ok(result);
    }

    /**
     * 分页查询支付条款
     *
     * @param param 查询参数
     * @return 支付条款列表
     */
    @PostMapping("/search")
    @ApiOperation("分页查询支付条款")
    public ApiResult<?> search(@RequestBody ComPaymentTermQueryParamVO param) {
        PagingVO<ComPaymentTermRespVO> result = comPaymentTermService.search(param);
        return ApiResult.ok(result);
    }

    /**
     * 删除支付条款
     *
     * @param paymentTermId 支付条款ID
     * @return 删除结果
     */
    @DeleteMapping("/deleteOne")
    @ApiOperation("删除支付条款")
    public ApiResult<?> deleteOne(Long paymentTermId) {
        comPaymentTermService.deleteOne(paymentTermId);
        return ApiResult.ok();
    }

    /**
     * 批量删除支付条款
     *
     * @param paymentTermIdList 字符条款ID
     * @return 删除结果
     */
    @DeleteMapping("/deleteBatch")
    @ApiOperation("批量删除支付条款")
    public ApiResult<?> deleteBatch(@RequestBody List<String> paymentTermIdList) {
        List<Long> idList = paymentTermIdList.stream().map(Long::valueOf).collect(Collectors.toList());
        comPaymentTermService.deleteBatch(idList);
        return ApiResult.ok();
    }

    /**
     * 更新支付条款删除标识
     *
     * @param id 字符条款ID
     * @return 更新结果
     */
    @PostMapping("/updateDeleteFlag")
    @ApiOperation("更新支付条款删除标识")
    public ApiResult<?> updateDeleteFlag(String id) {
        comPaymentTermService.updateDeleteFlag(Long.valueOf(id));
        return ApiResult.ok();
    }


}
