package com.elitescloud.cloudt.system.controller.mng.common;

import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.constant.BusinessObjectConstant;
import com.elitescloud.cloudt.system.model.vo.CommonTenantAppVO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonAppPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.extend.SysBankBranchPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.extend.SysBankPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.SysBankBranchRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.extend.SysBankRespVO;
import com.elitescloud.cloudt.system.service.model.vo.CommonAreaQueryVO;
import com.elitescloud.cloudt.system.service.model.vo.CommonAreaTreeQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.common.CommonDataRelationQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.extend.CurrencyRateQueryVO;
import com.elitescloud.cloudt.system.model.vo.query.udc.UdcPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.common.*;
import com.elitescloud.cloudt.system.model.vo.resp.sys.PwdStrategyRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.udc.UdcWithValuesPageRespVO;
import com.elitescloud.cloudt.system.service.SystemQueryService;
import com.elitescloud.cloudt.system.service.model.vo.CommonAreaTreeRespVO;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 通用系统接口.
 *
 * @author Kaiser（wang shao）
 * 2022/10/11
 */
@BusinessObject(businessType = BusinessObjectConstant.SYS_COMMON)
@Api(tags = "通用系统接口")
@RestController
@RequestMapping(value = "/mng/common/system", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
public class CommonSystemController {

    private final SystemQueryService systemQueryService;

    public CommonSystemController(SystemQueryService systemQueryService) {
        this.systemQueryService = systemQueryService;
    }

    /**
     * 应用列表
     *
     * @return 应用列表
     */
    @ApiOperation(value = "应用列表")
    @ApiOperationSupport(order = 1)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "outerApp", value = "是否外部应用"),
            @ApiImplicitParam(name = "authed", value = "是否需要授权")
    })
    @GetMapping(value = "/apps")
    public ApiResult<List<CommonTenantAppVO>> listApp(@RequestParam(value = "outerApp", required = false) Boolean outerApp,
                                                      @RequestParam(value = "authed", required = false) Boolean authed) {
        return systemQueryService.listApp(outerApp, authed);
    }

    /**
     * 应用分页列表
     *
     * @param queryVO 查询参数
     * @return 应用列表
     */
    @ApiOperation(value = "应用分页列表")
    @ApiOperationSupport(order = 1)
    @PostMapping(value = "/apps/page")
    public ApiResult<PagingVO<CommonTenantAppVO>> pageApp(@RequestBody CommonAppPageQueryVO queryVO) {
        return systemQueryService.pageApp(queryVO);
    }

    /**
     * 货币列表
     *
     * @return 货币列表
     */
    @ApiOperation(value = "货币列表")
    @ApiOperationSupport(order = 2)
    @GetMapping(value = "/currency")
    public ApiResult<List<CodeNameParam>> listCurrency() {
        return systemQueryService.listCurrency();
    }

    /**
     * 税率列表
     *
     * @return 税率列表
     */
    @ApiOperation(value = "税率列表")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "taxRateNo", value = "税率编码"),
            @ApiImplicitParam(name = "taxRateType", value = "税率类型, UDC[cloudt-system:taxRateType]"),
    })
    @GetMapping(value = "/taxRate")
    public ApiResult<List<CommonTaxRateRespVO>> listTaxRate(@RequestParam(name = "taxRateNo", required = false) String taxRateNo,
                                                            @RequestParam(name = "taxRateType", required = false) String taxRateType) {
        return systemQueryService.listTaxRate(taxRateNo, taxRateType);
    }

    /**
     * 查询汇率
     *
     * @return 税率列表
     */
    @ApiOperation(value = "查询汇率")
    @ApiOperationSupport(order = 3)
    @PostMapping(value = "/currencyRatio")
    public ApiResult<Double> findTaxRate(@RequestBody @Valid CurrencyRateQueryVO queryVO) {
        return systemQueryService.queryCurrentRate(queryVO);
    }

    @ApiOperation(value = "分页查询银行信息")
    @ApiOperationSupport(order = 4)
    @PostMapping(value = "/bank")
    public ApiResult<PagingVO<SysBankRespVO>> pageQueryBank(@RequestBody SysBankPageQueryVO queryVO) {
        return systemQueryService.pageQueryBank(queryVO);
    }

    @ApiOperation(value = "分页查询银行支行信息")
    @ApiOperationSupport(order = 4)
    @PostMapping(value = "/bankBranch")
    public ApiResult<PagingVO<SysBankBranchRespVO>> pageQueryBankBranch(@RequestBody SysBankBranchPageQueryVO queryVO) {
        return systemQueryService.pageQueryBankBranch(queryVO);
    }

    /**
     * 行政区域列表
     *
     * @return 行政区域列表
     */
    @ApiOperation(value = "行政区域列表")
    @ApiOperationSupport(order = 5)
    @GetMapping(value = "/area")
    public ApiResult<List<CommonAreaTreeRespVO>> listArea(CommonAreaQueryVO queryVO) {
        return systemQueryService.listArea(queryVO);
    }

    @ApiOperation(value = "行政区域树列表")
    @ApiOperationSupport(order = 5)
    @GetMapping(value = "/area/tree")
    public ApiResult<List<CommonAreaTreeRespVO>> listArea(CommonAreaTreeQueryVO queryVO) {
        return systemQueryService.treeArea(queryVO);
    }

    /**
     * 获取密码策略
     *
     * @return 密码策略
     */
    @ApiOperation(value = "获取密码策略")
    @ApiOperationSupport(order = 6)
    @GetMapping(value = "/pwdStrategy")
    public ApiResult<List<PwdStrategyRespVO>> listStrategy() {
        return systemQueryService.listPwdStrategyForCreateAccount();
    }

    /**
     * 获取外部应用设置
     *
     * @return 外部应用设置
     */
    @ApiOperation(value = "获取外部应用设置")
    @ApiOperationSupport(order = 7)
    @GetMapping(value = "/outerAppSettings")
    public ApiResult<List<OuterAppSettingRespVO>> listOuterAppSettings() {
        return systemQueryService.listOuterAppSettings();
    }

    @ApiOperation(value = "查询数据选择器")
    @ApiOperationSupport(order = 8)
    @GetMapping(value = "/dataSelector/tree")
    public ApiResult<List<CommonDataSelectorListRespVO>> listTree(@RequestParam(name = "tree", required = false) Boolean tree) {
        return systemQueryService.listTree(tree);
    }

    @ApiOperation(value = "根据数据查询被关联数据")
    @ApiOperationSupport(order = 9)
    @PostMapping(value = "/dataRelation")
    public ApiResult<List<CommonDataRelationRespVO>> queryRelation(@RequestBody CommonDataRelationQueryVO queryVO) {
        return systemQueryService.queryRelation(queryVO);
    }

    @ApiOperation(value = "根据被关联的数据查询数据")
    @ApiOperationSupport(order = 10)
    @PostMapping(value = "/dataRelation/ref")
    public ApiResult<List<CommonDataRelationRespVO>> queryRelationByRef(@RequestBody CommonDataRelationQueryVO queryVO) {
        return systemQueryService.queryRelationByRefData(queryVO);
    }

    @ApiOperation(value = "分页查询UDC")
    @ApiOperationSupport(order = 11)
    @PostMapping(value = "/udc/page")
    public ApiResult<PagingVO<UdcWithValuesPageRespVO>> pageQuery(@RequestBody UdcPageQueryVO queryVO) {
        return systemQueryService.pageQueryUdc(queryVO);
    }
}
