package com.elitescloud.cloudt.system.controller.mng.org;

import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.boot.common.param.IdCodeNameCheckParam;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.model.vo.query.org.EmployeePageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmpOrgRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeeDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.org.EmployeePagedRespVO;
import com.elitescloud.cloudt.system.model.vo.save.org.EmployeeSaveVO;
import com.elitescloud.cloudt.system.service.EmployeeMngService;
import com.elitescloud.cloudt.system.constant.BusinessObjectConstant;
import com.elitescloud.cloudt.system.service.common.constant.BusinessOperationConstant;
import com.elitescloud.cloudt.system.service.model.entity.SysEmployeeDO;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.dsl.EntityPathBase;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.core.types.dsl.MapPath;
import com.querydsl.core.types.dsl.StringPath;
import com.querydsl.jpa.impl.JPAQueryFactory;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.Map;

/**
 * 员工管理.
 *
 * @author Kaiser（wang shao）
 * 2022/9/20
 */
@Api(tags = "员工管理")
@RestController
@RequestMapping(value = "/mng/employee", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectConstant.SYS_EMPLOYEE, businessDoClass = SysEmployeeDO.class)
public class EmployeeMngController {

    private final EmployeeMngService employeeMngService;

    public EmployeeMngController(EmployeeMngService employeeMngService) {
        this.employeeMngService = employeeMngService;
    }

    /**
     * 分页查询员工
     *
     * @param queryVO 查询参数
     * @return 分页数据
     */
    @ApiOperation(value = "员工管理分页查询")
    @ApiOperationSupport(order = 1)
    @PostMapping(value = "/page")
    @BusinessObjectOperation(operationCode = BusinessOperationConstant.EMP_MNG_PAGE, operationType = OperationTypeEnum.SELECT_PAGE, operationDescription = "分页查询员工",
            dataPermissionEnabled = false, fieldPermissionAutoFilter = true)
    public ApiResult<PagingVO<EmployeePagedRespVO>> page(@RequestBody EmployeePageQueryVO queryVO) {
        return employeeMngService.page(queryVO);
    }

    @ApiOperation(value = "测试员工扩展信息修改")
    @PostMapping(value = "/testUpdate")
    public ApiResult testUpdate(@RequestBody @Valid EmployeeSaveVO saveVO) {
        return employeeMngService.testUpdate(saveVO);
    }


    /**
     * 切换启用禁用
     *
     * @param id 员工ID
     * @return 切换结果
     */
    @ApiOperation(value = "切换启用禁用")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "id", value = "员工ID", required = true, dataType = "int")
    @PatchMapping(value = "/switch/enabled/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "切换启用禁用"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> switchEnabled(@PathVariable("id") Long id) {
        return employeeMngService.updateEnabled(id);
    }

    /**
     * 删除员工
     *
     * @param id       员工ID
     * @param keepUser 是否保留账号
     * @return 删除结果
     */
    @ApiOperation(value = "删除员工")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "员工ID", required = true, dataType = "int", paramType = "path"),
            @ApiImplicitParam(name = "keepUser", value = "是否保留账号", required = true, dataType = "boolean", paramType = "query"),
    })
    @DeleteMapping(value = "/del/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.DELETE, operationDescription = "删除员工"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> delete(@PathVariable("id") Long id,
                                  @RequestParam(name = "keepUser", defaultValue = "false") Boolean keepUser) {
        return employeeMngService.delete(id, keepUser);
    }

    /**
     * 获取员工详情
     *
     * @param id 员工ID
     * @return 员工信息
     */
    @ApiOperation(value = "获取员工详情")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParam(name = "id", value = "员工ID", required = true, dataType = "int")
    @GetMapping(value = "/get/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "员工详情", dataPermissionEnabled = true)
    public ApiResult<EmployeeDetailRespVO> get(@PathVariable("id") Long id) {
        return employeeMngService.getDetail(id);
    }

    /**
     * 保存员工[全部信息]
     *
     * @param saveVO 员工信息
     * @return 新增结果
     */
    @ApiOperation(value = "保存员工[全部信息]")
    @ApiOperationSupport(order = 5)
    @PostMapping(value = "/save")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存员工[全部信息]"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true, allowRepeatRequest = false)
    public ApiResult<Long> save(@RequestBody @Valid EmployeeSaveVO saveVO) {
        return employeeMngService.upsert(saveVO);
    }
//
//    /**
//     * 新增员工[账户信息]
//     *
//     * @param orgId  组织ID
//     * @param saveVO 账户信息
//     * @return 新增结果
//     */
//    @ApiOperation(value = "新增员工[账户信息]")
//    @ApiOperationSupport(order = 7)
//    @ApiImplicitParam(name = "orgId", value = "组织ID", required = true, dataType = "int")
//    @PostMapping(value = "/{orgId}/add/userInfo")
//    @BusinessObjectOperation(operationType = OperationTypeEnum.ADD, operationDescription = "新增员工[账户信息]"
//            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true, allowRepeatRequest = false)
//    public ApiResult<Long> saveUser(@PathVariable("orgId") Long orgId, @RequestBody @Valid UserSaveVO saveVO) {
//        return employeeMngService.upsertUser(orgId, saveVO);
//    }
//
//    /**
//     * 修改员工[账户信息]
//     *
//     * @param saveVO 账户信息
//     * @return 新增结果
//     */
//    @ApiOperation(value = "修改员工[账户信息]")
//    @ApiOperationSupport(order = 8)
//    @PutMapping(value = "/update/userInfo")
//    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "修改员工[账户信息]"
//            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true, allowRepeatRequest = false)
//    public ApiResult<Long> updateUser(@RequestBody @Valid UserSaveVO saveVO) {
//        return employeeMngService.upsertUser(null, saveVO);
//    }
//
//    /**
//     * 修改员工[组织信息]
//     *
//     * @param saveVO 组织信息
//     * @return 新增结果
//     */
//    @ApiOperation(value = "保存员工[组织信息]")
//    @ApiOperationSupport(order = 10)
//    @PutMapping(value = "/save/orgInfo")
//    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存员工[组织信息]"
//            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true, allowRepeatRequest = false)
//    public ApiResult<Long> updateOrg(@RequestBody @Valid EmployeeOrgInfoSaveVO saveVO) {
//        return employeeMngService.upsertOrg(saveVO);
//    }
//
//    /**
//     * 保存员工的所有组织
//     *
//     * @param id      员工ID
//     * @param orgList 组织列表
//     * @return 保存结果
//     */
//    @ApiOperation(value = "保存员工的所有组织")
//    @ApiOperationSupport(order = 21)
//    @ApiImplicitParams({
//            @ApiImplicitParam(name = "id", value = "员工ID", required = true, dataType = "int"),
//    })
//    @PutMapping(value = "/update/{id}/org")
//    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存员工[所有组织]"
//            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true, allowRepeatRequest = false)
//    public ApiResult<Long> saveOrg(@PathVariable("id") Long id, @RequestBody List<EmployeeOrgSaveVO> orgList) {
//        return employeeMngService.saveEmployeeOrg(id, orgList);
//    }

    /**
     * 获取员工的所有组织
     *
     * @param id 员工ID
     * @return 保存结果
     */
    @ApiOperation(value = "获取员工的所有组织")
    @ApiOperationSupport(order = 22)
    @ApiImplicitParam(name = "id", value = "员工ID", required = true, dataType = "int")
    @GetMapping(value = "/get/{id}/org")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取员工的所有组织"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<List<EmpOrgRespVO>> getOrg(@PathVariable("id") Long id) {
        return employeeMngService.getEmployeeOrgs(id);
    }

    /**
     * 获取员工的角色
     *
     * @param id 员工ID
     * @return 角色
     */
    @ApiOperation(value = "获取员工的角色")
    @ApiOperationSupport(order = 31)
    @ApiImplicitParam(name = "id", value = "员工ID", required = true, dataType = "int")
    @GetMapping(value = "/get/{id}/role")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "获取员工的角色"
            , logEnabled = false, resultJsonEnabled = false, argsJsonEnabled = false)
    public ApiResult<List<IdCodeNameCheckParam>> getRole(@PathVariable("id") Long id) {
        return employeeMngService.getRole(id);
    }

    /**
     * 保存员工的角色
     *
     * @param id 员工ID
     * @return 角色
     */
    @ApiOperation(value = "保存员工的角色")
    @ApiOperationSupport(order = 32)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "员工ID", required = true, dataType = "int"),
            @ApiImplicitParam(name = "roleIds", value = "角色ID", required = true, dataType = "int", allowMultiple = true),
    })
    @PutMapping(value = "/save/{id}/role")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存员工的角色"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true, allowRepeatRequest = false)
    public ApiResult<Long> saveRole(@PathVariable("id") Long id, @RequestBody List<Long> roleIds) {
        return employeeMngService.saveRole(id, roleIds);
    }
}
