package com.elitescloud.cloudt.system.controller.mng.udc;

import com.elitescloud.boot.common.annotation.BusinessObject;
import com.elitescloud.boot.common.annotation.BusinessObjectOperation;
import com.elitescloud.boot.common.annotation.businessobject.OperationTypeEnum;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.system.service.model.entity.SysPlatformUdcDO;
import com.elitescloud.cloudt.system.model.vo.query.udc.UdcPageQueryVO;
import com.elitescloud.cloudt.system.model.vo.resp.udc.UdcDetailRespVO;
import com.elitescloud.cloudt.system.model.vo.resp.udc.UdcPageRespVO;
import com.elitescloud.cloudt.system.model.vo.save.udc.UdcSaveVO;
import com.elitescloud.cloudt.system.model.vo.save.udc.UdcValueSaveVO;
import com.elitescloud.cloudt.system.service.UdcMngService;
import com.elitescloud.cloudt.system.constant.BusinessObjectConstant;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiOperation;
import org.springframework.http.MediaType;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;

/**
 * UDC管理.
 *
 * @author Kaiser（wang shao）
 * 2022/9/19
 */
@Api(tags = "UDC管理")
@RestController
@RequestMapping(value = "/mng/udc", produces = MediaType.APPLICATION_JSON_VALUE)
@Validated
@BusinessObject(businessType = BusinessObjectConstant.SYS_PLATFORM_UDC, businessDoClass = SysPlatformUdcDO.class)
public class UdcMngController {

    private final UdcMngService udcMngService;

    public UdcMngController(UdcMngService udcMngService) {
        this.udcMngService = udcMngService;
    }

    /**
     * 分页查询UDC
     *
     * @param queryVO 查询参数
     * @return 分页数据
     */
    @ApiOperation(value = "分页查询UDC")
    @ApiOperationSupport(order = 1)
    @PostMapping(value = "/page")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "分页查询UDC", dataPermissionEnabled = true)
    public ApiResult<PagingVO<UdcPageRespVO>> page(@RequestBody UdcPageQueryVO queryVO) {
        return udcMngService.page(queryVO);
    }

    /**
     * 获取UDC详情
     *
     * @param id UDC ID
     * @return UDC详情
     */
    @ApiOperation(value = "获取UDC详情")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParam(name = "id", value = "UDC的ID", required = true, dataType = "int")
    @GetMapping(value = "/get/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SELECT, operationDescription = "id查询")
    public ApiResult<UdcDetailRespVO> detail(@PathVariable("id") Long id) {
        return udcMngService.detail(id);
    }

    /**
     * 保存UDC
     *
     * @param saveVO UDC信息
     * @return UDC ID
     */
    @ApiOperation(value = "保存UDC")
    @ApiOperationSupport(order = 3)
    @PostMapping(value = "/save")
    @BusinessObjectOperation(
            operationType = OperationTypeEnum.SAVE, operationDescription = "保存UDC"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true
    )
    public ApiResult<Long> save(@Valid @RequestBody UdcSaveVO saveVO) {
        return udcMngService.save(saveVO);
    }

    /**
     * 保存UDC的值
     *
     * @param id     UDC ID
     * @param saveVO UDC值
     * @return UDC值 ID
     */
    @ApiOperation(value = "保存UDC的值")
    @ApiOperationSupport(order = 4)
    @ApiImplicitParam(name = "id", value = "UDC的ID", required = true, dataType = "int")
    @PostMapping(value = "/save/{id}/value")
    @BusinessObjectOperation(operationType = OperationTypeEnum.SAVE, operationDescription = "保存UDC值"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> saveValue(@PathVariable Long id, @Valid @RequestBody UdcValueSaveVO saveVO) {
        return udcMngService.saveValue(id, saveVO);
    }

    /**
     * 切换启用状态
     *
     * @param id UDC值ID
     * @return UDC值 ID
     */
    @ApiOperation(value = "切换启用状态")
    @ApiOperationSupport(order = 5)
    @ApiImplicitParam(name = "id", value = "UDC值的ID", required = true, dataType = "int")
    @PatchMapping(value = "/switch/value/{id}/enabled")
    @BusinessObjectOperation(operationType = OperationTypeEnum.UPDATE, operationDescription = "切换启用状态"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> updateValueEnabled(@PathVariable Long id) {
        return udcMngService.updateValueEnabled(id);
    }

    /**
     * 删除UDC的值
     *
     * @param id UDC值ID
     * @return UDC值 ID
     */
    @ApiOperation(value = "删除UDC的值")
    @ApiOperationSupport(order = 6)
    @ApiImplicitParam(name = "id", value = "UDC值的ID", required = true, dataType = "int")
    @DeleteMapping(value = "/delete/value/{id}")
    @BusinessObjectOperation(operationType = OperationTypeEnum.DELETE, operationDescription = "删除UDC的值"
            , logEnabled = true, resultJsonEnabled = true, argsJsonEnabled = true)
    public ApiResult<Long> deleteValue(@PathVariable Long id) {
        return udcMngService.deleteValue(id);
    }
}
