package com.elitescloud.cloudt.system.model.vo.resp.extend;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.extra.spring.SpringUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import com.elitescloud.cloudt.system.common.SyncConstant;
import com.elitescloud.cloudt.system.dto.ThirdApiLogDTO;
import com.elitescloud.cloudt.system.service.impl.ThirdApiLogRetrySupportServiceImpl;
import com.elitescloud.cloudt.system.util.*;
//import com.lzhpo.tracer.util.TracerUtils;
import io.swagger.annotations.ApiModelProperty;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.web.reactive.function.client.ClientResponse;
import org.springframework.web.reactive.function.client.WebClient;
import reactor.core.publisher.Mono;

import java.time.LocalDateTime;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;

import static cn.hutool.core.date.DatePattern.NORM_DATETIME_MS_PATTERN;

/**
 * @Auther: Mark
 * @Date: 2024/12/17 17:05
 * @Description:
 */

@Slf4j
@AllArgsConstructor(staticName = "of")
public class InfinityResult<T> {

    @ApiModelProperty(value = "响应值的实体类型")
    private Class<T> responseType;

    private HttpParam param;

    private InfinityLogVO logVO;

    private WebClient.RequestBodySpec uri;

    public List<T> toList() {
        return wrapper(response -> response.bodyToFlux(responseType).collectList());
    }

    public T toEntity() {
        return wrapper(response -> response.bodyToMono(responseType));
    }

    /**
     * 重构接口日志的报文结构
     *
     * @since 0.3.1-SNAPSHOT
     */
    private <R> R wrapper(Function<ClientResponse, Mono<R>> fn) {
        LocalDateTime startTime = LocalDateTime.now();
        AtomicReference<HttpStatus> httpStatus = new AtomicReference<>(null);
        AtomicReference<String> errorMessage = new AtomicReference<>("");
        R result = uri.exchangeToMono(response -> {
            httpStatus.set(response.statusCode());
            if (response.statusCode() == HttpStatus.OK) {
                return fn.apply(response);
            }
            // Log error and retry logic
            return response.createException().flatMap(Mono::error);
        }).onErrorResume(e -> {
            httpStatus.set(HttpStatus.INTERNAL_SERVER_ERROR);
            errorMessage.set(ExceptionUtils.formatException(e, SyncConstant.ERR_EXCEED_LENGTH));
            log.error("[INFINITY][{}] 接口请求失败: ", param.getApiCode(), e);
            // Handle error silently without throwing exception
            return Mono.empty();
        }).block();
        LocalDateTime endTime = LocalDateTime.now();
        boolean success = httpStatus.get() == HttpStatus.OK;
        InfinityLogVO infinityLogVO = toInfinityParam(result, success, errorMessage.get(), startTime, endTime);
        ThirdApiLogDTO thirdApiLogDTO = toThirdApiLogDTO(infinityLogVO);
        ThirdApiLogRetrySupportServiceImpl sysThirdApiLogRpcService =
                SpringUtil.getBean(ThirdApiLogRetrySupportServiceImpl.class);
        sysThirdApiLogRpcService.saveLog(thirdApiLogDTO);
        //InfinityLogUtil.record(infinityLogVO);
        //log.info("[INFINITY][" + param.getApiCode() + "] 接口出入参数:", infinityLogVO);
        // Throw Exception
        if (!success) {
            String defaultMessage = "[" + param.getApiCode() + "] 接口调用异常：" + httpStatus.get().value();
            throw new RuntimeException(StrUtil.blankToDefault(errorMessage.get(), defaultMessage));
        }
        return result;
    }

    private ThirdApiLogDTO toThirdApiLogDTO(InfinityLogVO logVO) {

        ThirdApiLogDTO logDTO = new ThirdApiLogDTO();
        logDTO.setAppCode("YST-INFINITE");
        logDTO.setRestful(true);
        logDTO.setServer(false);
        logDTO.setServerAddr(logVO.getServerUrl()); // serverUrl
        logDTO.setThirdApp(logVO.getTargetApp());
        logDTO.setBusinessType(logVO.getBusinessType());
        logDTO.setBusinessKey(logVO.getBusinessKey());
        logDTO.setClientId("");
        logDTO.setUserId(SecurityContextUtil.currentUserId());
        logDTO.setUsername(SecurityContextUtil.currentUserName());
        logDTO.setUri(logVO.getApiUrl()); // apiUri
        logDTO.setReqMethod(logVO.getRequestMethod());
        logDTO.setReqQueryParams(MapUtils.convert(logVO.getQueryParam()));
        logDTO.setReqBody(BeanUtils.toJsonOrEmpty(logVO.getBodyParam()));
        logDTO.setReqHeaders(MapUtils.convert(logVO.getHeaderParam()));
        logDTO.setReqSuccess(logVO.getRequestSuccess());
        logDTO.setReqFailMsg(logVO.getRequestFailMessage());
        logDTO.setReqTime(DateUtil.parse(logVO.getRequestTime()).toLocalDateTime());
        logDTO.setReqIp(logVO.getAddressIp());
        logDTO.setRespBody(BeanUtils.toJsonOrEmpty(logVO.getResponseBody()));
        logDTO.setRespSuccess(logVO.getResponseSuccess());
        logDTO.setRespFailMsg(logVO.getResponseFailMessage());
        logDTO.setRespTime(DateUtil.parse(logVO.getResponseTime()).toLocalDateTime());
        logDTO.setNeedRetry(false);
        logDTO.setDetectedOperatorAuth(false);
        return logDTO;
    }

    /**
     * 报文转换：Api Result => InfinityLogVO
     *
     * @since 0.3.1-SNAPSHOT
     */
    private InfinityLogVO toInfinityParam(Object responseBody, boolean requestSuccess, String reqFailMsg,
                                          LocalDateTime startTime, LocalDateTime endTime) {
        // 接口定义
        InfinityApiDetailVO apiConfig = param.getApiConfig();
        logVO.setAddressIp(HttpServletUtil.currentClientIp());
        logVO.setRequestTime(DateUtil.format(startTime, NORM_DATETIME_MS_PATTERN));
        logVO.setResponseTime(DateUtil.format(endTime, NORM_DATETIME_MS_PATTERN));
        long spendTime = LocalDateTimeUtil.between(startTime, endTime).toMillis();
        logVO.setSpendTime(spendTime);
        logVO.setRequestSuccess(requestSuccess);
        logVO.setRequestFailMessage(reqFailMsg);
        logVO.setResponseBody(responseBody);
//        logVO.setTraceId(TracerUtils.getTraceId());
        logVO.setTenantCode(TenantSpiUtil.currentTenantCode());
        logVO.setTargetApp(apiConfig.getPlatformCode());
        logVO.setBusinessType(apiConfig.getApiCode());
        logVO.setBusinessTypeName(apiConfig.getApiName());
        logVO.setBusinessKey(param.getBizKey());

        // 成功状态值
        String successStatus = apiConfig.getResponseSuccessStatus();
        // 响应状态值
        String responseStatus = StrUtil.EMPTY;
        String successSpel = apiConfig.getResponseSuccessSpel();
        if (StrUtil.isNotBlank(successSpel)) {
            log.info("[INFINITY][ETL] 提取响应报文中，成功状态标识字段: ");
            responseStatus = TransformUtil.parse(responseBody, successSpel);
        }
        // 接口返回状态、接口定义状态，都不为空
        if (StrUtil.isNotBlank(responseStatus) && StrUtil.isNotBlank(successStatus)) {
            boolean status = responseStatus.equals(successStatus);
            logVO.setResponseSuccess(status);
            if (!status) {
                log.info("[INFINITY][ETL] 提取响应报文中，错误信息字段: ");
                logVO.setResponseFailMessage(TransformUtil.parse(responseBody,
                        apiConfig.getResponseErrorMessageSpel()));
            }
        }
        // 任一为空，则以 request 状态码为准
        else {
            logVO.setResponseSuccess(requestSuccess);
            logVO.setResponseFailMessage(reqFailMsg);
        }
        return logVO;
    }
}
