package com.elitescloud.cloudt.system.modules.dpr;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ArrayUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.datasecurity.common.extension.DprValueResolverSPI;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleConditionEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleRelationEnum;
import com.elitescloud.boot.datasecurity.dpr.content.DprRuleValueTypeEnum;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.util.ObjUtil;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.convert.SysDprRoleApiRuleConvert;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiDataRuleListQueryDTO;
import com.elitescloud.cloudt.system.dto.req.UnifyQueryQueryDTO;
import com.elitescloud.cloudt.system.model.vo.sbean.SysDprRoleApiDataRuleListQueryBean;
import com.elitescloud.cloudt.system.modules.dpr.strategy.DprValueResolverFactory;
import com.elitescloud.cloudt.system.service.SysUnifyQueryQueryService;
import com.elitescloud.cloudt.system.service.model.bo.DprRuleValueSysInternallyBo;
import com.elitescloud.cloudt.system.service.repo.BusinessObjectRepoProc;
import com.elitescloud.cloudt.system.service.repo.RoleDataPermissionRepoProc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.task.TaskExecutor;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;


/**
 * @author : chen
 * <p>
 * 2022-11-24 14:22
 */
@Slf4j
public class RoleAppApiDataPermissionUtil {
    public static final String NO_PERMISSION = "-1";


    /**
     * @param userInfo 当前用户信息
     *                 valueServiceSpi 接口
     *                 sysBean 角色规则对象
     * @return SysDprRoleApiDataRuleListQueryDTO 角色规则对象包括规则值
     * @description 设置系统内置规则值 会根据不同的规则策略动态获取。
     */
    public static List<SysDprRoleApiDataRuleListQueryDTO> setSysSysDprRoleApiRuleValue(GeneralUserDetails userInfo,
                                                                                       List<SysDprRoleApiDataRuleListQueryBean> sysBean) {
        Map<String, DprRuleValueSysInternallyBo> dprSysInternallyGroupMap = new HashMap<>(16);
        List<SysDprRoleApiDataRuleListQueryDTO> sysDprRoleApiDataRuleList =
                sysBean.stream().map(sysDprRoleApiDataRuleListQueryBean -> {
                    var ruleDTO = SysDprRoleApiRuleConvert.INSTANCE.beanToDto(sysDprRoleApiDataRuleListQueryBean);
                    String dprSysInternally = CharSequenceUtil.blankToDefault(ruleDTO.getDprRuleValue(), ruleDTO.getDprSysInternally());
                    //本人-下级等内置策略,不同的菜单权限只需要查询一次，没有必要多次循环，已经查询过了存储起来，下次直接使用。
                    DprRuleValueSysInternallyBo valueBo;
                    if (dprSysInternallyGroupMap.containsKey(dprSysInternally)) {
                        valueBo = dprSysInternallyGroupMap.get(dprSysInternally);
                    } else {
                        valueBo = resolveDprSysInternal(userInfo, ruleDTO);
                        dprSysInternallyGroupMap.put(dprSysInternally, valueBo);
                    }
                    ruleDTO.setDprRuleValue(valueBo.getDprRuleValue());
                    ruleDTO.setDprRuleFieldType(valueBo.getDprRuleFieldType());
                    return ruleDTO;
                }).collect(Collectors.toList());
        return sysDprRoleApiDataRuleList;
    }

    private static DprRuleValueSysInternallyBo resolveDprSysInternal(GeneralUserDetails userInfo, SysDprRoleApiDataRuleListQueryDTO dprRule) {
        DprValueResolverSPI.DprRuleValue dprRuleValue = DprValueResolverFactory.resolveDprValue(dprRule, userInfo);
        Assert.notNull(dprRuleValue, "规则值为空");

        DprRuleValueSysInternallyBo dprRuleValueSysInternallyBo = new DprRuleValueSysInternallyBo();
        dprRuleValueSysInternallyBo.setDprRuleValue(normalizeDprRuleValue(dprRuleValue));
        dprRuleValueSysInternallyBo.setDprRuleValueDeclare(dprRuleValue.getValueType().showName());
        dprRuleValueSysInternallyBo.setDprRuleValueCode(dprRuleValue.getValueType().code());
        if (dprRuleValue.getFieldType() == null) {
            dprRuleValueSysInternallyBo.setDprRuleFieldType(dprRule.getDprRuleFieldType());
        } else {
            dprRuleValueSysInternallyBo.setDprRuleFieldType(dprRuleValue.getFieldType().name());
            dprRuleValueSysInternallyBo.setDprRuleFieldTypeName(dprRuleValue.getFieldType().getValueDescription());
        }

//        dprRuleValueSysInternallyBo.setDprRuleRelation(DprRuleRelationEnum.DPR_RULE_RELATION_AND.name());
//        dprRuleValueSysInternallyBo.setDprRuleRelationName(DprRuleRelationEnum.DPR_RULE_RELATION_AND.getValueDescription());
        dprRuleValueSysInternallyBo.setDprRuleField(dprRule.getDprRuleField());
//        dprRuleValueSysInternallyBo.setDprRuleFieldDeclare("数据归属用户ID");
//        dprRuleValueSysInternallyBo.setDprRuleCondition(DprRuleConditionEnum.Equal.name());
//        dprRuleValueSysInternallyBo.setDprRuleConditionName(DprRuleConditionEnum.Equal.getValueDescription());

        return dprRuleValueSysInternallyBo;
    }

    private static String normalizeDprRuleValue(DprValueResolverSPI.DprRuleValue ruleValue) {
        if (ruleValue.getValue() == null) {
            return NO_PERMISSION;
        }

        if (Collection.class.isAssignableFrom(ruleValue.getValue().getClass())) {
            if (CollUtil.isEmpty((Collection<?>) ruleValue.getValue())) {
                return NO_PERMISSION;
            }
            return StringUtils.collectionToDelimitedString((Collection<?>) ruleValue.getValue(), ",");
        }

        if (ruleValue.getValue().getClass().isArray()) {
            if (ArrayUtil.isEmpty((Object[]) ruleValue.getValue())) {
                return NO_PERMISSION;
            }
            return StringUtils.arrayToDelimitedString((Object[]) ruleValue.getValue(), ",");
        }

        return ruleValue.getValue().toString();
    }

    public static String normalizeRuleValue(Collection<?> values) {
        if (CollUtil.isEmpty(values)) {
            // 没有有效值，则返回-1，标识无权限
            return RoleAppApiDataPermissionUtil.NO_PERMISSION;
        }
        return StringUtils.collectionToDelimitedString(values, ",");
    }

    /**
     * 字段是否是ID类型
     *
     * @param fieldName
     * @return
     */
    public static boolean isIdField(String fieldName) {
        return fieldName != null && (fieldName.contains("id") || fieldName.contains("Id"));
    }

    /**
     * 字段是否是ID类型
     *
     * @param fieldName
     * @return
     */
    public static boolean isCodeField(String fieldName) {
        return fieldName != null && (fieldName.contains("code") || fieldName.contains("Code"));
    }

    /**
     * 系统内置的值
     *
     * @param userInfo
     * @param ruleList
     */
    public static void setRuleValueForInternal(GeneralUserDetails userInfo,
                                               List<SysDprRoleApiDataRuleListQueryDTO> ruleList) {
        if (CollUtil.isEmpty(ruleList)) {
            return;
        }
        Map<String, DprRuleValueSysInternallyBo> dprSysInternallyGroupMap = new HashMap<>(16);
        for (SysDprRoleApiDataRuleListQueryDTO ruleDTO : ruleList) {
            String dprSysInternally = ruleDTO.getDprRuleValue();
            //本人-下级等内置策略,不同的菜单权限只需要查询一次，没有必要多次循环，已经查询过了存储起来，下次直接使用。
            DprRuleValueSysInternallyBo valueBo;
            if (dprSysInternallyGroupMap.containsKey(dprSysInternally)) {
                valueBo = dprSysInternallyGroupMap.get(dprSysInternally);
            } else {
                valueBo = resolveDprSysInternal(userInfo, ruleDTO);
                dprSysInternallyGroupMap.put(dprSysInternally, valueBo);
            }
            ruleDTO.setDprRuleValue(valueBo.getDprRuleValue());
            ruleDTO.setDprRuleFieldType(valueBo.getDprRuleFieldType());
        }
    }

    /**
     * 业务数据集设置规则值
     */
    public static List<SysDprRoleApiDataRuleListQueryDTO> setBusinessSysDprRoleApiRuleValue(GeneralUserDetails userInfo,
                                                                                            List<SysDprRoleApiDataRuleListQueryBean> businessBean) {
        var beanValueList = businessBean.stream().map(sysDprRoleApiDataRuleListQueryBean -> {
            var ruleDTO = SysDprRoleApiRuleConvert.INSTANCE.beanToDto(sysDprRoleApiDataRuleListQueryBean);
            Assert.hasText(ruleDTO.getDprRuleValue(), "值空，请检查是否角色中配置了自定义值。Business规则：" + ruleDTO.getDprRuleId());
            return ruleDTO;
        }).collect(Collectors.toList());
        return beanValueList;
    }

    /**
     * 业务数据集的规则值处理
     *
     * @param userInfo
     * @param ruleList
     */
    public static void setRuleValueForBusinessSet(GeneralUserDetails userInfo, List<SysDprRoleApiDataRuleListQueryDTO> ruleList) {
        if (CollUtil.isEmpty(ruleList)) {
            return;
        }

        List<CompletableFuture<Void>> completableFutureList = new ArrayList<>(ruleList.size());
        TaskExecutor taskExecutor = SpringContextHolder.getBean(TaskExecutor.class);
        for (SysDprRoleApiDataRuleListQueryDTO rule : ruleList) {
            if (Boolean.FALSE.equals(rule.getRefResource())) {
                continue;
            }

            completableFutureList.add(CompletableFuture.runAsync(() -> {
                log.info("query public resource permission start ...");
                long start = System.currentTimeMillis();
                String value = queryRefBusinessResourceValue(rule, userInfo);
                log.info("public resource value:{}, {}, {}, {}", rule.getRoleCode(), rule.getDprRuleField(), rule.getRefBusinessObject(), value);
                log.info("query public resource permission end, cost {}ms", System.currentTimeMillis() - start);
                rule.setDprRuleValue(value);
            }, taskExecutor));
        }
        if (completableFutureList.isEmpty()) {
            return;
        }
        CompletableFuture.allOf(completableFutureList.toArray(CompletableFuture[]::new)).join();
    }

    /**
     * 自定义数据集设置规则值
     */
    public static List<SysDprRoleApiDataRuleListQueryDTO> setCustomSysDprRoleApiRuleValue(GeneralUserDetails userInfo,
                                                                                          List<SysDprRoleApiDataRuleListQueryBean> businessBean) {
        var beanValueList = businessBean.stream().map(sysDprRoleApiDataRuleListQueryBean -> {
            var ruleDTO = SysDprRoleApiRuleConvert.INSTANCE.beanToDto(sysDprRoleApiDataRuleListQueryBean);
            Assert.hasText(ruleDTO.getDprRuleValue(), "值空，请检查是否角色中配置了自定义值。Custom规则：" + ruleDTO.getDprRuleId());
            //设置自定义规则的规则值
            return ruleDTO;
        }).collect(Collectors.toList());
        return beanValueList;
    }

    /**
     * 设置权限规则中的值
     *
     * @param ruleList
     * @param userInfo
     */
    public static void fillDataPermissionValue(List<SysDprRoleApiDataRuleListQueryDTO> ruleList, GeneralUserDetails userInfo) {
        if (ruleList.isEmpty()) {
            return;
        }

        var valueTypeMap = ruleList.stream().filter(t -> !Boolean.TRUE.equals(t.getRuleGroup())).collect(Collectors.groupingBy(SysDprRoleApiDataRuleListQueryDTO::getDprRuleValueType));
        for (Map.Entry<String, List<SysDprRoleApiDataRuleListQueryDTO>> entry : valueTypeMap.entrySet()) {
            var type = DprRuleValueTypeEnum.parse(entry.getKey());
            Assert.notNull(type, "未知规则值类型：" + entry.getKey());
            switch (type) {
                case DPR_RULE_VALUE_TYPE_SYS:
                    // 系统内置
                    RoleAppApiDataPermissionUtil.setRuleValueForInternal(userInfo, entry.getValue());
                    break;
                case DPR_RULE_VALUE_TYPE_CUSTOM:
                    // 自定义
                    // 原始值，无需处理
                    break;
                case DPR_RULE_VALUE_TYPE_BUSINESS, DPR_RULE_VALUE_TYPE_COMP:
                    RoleAppApiDataPermissionUtil.setRuleValueForBusinessSet(userInfo, entry.getValue());
                    break;
                default:
                    throw new BusinessException("暂不支持的值类型：" + entry.getKey());
            }
        }
    }

    private static String queryRefBusinessResourceValue(SysDprRoleApiDataRuleListQueryDTO ruleDTO, GeneralUserDetails userInfo) {
        // 获取关联资源的权限规则
        Assert.hasText(ruleDTO.getRefBusinessObject(), "存在关联的业务对象为空：" + ruleDTO.getRoleCode());
        Assert.hasText(ruleDTO.getRoleCode(), "存在权限配置的角色编码为空");
        Assert.hasText(ruleDTO.getRefField(), "存在管理的业务对象字段为空");
        var refRuleList = SpringContextHolder.getBean(RoleDataPermissionRepoProc.class).queryForRefBusinessObjectResourceByRole(ruleDTO.getRoleCode(), ruleDTO.getRefBusinessObject());
        if (refRuleList.isEmpty()) {
            // 公共资源未配置，则无权限
            return NO_PERMISSION;
        }
        fillDataPermissionValue(refRuleList, userInfo);

        // 获取条件
        Map<String, List<String>> refRuleValues = new HashMap<>(refRuleList.size());
        Map<String, String> fieldTypeMap = new HashMap<>(refRuleList.size());
        boolean relationAnd = true;
        boolean valueEquals = true;
        for (SysDprRoleApiDataRuleListQueryDTO refRule : refRuleList) {
            if (CharSequenceUtil.isBlank(refRule.getDprRuleField()) || CharSequenceUtil.isBlank(refRule.getDprRuleValue())
                    || CharSequenceUtil.isBlank(refRule.getDprRuleFieldType())) {
                log.warn("存在数据权限规则字段或值为空：{}，{}，{}，{}", refRule.getRoleCode(), refRule.getDprRuleField(), refRule.getDprRuleFieldType(),
                        refRule.getDprRuleValue());
                continue;
            }
            if (NO_PERMISSION.equals(refRule.getDprRuleValue())) {
                if (DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_SYS.name().equals(refRule.getDprRuleValueType()) ||
                        DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_BUSINESS.name().equals(refRule.getDprRuleValueType()) ||
                        DprRuleValueTypeEnum.DPR_RULE_VALUE_TYPE_COMP.name().equals(refRule.getDprRuleValueType())) {
                    // 无权限
                    return NO_PERMISSION;
                }
            }
            fieldTypeMap.put(refRule.getDprRuleField(), refRule.getDprRuleFieldType());

            List<String> values = refRuleValues.getOrDefault(refRule.getDprRuleField(), new ArrayList<>(0));
            if (refRule.getDprRuleValue().contains(",")) {
                values.addAll(Arrays.stream(refRule.getDprRuleValue().split(",")).collect(Collectors.toList()));
            } else {
                values.add(refRule.getDprRuleValue());
            }
            refRuleValues.put(refRule.getDprRuleField(), values);

            relationAnd = DprRuleRelationEnum.DPR_RULE_RELATION_AND.name().equals(refRule.getDprRuleRelation());
            valueEquals = DprRuleConditionEnum.Equal.name().equals(refRule.getDprRuleCondition()) || DprRuleConditionEnum.InList.name().equals(refRule.getDprRuleCondition());
        }
        Map<String, Object> conditions = new HashMap<>(refRuleValues.size());
        for (Map.Entry<String, List<String>> entry : refRuleValues.entrySet()) {
            if (entry.getValue().contains(NO_PERMISSION)) {
                return NO_PERMISSION;
            }
            if (entry.getValue().size() == 1) {
                conditions.put(entry.getKey(), ObjUtil.convertBasicValue(entry.getValue().get(0), fieldTypeMap.get(entry.getKey())));
                continue;
            }
            conditions.put(entry.getKey(), entry.getValue().stream().map(t -> ObjUtil.convertBasicValue(t, fieldTypeMap.get(entry.getKey()))).collect(Collectors.toList()));
        }

        UnifyQueryQueryDTO queryDTO = new UnifyQueryQueryDTO();
        queryDTO.setBusinessObjectCode(ruleDTO.getRefBusinessObject());
        queryDTO.setQueryFields(List.of(ruleDTO.getRefField()));
        queryDTO.setConditions(conditions);
        queryDTO.setConditionFieldTypes(fieldTypeMap);
        queryDTO.setAnd(relationAnd);
        queryDTO.setEquals(valueEquals);

        var supportTenant = SpringContextHolder.getBean(BusinessObjectRepoProc.class).getSupportTenant(queryDTO.getBusinessObjectCode());
        var unifyQueryService = SpringContextHolder.getBean(SysUnifyQueryQueryService.class);
        var unifyQueryData = Boolean.TRUE.equals(supportTenant) ? unifyQueryService.query(queryDTO).computeData() :
                SpringContextHolder.getBean(TenantDataIsolateProvider.class).byDefaultDirectly(() -> unifyQueryService.query(queryDTO)).computeData();
        Assert.isTrue(unifyQueryData.getSuccess(), unifyQueryData.getFailMsg());
        return obtainValueFromUnifyQueryData(ruleDTO.getRefField(), unifyQueryData.getData());
    }

    private static String obtainValueFromUnifyQueryData(String field, Object data) {
        if (data == null) {
            return NO_PERMISSION;
        }
        if (data instanceof List) {
            var values = ((List<?>) data).stream()
                    .map(t -> {
                        if (t instanceof Map) {
                            return ((Map<?, ?>) t).get(field);
                        }
                        return t;
                    }).filter(t -> {
                        if (t == null) {
                            return false;
                        }
                        if (t instanceof String) {
                            return CharSequenceUtil.isNotBlank((String) t);
                        }
                        return true;
                    }).collect(Collectors.toList());
            if (values.isEmpty()) {
                return NO_PERMISSION;
            }
            return normalizeRuleValue(values);
        }
        log.error("解析权限配置数据：{}", JSONUtil.toJsonString(data));
        throw new BusinessException("解析权限配置异常");
    }
}
