package com.elitescloud.cloudt.system.modules.dpr.strategy;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.common.param.CodeNameParam;
import com.elitescloud.boot.datasecurity.common.extension.DprValueResolverSPI;
import com.elitescloud.boot.datasecurity.common.extension.DprValueType;
import com.elitescloud.boot.spi.support.ProviderInstanceHolder;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.dto.SysDprRoleApiDataRuleListQueryDTO;
import com.elitescloud.cloudt.system.modules.dpr.SysDprValueType;
import com.elitescloud.cloudt.system.modules.dpr.strategy.impl.DprSysInternallyDynamic;
import org.beetl.core.util.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import javax.validation.constraints.NotNull;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 数据权限规则值的处理器工厂.
 *
 * @author Kaiser（wang shao）
 * @date 2024/11/23
 */
public class DprValueResolverFactory {
    private static final Logger logger = LoggerFactory.getLogger(DprValueResolverFactory.class);

    /**
     * 是否已初始化
     */
    private static volatile boolean initialized = false;
    private static final Map<String, DprValueResolverSPI> valueResolverSpiMap = new HashMap<>(32);
    private static final Map<String, DprValueType> valueTypeMap = new HashMap<>(32);

    /**
     * 获取支持的值类型
     *
     * @return 值类型
     */
    public static List<CodeNameParam> getValueTypes() {
        if (!initialized) {
            init();
        }

        return valueTypeMap.values().stream()
                .sorted(Comparator.comparing(DprValueType::sortNo))
                .map(t -> new CodeNameParam(t.code(), t.showName()))
                .collect(Collectors.toList());
    }

    /**
     * 解析值类型
     *
     * @param code 值类型编码
     * @return 值类型
     */
    public static DprValueType parseValueType(String code) {
        if (!initialized) {
            init();
        }

        return code == null ? null : valueTypeMap.get(code);
    }

    /**
     * 将值类型编码转换为值类型名称
     *
     * @param code 值类型编码
     * @return 值类型名称
     */
    public static String convertDprValueTypeName(String code) {
        if (!initialized) {
            init();
        }
        DprValueType valueType = parseValueType(code);
        return valueType == null ? null : valueType.showName();
    }

    /**
     * 处理数据权限规则值
     *
     * @param rule        数据权限规则
     * @param userDetails 用户详细信息
     * @return 规则值
     */
    public static DprValueResolverSPI.DprRuleValue resolveDprValue(@NotNull SysDprRoleApiDataRuleListQueryDTO rule, @NotNull GeneralUserDetails userDetails) {
        if (!initialized) {
            init();
        }

        Assert.notNull(rule, "数据权限规则为空");
        Assert.notNull(userDetails, "当前用户信息为空");

        // 获取数据权限规则解析器
        String ruleValueType = CharSequenceUtil.blankToDefault(rule.getDprRuleValue(), rule.getDprSysInternally());
        logger.info("开始解析数据权限规则：{}, {}", rule.getDprRuleId(), ruleValueType);
        Assert.hasText(ruleValueType, "未知规则值类型");
        if (ruleValueType.startsWith(DprSysInternallyDynamic.PREFIX)) {
            ruleValueType = SysDprValueType.DPR_SYS_INTERNALLY_DYNAMIC.code();
        }

        DprValueResolverSPI resolverSPI = valueResolverSpiMap.get(ruleValueType);
        if (resolverSPI == null) {
            logger.error("未发现规则值{}的解析器", ruleValueType);
            return null;
        }

        // 开始解析
        var valueType = valueTypeMap.get(ruleValueType);
        var value = resolverSPI.resolve(valueType, userDetails, rule);
        Assert.notNull(value, "数据权限规则解析的值为空");
        value.setValueType(valueType);

        logger.info("解析得结果：{}", value.getValue());
        return value;
    }

    private static void init() {
        if (initialized) {
            return;
        }

        synchronized (DprValueResolverFactory.class) {
            if (initialized) {
                return;
            }

            // 通过SPI方式加载
            List<DprValueResolverSPI> spiList = ProviderInstanceHolder.loadProviderInstances(DprValueResolverSPI.class, false);
            if (!spiList.isEmpty()) {
                for (DprValueResolverSPI dprValueResolverSPI : spiList) {
                    var valueTypes = dprValueResolverSPI.valueTypes();
                    if (ArrayUtils.isEmpty(valueTypes)) {
                        logger.error("{}未配置值类型，忽略", dprValueResolverSPI.getClass().getName());
                        continue;
                    }

                    for (DprValueType valueType : valueTypes) {
                        Assert.hasText(valueType.code(), () -> valueType.getClass().getName() + "存在值类型编码为空");

                        if (valueResolverSpiMap.containsKey(valueType.code())) {
                            logger.error("{}存在重复，忽略", valueType.code());
                            continue;
                        }

                        valueResolverSpiMap.put(valueType.code(), dprValueResolverSPI);
                        valueTypeMap.put(valueType.code(), valueType);
                    }
                }
            }
            initialized = true;
        }
    }
}
