package com.elitescloud.cloudt.system.modules.orgtree.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.core.base.UdcProvider;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.elitescloud.cloudt.system.dto.SysOrgBasicDTO;
import com.elitescloud.cloudt.system.modules.orgtree.common.ConvertUtil;
import com.elitescloud.cloudt.system.modules.orgtree.common.OrgUdcEnum;
import com.elitescloud.cloudt.system.modules.orgtree.common.UserUtil;
import com.elitescloud.cloudt.system.modules.orgtree.convert.OrgBuTreeDConvert;
import com.elitescloud.cloudt.system.modules.orgtree.model.OrgBuTreeDDTO;
import com.elitescloud.cloudt.system.modules.orgtree.model.OrgBuTreeDTO;
import com.elitescloud.cloudt.system.modules.orgtree.model.SystemBuTreeDTO;
import com.elitescloud.cloudt.system.modules.orgtree.model.entity.OrgBuTreeDDO;
import com.elitescloud.cloudt.system.modules.orgtree.model.param.OrgBuTreeParam;
import com.elitescloud.cloudt.system.modules.orgtree.model.param.OrgBuTreeVDSaveParam;
import com.elitescloud.cloudt.system.modules.orgtree.model.param.OrgBuTreeVListParam;
import com.elitescloud.cloudt.system.modules.orgtree.service.OrgBuTreeVersionDomainService;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.OrgBuTreeDRepo;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.OrgBuTreeDRepoProc;
import com.elitescloud.cloudt.system.modules.orgtree.service.repo.OrgBuTreeRepoProc;
import com.elitescloud.cloudt.system.param.SysOrgQueryDTO;
import com.elitescloud.cloudt.system.service.ISysSettingService;
import com.elitescloud.cloudt.system.service.OrgQueryService;
import com.elitescloud.cloudt.system.vo.SysSettingVO;
import com.elitescloud.cloudt.ucenter.utils.TreeUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RedissonClient;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * <p>
 * 功能说明
 * </p>
 *
 * @author Tristan
 * @date 2020/7/2
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class OrgBuTreeVersionDomainServiceImpl implements OrgBuTreeVersionDomainService {

    private final ISysSettingService settingService;
    private final OrgBuTreeRepoProc orgBuTreeRepoProc;
    private final OrgBuTreeDRepoProc orgBuTreeDRepoProc;
    private final OrgBuTreeDRepo orgBuTreeDRepo;


    private final UdcProvider sysUdcService;
    private final RedissonClient redissonClient;
    private final RedisUtils redisUtils;
    private final OrgQueryService rmiOrgOuService;

    /**
     * 根据组织树编码获取系统域生效的版本号 如果系统域没有配置则以最新生效的版本  如果系统域已经配置 以系统域配置的版本为准
     * 影响的页面：商品经营目录组织树，商品冻结选择冻结范围，数据权限，同步激荡云
     * 不影响：员工页面的所有组织树选项（全部以生效的为准），组织树履历展现当前的生效的组织也不受此影响
     *
     * @param systemCode 系统参数
     * @return 版本号
     */
    @Override
    public SystemBuTreeDTO findActiveVersionByBuTreeCode(String systemCode) {
        log.debug("获取组织树当前版本-参数：" + systemCode);
        //  获取系统域的配置参数
        SysSettingVO sysSettingByNo = settingService.oneByNo(systemCode);
        log.debug("获取组织树当前版本-获取系统域配置：" + sysSettingByNo);
        if (null != sysSettingByNo) {
            String settingVal = sysSettingByNo.getSettingVal();
            // 分割字符串 获取配置的版本号

            List<String> strings = Arrays.asList(settingVal.split("-"));

            String buTreeCode = null;
            String configVersion = null;
            // 如果进行了组织树版本配置
            if (strings.size() == 2) {
                buTreeCode = strings.get(0);
                configVersion = strings.get(1);
            }
            // 只配置了组织树编码 并没有配置组织树版本
            else if (strings.size() == 1) {
                buTreeCode = strings.get(0);
            }
            // 错误的长度 直接抛出错误
            else {
                log.debug("获取组织树当前版本-参数格式配置错误，请检查系统设置：" + systemCode + ";");
                throw new BusinessException("参数格式配置错误，请检查系统设置：" + systemCode + ";");
            }


            List<OrgBuTreeDTO> buTreeListByParam = orgBuTreeRepoProc.getBuTreeListByBuTreeCode(OrgBuTreeVListParam.builder().buTreeCode(buTreeCode).build());
            if (CollectionUtils.isEmpty(buTreeListByParam)) {
                log.debug("获取组织树当前版本-组织树编码：" + buTreeCode + "不存在");
                throw new BusinessException("组织树编码：" + buTreeCode + "不存在");
            }
            // 获取当前 已停用/生效的版本数据
            List<String> versionList = buTreeListByParam.stream().filter(param -> {
                String buTreeStatus = param.getBuTreeStatus();
                return OrgUdcEnum.ORG_BUTREE_STATUS_ACTIVE.getUdcVal().equals(buTreeStatus)
                        || OrgUdcEnum.ORG_BUTREE_STATUS_CLOSED.getUdcVal().equals(buTreeStatus);
            }).map(OrgBuTreeDTO::getNowVersion).collect(Collectors.toList());
            // 判断系统域配置的数据 是否存在于当前组织版本中
            if (StringUtils.isNotBlank(configVersion) && versionList.contains(configVersion)) {
                SystemBuTreeDTO build = SystemBuTreeDTO.builder().buTreeCode(buTreeCode).version(configVersion).build();
                log.debug("获取组织树当前版本-系统域配置的信息：" + build);
                this.checkBuTreeVersionStatus(buTreeCode, configVersion, true);
                return build;
            } else {
                String nowBuTreeVersion = this.getNowBuTreeVersion(buTreeCode);
                SystemBuTreeDTO build = SystemBuTreeDTO.builder().buTreeCode(buTreeCode).version(nowBuTreeVersion).build();
                log.debug("获取组织树当前版本-支撑域当前生效配置：" + build);
                this.checkBuTreeVersionStatus(buTreeCode, nowBuTreeVersion, false);
                return build;
            }
        } else {
            log.debug("获取组织树当前版本-系统参数中并未配置参数：" + systemCode);
            throw new BusinessException("获取组织树当前版本-系统参数中并未配置参数：" + systemCode);
        }
    }

    /**
     * 保存组织树版本节点信息
     *
     * @param buTreeDList 节点信息
     * @param buTreeId    组织树id
     */
    @Override
    @CacheEvict(cacheNames = "ORG_BU_TREE", key = "'BU_TREE_ID_' + #buTreeId", condition = "#buTreeId != null")
    public void saveBuTreeDV(List<OrgBuTreeVDSaveParam> buTreeDList, Long buTreeId, Boolean updateFlag) {
        // 如果id不为空 认定为编辑  全删除树节点
        if (updateFlag) {
            orgBuTreeDRepo.deleteByBuTreeId(buTreeId);
        }

        // 如果树节点为空 直接返回
        if (CollectionUtils.isEmpty(buTreeDList)) {
            return;
        }

        // 开始将tree  转为 list
        List<OrgBuTreeVDSaveParam> treeToList = new ArrayList<>();
        TreeUtils.treeToList(buTreeDList, treeToList, 0L);
        // 数据格式转换
        List<OrgBuTreeDDO> orgBuTreeDDOS = OrgBuTreeDConvert.INSTANCE.saveListToDoList(treeToList);


        // 校验是否存在重复的组织
        Set<String> collect = orgBuTreeDDOS.stream().collect(Collectors.collectingAndThen(
                Collectors.groupingBy(OrgBuTreeDDO::getBuCode, Collectors.counting()),
                map -> map.entrySet().stream().filter(maps -> maps.getValue() > 1).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)).keySet())
        );
//        if (CollectionUtils.isNotEmpty(collect)) {
//            List<OrgBuDTO> buDtoByParam = orgBuService.findBuDtoByParam(OrgBuDtoParam.builder().buCodes(new ArrayList<>(collect)).build());
//            if (CollectionUtils.isNotEmpty(buDtoByParam)) {
//                String repeatStr = buDtoByParam.stream().map(dto -> dto.getBuName() + "-" + dto.getBuCode()).collect(Collectors.joining(","));
//                throw new BusinessException("组织节点重复：" + repeatStr);
//            }
//        }


        Map<Long, OrgBuTreeDDO> idMap = orgBuTreeDDOS.stream().collect(Collectors.toMap(OrgBuTreeDDO::getId, t -> t));
        // 开始设置 组织节点路径 与 层级
        orgBuTreeDDOS.forEach(treeD -> {
            // 设置组织树id
            treeD.setBuTreeId(buTreeId);
            List<String> codePaths = new ArrayList<>();
            // 递归 获取codePath
            recursionStart(idMap, treeD.getPid(), codePaths);
            //反转列表
            Collections.reverse(codePaths);
            String join = String.join("/", codePaths);
            // 如果为空 则表示为根节点
            if (StringUtils.isBlank(join)) {
                treeD.setCodePath(String.valueOf(treeD.getBuCode()));
                treeD.setLevel(1);
            }
            // 不为空则表示为非空节点
            else {
                treeD.setCodePath(join + "/" + treeD.getBuCode());
                treeD.setLevel(treeD.getCodePath().split("/").length);
            }
        });
        List<OrgBuTreeDDO> tempList = new ArrayList<>();
        try {
            for (OrgBuTreeDDO orgBuTreeDDO : orgBuTreeDDOS) {
                tempList.add(orgBuTreeDDO.clone());
            }
        } catch (CloneNotSupportedException e) {
            throw new BusinessException("深克隆对象失败");
        }
        // 保存新节点 需要将id设置为空
        orgBuTreeDDOS.forEach(dos -> dos.setId(null));

        // 首先保存组织树节点信息  获取到各自的id
        List<OrgBuTreeDDO> newTreeDDOs = orgBuTreeDRepo.saveAll(orgBuTreeDDOS);
        // 构建联系
        Map<Long, Long> connectMap = connect(tempList, newTreeDDOs);
        // 开始设置pid
        newTreeDDOs.forEach(b -> {
            Long pid = connectMap.get(b.getPid());
            if (null != pid) {
                b.setPid(pid);
            }
        });

        // 更新 pid 至数据库
        orgBuTreeDRepo.saveAll(newTreeDDOs);
    }

    @Override
//    @Cacheable(cacheNames = "ORG_BU_TREE", key = "'BU_TREE_ID_' + #buTreeId+'#20'", condition = "#buTreeId != null")
    public List<OrgBuTreeDDTO> getOrgBuTreeDByCache(Long buTreeId, String searchType, Long buTreeDId) {
        String redisKey = "ORG_BU_TREEBU_TREE_ID_" + buTreeId;
        Object result = redisUtils.get(redisKey);
        if (null != result) {
            return ConvertUtil.castList(result, OrgBuTreeDDTO.class);
        } else {
            // var treeVos = orgBuTreeDRepoProc.searchByBuTreeId(buTreeId, searchType);
            var treeVos = getOrgBuTreeDFromSystem(buTreeId, searchType, buTreeDId);
            if (CollectionUtils.isNotEmpty(treeVos)) {
                // var udcMap = sysUdcService.getValueMapByUdcCode("yst-supp", "BU_TYPE");
                // treeVos.forEach(s -> s.setBuTypeName(orgBuService.buTypeConvert(s.getBuType(), udcMap)));
                List<OrgBuTreeDDTO> orgBuTreeDVOS = TreeUtils.buildTree(buTreeDId, true, treeVos);
                redisUtils.set(redisKey, orgBuTreeDVOS, 1, TimeUnit.DAYS);
                return orgBuTreeDVOS;
            } else {
                return Collections.emptyList();
            }
        }


    }

    @Override
    public List<OrgBuTreeDDTO> getOrgBuTreeDFromSystem(Long buTreeId, String searchType, Long buTreeDId) {
        // 查询组织树详情
        Optional<List<OrgBuTreeDDO>> byBuTreeId = orgBuTreeDRepo.findByBuTreeId(buTreeId);
        if (byBuTreeId.isEmpty()) {
            return Collections.emptyList();
        }
        List<OrgBuTreeDDO> orgBuTreeDDOS = byBuTreeId.get();
        List<OrgBuTreeDDTO> treeDtos = OrgBuTreeDConvert.INSTANCE.doToTreeDto(orgBuTreeDDOS);

        Map<Long, OrgBuTreeDDTO> buIdMap = treeDtos.stream().collect(Collectors.toMap(OrgBuTreeDDTO::getBuId, t -> t, (t1, t2) -> t1));

        // 系统域查询组织信息
        Set<Long> buIdSet = buIdMap.keySet();
        SysOrgQueryDTO queryDTO = new SysOrgQueryDTO();
        queryDTO.setIds(buIdSet);
        List<SysOrgBasicDTO> sysOrgBasicDTOS = rmiOrgOuService.queryList(queryDTO).computeData();
        var udcMap = sysUdcService.getValueMapByUdcCode("cloudt-system", "orgType");
        // 设置组织信息
        if (CollectionUtils.isNotEmpty(sysOrgBasicDTOS)) {
            sysOrgBasicDTOS.stream().forEach(item -> {
                if (buIdMap.containsKey(item.getId())) {
                    OrgBuTreeDDTO dto = buIdMap.get(item.getId());
                    dto.setBuCode(item.getCode());
                    dto.setBuName(item.getName());
                    dto.setBuType(item.getType());
                    dto.setBuAbbr(item.getShortName());
                    if (ObjectUtil.isNotNull(item.getEnabled())) {
                        dto.setBuStatus(Boolean.toString(item.getEnabled()));
                        dto.setBuStatusName(item.getEnabled() ? OrgUdcEnum.BU_STATUS_ACTIVE.getValDesc() : OrgUdcEnum.BU_STATUS_CLOSED.getValDesc());
                    }
                    dto.setBuTypeName(udcMap.get(item.getType()));
                }
            });
        }

        var treeVos = buIdMap.values().stream().collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(treeVos)) {
            List<OrgBuTreeDDTO> orgBuTreeDVOS = TreeUtils.buildTree(buTreeDId, true, treeVos);
            return orgBuTreeDVOS;
        } else {
            return Collections.emptyList();
        }
    }

    /**
     * 构建id对应关系
     *
     * @param from
     * @param to
     * @return
     */
    private Map<Long, Long> connect(List<OrgBuTreeDDO> from, List<OrgBuTreeDDO> to) {
        Map<Long, Long> map = new HashMap<>(16);
        int size = from.size();

        for (int i = 0; i < size; i++) {
            map.put(from.get(i).getId(), to.get(i).getId());
        }

        return map;
    }

    /**
     * 递归 获取codePath
     *
     * @param collect
     * @param pid
     * @param codePaths
     * @return
     */
    private void recursionStart(Map<Long, OrgBuTreeDDO> collect, Long pid, List<String> codePaths) {
        OrgBuTreeDDO treeClass = collect.get(pid);
        if (null != treeClass) {
            codePaths.add(String.valueOf(treeClass.getBuCode()));
            recursionStart(collect, treeClass.getPid(), codePaths);
        }
    }

    /**
     * 根据组织树编码 buTreeCode  获取目前失效的组织树版本
     *
     * @param buTreeCode 组织树编码
     * @return 组织树版本（当前生效）
     */
    private String getNowBuTreeVersion(String buTreeCode) {
        UserUtil.getNowUser();
        // 如果系统域的配置参数没有获取到  直接获取支撑域当前生效的组织树版本号
        List<String> activeVersionByBuTreeCode = orgBuTreeRepoProc.findActiveVersionByBuTreeCode(buTreeCode);
        if (CollectionUtils.isNotEmpty(activeVersionByBuTreeCode)) {
            if (activeVersionByBuTreeCode.size() > 1) {
                throw new BusinessException("组织树:" + buTreeCode + ",存在多个生效版本");
            } else {
                return activeVersionByBuTreeCode.get(0);
            }
        } else {
            throw new BusinessException("组织中心商品经营目录树" + buTreeCode + "无生效版本，请检查组织树状态。");
        }
    }

    /**
     * 根据组织树编码和版本 判断是否正常状态
     *
     * @param buTreeCode 组织树编码
     * @param version    版本
     * @param isSystem   判断是否为系统参数配置的组织树版本 （用于报错信息的返回）
     */
    public void checkBuTreeVersionStatus(String buTreeCode, String version, boolean isSystem) {
        // 根据组织树编码和版本获取组织树数据
        OrgBuTreeParam build = OrgBuTreeParam.builder().buTreeCode(buTreeCode).nowVersion(version).build();
        List<OrgBuTreeDTO> orgBuTreeVoByParam = orgBuTreeRepoProc.findOrgBuTreeVoByParam(build);
        //
        if (CollectionUtils.isNotEmpty(orgBuTreeVoByParam)) {
            if (orgBuTreeVoByParam.size() > 1) {
                throw new BusinessException("组织树:" + buTreeCode + ",存在多个版本：V" + version);
            }
            // 开始对数据进行校验
            OrgBuTreeDTO orgBuTreeVO = orgBuTreeVoByParam.get(0);
            if (!(OrgUdcEnum.ORG_BUTREE_STATUS_ACTIVE.getUdcVal().equals(orgBuTreeVO.getBuTreeStatus()) && orgBuTreeVO.getIsNowVersion())) {
                if (isSystem) {
                    throw new BusinessException("系统配置参数为:组织树" + buTreeCode + "版本V" + version + "，该版本为已停用状态，请检查系统配置参数。");
                } else {
                    throw new BusinessException("组织中心商品经营目录树" + buTreeCode + "无生效版本，请检查组织树状态。");
                }
            }
        } else {
            throw new BusinessException("组织树：" + buTreeCode + "版本V" + version + ",不存在，请检查数据");
        }
    }
}
