package com.elitescloud.cloudt.system.modules.wecom.controller;

import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.auth.model.OAuthToken;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.constant.SysThirdPartyAccountBusinessType;
import com.elitescloud.cloudt.system.modules.wecom.model.login.WecomLoginPropsVO;
import com.elitescloud.cloudt.system.modules.wecom.service.WecomAuthService;
import com.elitescloud.cloudt.system.modules.wecom.util.crypt.AesException;
import com.elitescloud.cloudt.system.modules.wecom.util.crypt.WXBizMsgCrypt;
import com.elitescloud.cloudt.system.service.ThirdPartAccountQueryService;
import com.elitescloud.cloudt.system.service.common.constant.ThirdPartAccountType;
import com.elitescloud.cloudt.system.service.common.constant.ThirdPartyConstant;
import com.github.xiaoymin.knife4j.annotations.ApiOperationSupport;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotBlank;

/**
 * .
 *
 * @author Kaiser（wang shao）
 * @date 2024/12/1
 */
@Api(tags = "企微认证")
@RestController
@RequestMapping(value = "/oauth/wecom")
@Validated
public class WecomAuthController {
    private static final Logger logger = LoggerFactory.getLogger(WecomAuthController.class);

    @Autowired
    private WecomAuthService service;

    @ApiOperation(value = "供企微验证服务", hidden = true)
    @GetMapping(value = "/verify")
    public String validateServer(@RequestParam(name = "msg_signature", required = false) String msg_signature,
                                 @RequestParam(name = "timestamp", required = false) String timestamp,
                                 @RequestParam(name = "nonce", required = false) String nonce,
                                 @RequestParam(name = "echostr", required = false) String echostr,
                                 HttpServletRequest request) {
        logger.info("随机字符串：{}", echostr);
        logger.info("请求：{}", request.getQueryString());

        var tenant = SpringContextHolder.getBean(TenantClientProvider.class).getSessionTenant();
        var accountCfg = SpringContextHolder.getBean(ThirdPartAccountQueryService.class).get(tenant == null ? TenantConstant.DEFAULT_TENANT_ID : tenant.getId(),
                ThirdPartAccountType.WECOM, SysThirdPartyAccountBusinessType.DEFAULT).computeData();
        if (accountCfg == null) {
            logger.error("未配置企微信息");
            return "fail";
        }

        String sToken = (String) accountCfg.getConfigValue(ThirdPartyConstant.CFG_WECOM_TOKEN);
        String sEncodingAESKey = (String) accountCfg.getConfigValue(ThirdPartyConstant.CFG_WECOM_ENCODING_AES_KEY);
        String res = null;
        try {
            WXBizMsgCrypt wxcpt = new WXBizMsgCrypt(sToken, sEncodingAESKey, (String) accountCfg.getConfigValue(ThirdPartyConstant.CFG_WECOM_CORPID));
            res = wxcpt.VerifyURL(msg_signature, timestamp, nonce, echostr);
        } catch (Exception e) {
            logger.error("验证失败", e);
        }
        logger.info("返回：{}", res);
        return res;
    }

    @ApiOperation(value = "获取登录配置")
    @ApiOperationSupport(order = 1)
    @GetMapping(value = "/props")
    public ApiResult<WecomLoginPropsVO> getLoginProps() {
        return service.getLoginProps();
    }

    @ApiOperation(value = "获取认证地址")
    @ApiOperationSupport(order = 2)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "redirectUrl", value = "认证成功后返回至前端的url路径", required = true),
            @ApiImplicitParam(name = "state", value = "自定义参数"),
    })
    @GetMapping(value = "/authorizeUrl")
    public ApiResult<String> getAuthorizeUrl(@NotBlank(message = "认证后的回调地址") @RequestParam(name = "redirectUrl") String redirectUrl,
                                             @RequestParam(name = "state", required = false) String state) {
        return service.getAuthorizeUrl(redirectUrl, state);
    }

    @ApiOperation(value = "根据授权码获取token")
    @ApiOperationSupport(order = 3)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "code", value = "授权码", required = true),
    })
    @GetMapping(value = "/code2Token")
    public ApiResult<OAuthToken> code2Token(HttpServletRequest request, HttpServletResponse response,
                                            @NotBlank(message = "授权码为空") String code) {
        return service.code2Token(request, response, code);
    }
}
