package com.elitescloud.cloudt.system.provider.orgsync;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.model.entity.BaseTreeModel;
import com.elitescloud.cloudt.system.dto.req.EmployeeQueryDTO;
import com.elitescloud.cloudt.system.modules.wecom.model.department.DepartmentIdListResult;
import com.elitescloud.cloudt.system.modules.wecom.model.department.DepartmentSaveParam;
import com.elitescloud.cloudt.system.modules.wecom.util.WeComTool;
import com.elitescloud.cloudt.system.service.callback.OrgChangedCallback;
import com.elitescloud.cloudt.system.service.common.constant.SyncDataType;
import com.elitescloud.cloudt.system.service.manager.SysSyncManager;
import com.elitescloud.cloudt.system.service.model.bo.SysOrgSaveBO;
import com.elitescloud.cloudt.system.service.model.bo.SysSyncSaveBO;
import com.elitescloud.cloudt.system.service.model.entity.SysOrgDO;
import com.elitescloud.cloudt.system.service.repo.EmployeeRepoProc;
import com.elitescloud.cloudt.system.service.repo.OrgRepoProc;
import com.elitescloud.cloudt.system.service.repo.SyncRepoProc;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.task.TaskExecutor;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 企微通讯录的同步.
 *
 * @author Kaiser（wang shao）
 * @date 2024/8/26
 */
@Component
public class WecomOrgSync extends BaseWecomSync implements OrgChangedCallback {
    private static final Logger logger = LoggerFactory.getLogger(WecomOrgSync.class);

    @Autowired
    private OrgRepoProc orgRepoProc;
    @Autowired
    private EmployeeRepoProc employeeRepoProc;
    @Autowired
    private SyncRepoProc syncRepoProc;
    @Autowired
    private SysSyncManager sysSyncManager;
    @Autowired
    private TaskExecutor taskExecutor;

    @Override
    public void onUpsert(boolean add, SysOrgSaveBO saveBO, SysOrgDO orgDO) {
        if (add && Boolean.FALSE.equals(orgDO.getSyncOuter()) && Boolean.TRUE.equals(orgDO.getExecutive())) {
            // 新增且不需要同步
            return;
        }

        taskExecutor.execute(() -> {
            // 获取企微配置
            var cfg = getConfig();
            if (cfg == null) {
                logger.info("企微未配置，忽略同步");
                return;
            }

            // 不需要同步，判断是否已同步，如果同步，则删除
            if (Boolean.FALSE.equals(orgDO.getSyncOuter())) {
                this.delete(orgDO.getId(), cfg);
                return;
            }

            // 同步企微信息
            this.update(orgDO, cfg);
        });
    }

    @Override
    public void onEnabled(Long orgId, boolean enabled) {

    }

    @Override
    public void onDelete(Long orgId) {
        taskExecutor.execute(() -> {
            // 获取企微配置
            var cfg = getConfig();
            if (cfg == null) {
                logger.info("企微未配置，忽略同步");
                return;
            }

            // 删除企微信息
            this.delete(orgId, cfg);
        });
    }

    /**
     * 删除下面所有组织
     *
     * @param orgId
     */
    public void deleteAllOrg(Long orgId) {
        taskExecutor.execute(() -> {
            // 获取企微配置
            var cfg = getConfig();
            if (cfg == null) {
                logger.info("企微未配置，忽略同步");
                return;
            }

            // 删除企微信息
            var childrenIds = orgId == null ? orgRepoProc.allId() : orgRepoProc.getChildrenIdByPid(orgId);
            if (childrenIds.isEmpty()) {
                if (orgId == null) {
                    deleteAllByWecom(cfg);
                }
                return;
            }
            childrenIds = childrenIds.stream().sorted(Comparator.comparingLong(t -> (long) t).reversed()).collect(Collectors.toList());
            for (Long childrenId : childrenIds) {
                this.delete(childrenId, cfg);
            }

            if (orgId == null) {
                deleteAllByWecom(cfg);
            }
        });
    }

    private void deleteAllByWecom(WecomConfig cfg) {
        var token = getAccessTokenOfWecom(cfg);
        var departmentIdList = WeComTool.departmentIdList(token, null);
        if (!departmentIdList.isSuccess()) {
            throw new BusinessException("删除企微组织失败，" + departmentIdList.getErrcode() + ":" + departmentIdList.getErrmsg());
        }

        if (CollUtil.isNotEmpty(departmentIdList.getDepartment_id())) {
            for (var departmentId : departmentIdList.getDepartment_id()) {
                var deleteDepartmentResult = WeComTool.departmentDelete(token, departmentId.getId());
                if (!deleteDepartmentResult.isSuccess()) {
                    logger.info("删除企微组织失败，" + departmentIdList.getErrcode() + ":" + departmentIdList.getErrmsg());
                }
            }
        }
    }

    private void delete(long orgId, WecomConfig cfg) {
        // 判断是否已同步
        var syncInfo = syncRepoProc.getIdCode(SyncDataType.ORG.name(), orgId + "", cfg.getCorpid());
        if (syncInfo == null) {
            logger.info("未向企业微信同步，则忽略删除");
            return;
        }
        logger.info("删除企微的组织：{}", orgId);

        // 删除本地同步记录
        sysSyncManager.delete(syncInfo.getId());

        // 同步企业微信
        if (!StringUtils.hasText(syncInfo.getCode())) {
            return;
        }
        var token = getAccessTokenOfWecom(cfg);
        var syncResult = WeComTool.departmentDelete(token, Long.valueOf(syncInfo.getCode()));
        if (!syncResult.isSuccess()) {
            throw new BusinessException("删除企微组织失败，" + syncResult.getErrcode() + ":" + syncResult.getErrmsg());
        }
    }

    private void update(SysOrgDO orgDOUpdate, WecomConfig cfg) {
        var orgId = orgDOUpdate.getId();
        var syncInfo = syncRepoProc.getIdCode(SyncDataType.ORG.name(), orgId + "", cfg.getCorpid());
        var syncId = syncInfo == null ? null : syncInfo.getId();

        String token = null;
        if (syncInfo != null && StringUtils.hasText(syncInfo.getCode())) {
            logger.info("更新企微的组织：{}", orgId);
            // 已同步过，则只更新基本信息
            var saveParam = convertSaveParam(orgDOUpdate, cfg.getCorpid());

            // 同步给企微
            token = getAccessTokenOfWecom(cfg);
            var syncResult = WeComTool.departmentUpdate(token, saveParam);
            sysSyncManager.updateSyncResult(syncId, syncResult.isSuccess(), syncInfo.getCode(),
                    syncResult.isSuccess() ? null : syncResult.getErrcode() + ":" + syncResult.getErrmsg());
            return;
        }

        // 未同步过，则同步组织及下级组织和员工
        var orgList = orgRepoProc.listChildrens(orgId).stream().sorted(Comparator.comparingLong(SysOrgDO::getId)).collect(Collectors.toList());
        for (SysOrgDO sysOrgDO : orgList) {
            orgId = sysOrgDO.getId();
            logger.info("新增企微的组织：{}", orgId);
            syncInfo = syncRepoProc.getIdCode(SyncDataType.ORG.name(), orgId + "", cfg.getCorpid());
            syncId = syncInfo == null ? null : syncInfo.getId();

            if (syncInfo == null) {
                // 初始化同步信息
                SysSyncSaveBO sysSyncSaveBO = initSyncSaveBO(orgId, cfg.getCorpid());
                syncId = sysSyncManager.save(sysSyncSaveBO);
            }

            // 同步企微
            var saveParam = convertSaveParam(sysOrgDO, cfg.getCorpid());
            token = getAccessTokenOfWecom(cfg);
            var syncResult = syncInfo != null && StringUtils.hasText(syncInfo.getCode()) ?
                    WeComTool.departmentUpdate(token, saveParam) : WeComTool.departmentCreate(token, saveParam);
            sysSyncManager.updateSyncResult(syncId, syncResult.isSuccess(), syncResult.getId() != null ? syncResult.getId().toString() : null,
                    syncResult.isSuccess() ? null : syncResult.getErrcode() + ":" + syncResult.getErrmsg());
        }

        // 同步员工
        this.syncEmployee(orgDOUpdate.getId(), cfg);
    }

    private void syncEmployee(Long orgId, WecomConfig cfg) {
        logger.info("同步组织下的员工：{}", orgId);

        EmployeeQueryDTO employeeQueryDTO = new EmployeeQueryDTO();
        employeeQueryDTO.setOrgIdBelong(orgId);
        var employeeList = employeeRepoProc.queryList(employeeQueryDTO);
        if (employeeList.isEmpty()) {
            return;
        }

        for (var employee : employeeList) {
            WecomEmpSync.syncToWeCom(employee, cfg);
        }
    }

    static SysSyncSaveBO initSyncSaveBO(Long orgId, String corpid) {
        SysSyncSaveBO sysSyncSaveBO = new SysSyncSaveBO();
        sysSyncSaveBO.setDataType(SyncDataType.ORG.name());
        sysSyncSaveBO.setDataKey(orgId.toString());
        sysSyncSaveBO.setSyncOuter(true);
        sysSyncSaveBO.setOuterApp(corpid);
        sysSyncSaveBO.setSyncDataTime(LocalDateTime.now());
        sysSyncSaveBO.setManual(true);
        sysSyncSaveBO.setSyncSuccess(false);

        return sysSyncSaveBO;
    }

    static DepartmentSaveParam convertSaveParam(SysOrgDO orgDO, String corpid) {
        DepartmentSaveParam saveParam = new DepartmentSaveParam();
        saveParam.setName(orgDO.getName());
        saveParam.setName_en(orgDO.getEnglishName());
        if (orgDO.getPId() != null && orgDO.getPId() != BaseTreeModel.DEFAULT_PARENT) {
            saveParam.setParentid(getWecomId(orgDO.getPId(), corpid));
        }
        saveParam.setOrder(orgDO.getSortNo());
        saveParam.setId(getWecomId(orgDO.getId(), corpid));

        return saveParam;
    }

    static Long getWecomId(Long dataKey, String corpid) {
        var pid = SpringContextHolder.getBean(SyncRepoProc.class).getOuterDataKey(SyncDataType.ORG.name(), dataKey.toString(), corpid);
        return StringUtils.hasText(pid) ? Long.parseLong(pid) : null;
    }

    static Map<Long, Long> getWecomIds(Collection<Long> dataKeys, String corpid) {
        var keys = dataKeys.stream().map(Object::toString).collect(Collectors.toList());
        var pidList = SpringContextHolder.getBean(SyncRepoProc.class).getOuterDataKeys(SyncDataType.ORG.name(), keys, corpid);

        if (pidList.isEmpty()) {
            return Collections.emptyMap();
        }
        Map<Long, Long> result = new HashMap<>(pidList.size());
        for (Map.Entry<String, String> entry : pidList.entrySet()) {
            result.put(Long.parseLong(entry.getKey()), Long.parseLong(entry.getValue()));
        }
        return result;
    }
}
